/*
 *
 *	dump_byt
 *		Utility zum SOWAM-Bytecode-Interpreter
 *
 *
 *	FILE
 *		dump_byt.c
 *
 *	PURPOSE
 *		Symbolische Ausgabe von Bytecode-Files
 *
 *	AUTHORS
 *		Andreas Schwab
 *
 *	HISTORY
 *		[001]	22.05.1990
 *			Erstimplementation
 *
 */

#include <fcntl.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include "sowam.h"
#include "bytecode.h"

code_addr code_area;
struct ex_af_entry *af_area;
char *text_area;
struct codeheader header;
int n_af;

char *prg_name;

#define illegal_opcode(p, oc)	trace(p, "illegal opcode %d", oc)
#define X_AF_NAME(i)		(&text_area[af_area[A_F(i)].text_addr])
#define X_NAME(p)		(&text_area[(p)->text_addr])

void fatal(str)
     char *str;
{
  fprintf(stderr, "%s: fatal error: %s\n", prg_name, str);
  exit(2);
}

void *
xmalloc(size)
     unsigned size;
{
  void *tmp = (void *)malloc(size);
  if (tmp == NULL)
    fatal("virtual memory exhausted");
  return tmp;
}

void
read_bytecode(name)
     char *name;
{
  int fd;
  char file_name[1024];

  fd = open (name, O_RDONLY);
  if (fd < 0)
    {
      strcpy(file_name, name);
      strcat(file_name, ".byt");
      fd = open(file_name, O_RDONLY);
    }
  if (fd < 0)
    {
      perror(name);
      exit(2);
    }

  if (read (fd, &header, sizeof(struct codeheader)) != sizeof(struct codeheader) ||
      strncmp (header.ch_magic, "FALF", 4))
    {
      fprintf(stderr, "%s: %s: illegal file type\n", prg_name, name);
      exit(1);
    }

  if (header.ch_version != bytecode_version)
    {
      fprintf(stderr, "%s: %s: wrong bytecode version (expected %d, found %d)\n",
	      prg_name, name, bytecode_version, header.ch_version);
      exit(1);
    }

  code_area = (code_addr) xmalloc (header.ch_ca_max);
  af_area = (struct ex_af_entry *)xmalloc(header.ch_af_max);
  text_area = (char *)xmalloc(header.ch_ta_max);
  n_af = header.ch_af_max/sizeof(struct ex_af_entry);

  if (read (fd, code_area, header.ch_ca_max) != header.ch_ca_max ||
      read (fd, af_area, header.ch_af_max) != header.ch_af_max ||
      read (fd, text_area, header.ch_ta_max) != header.ch_ta_max)
    {
      perror(name);
      exit(1);
    }

  close (fd);

  return;
}

struct ex_af_entry *
find_name(addr)
     long addr;
{
  struct ex_af_entry *p;

  for (p = af_area; p < af_area + n_af; p++)
    if (p->code1_addr == addr || p->code2_addr == addr)
      return p;
  return NULL;
}

#ifdef __STDC__
void trace(code_addr p, char *fmt,...)
#else
void
trace (p, fmt, va_alist)
     char *fmt;
     code_addr p;
     va_dcl
#endif
{
    va_list args;
    struct ex_af_entry *name;

#ifdef __STDC__
    va_start(args, fmt);
#else
    va_start(args);
#endif
    if (p != code_area && (name = find_name(p-code_area)))
	printf("%s/%d:\n", X_NAME(name),name->arity);
    printf("L%d: ", p-code_area);
    vprintf(fmt, args);
    putchar('\n');
    va_end(args);
}

main(argc, argv)
     int argc;
     char *argv[];
{
  int addr;
  int arg1,arg2,arg3, arg4;
  code_addr p, c_max;
  struct ex_af_entry *name;

  prg_name = argv[0];

  if (argc != 2)
    {
      fprintf(stderr, "Usage: %s filename[.byt]", prg_name);
      exit(1);
    }

  read_bytecode(argv[1]);

  printf("%d Bytes Code, %d Bytes AF-Area, %d Bytes Text-Area, %d Bytes Reloc-Table\n",
	 header.ch_ca_max, header.ch_af_max, header.ch_ta_max, header.ch_rel_max);

  c_max = (code_addr)((char *)code_area + header.ch_ca_max);
  for (p = code_area; p < c_max; p++)
    {
      int opcode = *p;

      switch (MAJOR_OPCODE(opcode))
	{
	case 0:
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 1:
	      trace(p, "allocate");
	      break;
	    case 2:
	      trace(p, "deallocate");
	      break;
	    case 3:
	      trace(p, "allocate-occ");
	      break;
	    case 4:
	      trace(p, "deallocate-occ");
	      break;
	    case 5:
	      trace(p, "proceed");
	      break;
	    case 6:
	      trace(p, "unify-nil");
	      break;
	    case 7:
	      trace(p, "fail");
	      break;
	    case 8:
	      trace(p, "stop");
	      break;
	    case 9:
	      trace(p, "pop-occ");
	      break;
	    case 10:
	      trace(p, "execute-narrowing(ao)");
	      break;
	    case 11:
	      trace(p, "execute-rewriting(ao)");
	      break;
	    case 12:
	      trace(p, "inner-reflection");
	      break;
	    case 13:
	      trace(p, "put-nil-occ");
	      break;
	    case 14:
	      trace(p, "put-list-occ");
	      break;
	    case 15:
	      trace(p, "trust-me-else-fail");
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  break;

	case 1:
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "r-trust-me-else-fail");
	      break;
	    case 1:
	      trace(p, "rebuild-occ-stack");
	      break;
	    case 2:
	      trace(p, "write-nil");
	      break;
	    case 3:
	      trace(p, "read-nil");
	      break;
	    case 4:
	      trace(p, "invalid-os");
	      break;
	    case 5:
	      trace(p, "push-act-occ");
	      break;
	    case 6:
	      trace(p, "l-trust-me-else-fail");
	      break;
	    case 7:
	      trace(p, "call-rewriting(ao)");
	      break;
	    case 8:
	      trace(p, "reject");
	      break;
	    case 9:
	      trace(p, "copy-pop-occ");
	      break;
	    case 10:
	      trace(p, "reflection");
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  break;

	case 2:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "write-void(%d)", arg1);
	      break;
	    case 1:
	      trace(p, "put-nil(X%d)",arg1);
	      break;
	    case 2:
	      trace(p, "get-nil(X%d)",arg1);
	      break;
	    case 3:
	      trace(p, "put-list(X%d)",arg1);
	      break;
	    case 4:
	      trace(p, "get-list(X%d)",arg1);
	      break;
	    case 5:
	      trace(p, "unify-void(%d)",arg1);
	      break;
	    case 6:
	      trace(p, "put-unsafe-value-occ(Y%d)", arg1);
	      break;
	    case 7:
	      trace(p, "unify-variable(X%d)",arg1);
	      break;
	    case 8:
	      trace(p, "write-variable(X%d)",arg1);
	      break;
	    case 9:
	      trace(p, "unify-variable(Y%d)",arg1);
	      break;
	    case 10:
	      trace(p, "unify-value(X%d)",arg1);
	      break;
	    case 11:
	      trace(p, "write-value(X%d)",arg1);
	      break;
	    case 12:
	      trace(p, "unify-value(Y%d)",arg1);
	      break;
	    case 13:
	      trace(p, "unify-local-value(X%d)",arg1);
	      break;
	    case 14:
	      trace(p, "write-local-value(X%d)",arg1);
	      break;
	    case 15:
	      trace(p, "unify-local-value(Y%d)",arg1);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  break;

	case 3:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "put-var-occ(X%d)",arg1);
	      break;
	    case 1:
	      trace(p, "put-var-occ(Y%d)",arg1);
	      break;
	    case 2:
	      trace(p, "put-value-occ(X%d)",arg1);
	      break;
	    case 3:
	      trace(p, "put-value-occ(Y%d)",arg1);
	      break;
	    case 4:
	      trace(p, "load-occ(X%d)",arg1);
	      break;
	    case 5:
	      trace(p, "set-begin-of-term(X%d)",arg1);
	      break;
	    case 6:
	      trace(p, "push-occ(X%d)", arg1);
	      break;
	    case 7:
	      trace(p, "write-variable(Y%d)", arg1);
	      break;
	    case 8:
	      trace(p, "write-value(Y%d)", arg1);
	      break;
	    case 9:
	      trace(p, "write-local-value(Y%d)", arg1);
	      break;
	    case 11:
	      trace(p, "match-list(X%d)", arg1);
	      break;
	    case 12:
	      trace(p, "load-occ(Y%d)", arg1);
	      break;
	    case 13:
	      trace(p, "push-occ(Y%d)", arg1);
	      break;
	    case 14:
	      trace(p, "write-and-ask(X%d)", arg1);
	      break;
	    case 15:
	      trace(p, "built-in(%d)", arg1);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  break;

	case 4:
	  arg1 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "set-begin-of-term(Y%d)", arg1);
	      break;
	    case 1:
	      trace(p, "read-void(%d)", arg1);
	      break;
	    case 2:
	      trace(p, "read-variable(X%d)", arg1);
	      break;
	    case 3:
	      trace(p, "read-variable(Y%d)", arg1);
	      break;
	    case 4:
	      trace(p, "read-value(X%d)", arg1);
	      break;
	    case 5:
	      trace(p, "read-value(Y%d)", arg1);
	      break;
	    case 6:
	      trace(p, "match-nil(X%d)", arg1);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  break;

	case 5:
	  arg1 = LONG_ARG(p+1);
	  switch(MINOR_OPCODE(opcode))
	    {
	    case 0:
	      if (name = find_name(arg1))
		trace(p, "execute(%s/%d)", X_NAME(name), name->arity);
	      else
		trace(p, "execute(L%d)",arg1);
	      break;
	    case 1:
	      trace(p, "retry_me_else(L%d)",arg1);
	      break;
	    case 2:
	      trace(p, "r-retry-me-else(L%d)",arg1);
	      break;
	    case 3:
	      trace(p, "retry(L%d)",arg1);
	      break;
	    case 4:
	      trace(p, "r-retry(L%d)",arg1);
	      break;
	    case 5:
	      trace(p, "trust(L%d)",arg1);
	      break;
	    case 6:
	      trace(p, "r-trust(L%d)",arg1);
	      break;
	    case 7:
	      trace(p, "unify-constant(%s/%d)",X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    case 8:
	      trace(p, "put-struct-occ(%s/%d)",X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    case 9:
	      trace(p, "put-const-occ(%s/%d)", X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    case 10:
	      if (name = find_name(arg1))
		trace(p, "execute-rewriting(%s/%d)", X_NAME(name), name->arity);
	      else
		trace(p, "execute-rewriting(L%d)",arg1);
	      break;
	    case 11:
	      trace(p, "put-function-occ(%s/%d)", X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    case 12:
	      if (name = find_name(arg1))
		trace(p, "call-rewriting(%s/%d)", X_NAME(name), name->arity);
	      else
		trace(p, "call-rewriting(L%d)", arg1);
	      break;
	    case 13:
	      trace(p, "write-constant(%s/%d)",X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    case 14:
	      trace(p, "r-try-me-else(L%d)",arg1);
	      break;
	    case 15:
	      trace(p, "r-try(L%d)",arg1);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  p += 2;
	  break;


	case 6:
	  arg1 = LONG_ARG(p+1);
	  switch(MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "read-constant(%s/%d)",X_AF_NAME(arg1),ARITY(arg1));
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  p += 2;
	  break;


	case 7:
	  arg1 = LONG_ARG(p+1);
	  arg2 = OP2ARG(opcode);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      if (name = find_name(arg1))
		trace(p, "call(%s/%d,%d)", X_NAME(name), name->arity, arg2);
	      else
		trace(p, "call(L%d,%d)",arg1,arg2);
	      break;
	    case 1:
	      trace(p, "call-narrowing(ao,%d)",arg1);
	      break;
	    case 2:
	      trace(p, "call-rewriting(ao,%d)",arg1);
	      break;
	    case 3:
	      trace(p, "try-me-else(L%d,%d)",arg1,arg2);
	      break;
	    case 4:
	      trace(p, "match-structure(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 5:
	      trace(p, "try(L%d,%d)",arg1,arg2);
	      break;
	    case 6:
	      trace(p, "match-constant(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 7:
	      if (name = find_name(arg1))
		trace(p, "call-rewriting(%s/%d,%d)",X_NAME(name),name->arity,arg2);
	      else
		trace(p, "call-rewriting(L%d,%d)",arg1,arg2);
	      break;
	    case 9:
	      trace(p, "put-structure(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 10:
	      trace(p, "put-structure(%s/%d,Y%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 11:
	      trace(p, "get-structure(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 13:
	      trace(p, "put-constant(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    case 14:
	      trace(p, "get-constant(%s/%d,X%d)",X_AF_NAME(arg1),ARITY(arg1),arg2);
	      break;
	    default:
	      illegal_opcode (p, opcode);
	      break;
	    }
	  p += 2;
	  break;

	case 8:
	  arg2 = p[1];
	  arg1 = OP2ARG(arg2);
	  arg2 = OP1ARG(arg2);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 1:
	      trace(p, "match-value(X%d,X%d)", arg1, arg2);
	      break;
	    case 2:
	      trace(p, "match-value(Y%d,X%d)", arg1, arg2);
	      break;
	    case 3:
	      trace(p, "put-variable(Y%d,X%d)", arg1, arg2);
	      break;
	    case 4:
	      trace(p, "put-value(X%d,X%d)", arg1, arg2);
	      break;
	    case 5:
	      trace(p, "put-value(Y%d,X%d)", arg1, arg2);
	      break;
	    case 6:
	      trace(p, "get-variable(X%d,X%d)", arg1, arg2);
	      break;
	    case 7:
	      trace(p, "get-variable(Y%d,X%d)", arg1, arg2);
	      break;
	    case 8:
	      trace(p, "get-value(X%d,X%d)", arg1, arg2);
	      break;
	    case 9:
	      trace(p, "get-value(Y%d,X%d)", arg1, arg2);
	      break;
	    case 10:
	      trace(p, "put-unsafe-value(Y%d,X%d)", arg1, arg2);
	      break;
	    case 11:
	      trace(p, "put-variable(X%d,X%d)", arg1, arg2);
	      break;
	    case 12:
	      trace(p, "put-unsafe-value(X%d,X%d)", arg1, arg2);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  ++p;
	  break;

	case 10:
	  arg1 = p[1];
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "switch-on-structure(%d,...)", arg1);
	      break;
	    case 1:
	      trace(p, "switch-on-constant(%d,...)", arg1);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  p += 4 * arg1 + 1;
	  break;

	case 12:
	  arg1 = LONG_ARG(p+1);
	  arg2 = LONG_ARG(p+3);
	  arg3 = LONG_ARG(p+5);
	  arg4 = LONG_ARG(p+7);
	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "switch-on-term(L%d,L%d,L%d,L%d)", arg1, arg2, arg3, arg4);
	      break;
	    default:
	      illegal_opcode(p, opcode);
	      break;
	    }
	  p += 8;
	  break;

	case 14:
	  arg1 = LONG_ARG(p);
	  arg2 = p[2];
	  arg3 = OP2ARG(arg2);
	  arg2 = OP1ARG(arg2);

	  switch (MINOR_OPCODE(opcode))
	    {
	    case 0:
	      trace(p, "l-try-me-else(L%d,%d,%d)",arg1,arg2,arg3);
	      break;

	    default:
	      illegal_opcode (p, opcode);
	      break;
	    }
	  p += 3;
	  break;

	default:
	  illegal_opcode(p, opcode);
	  break;
	}
    }
  exit(0);
}
