/*
 *
 *	sowam
 *		Bytecode-Interpreter fuer die SOWAM
 *
 *
 *	FILE
 *		builtins.c
 *
 *	PURPOSE
 *		Eingebaute Praedikate der SOWAM
 *		
 *
 *
 *	AUTHORS
 *		Volker Siebert, Andreas Schwab
 *
 */

#include <ctype.h>
#include <string.h>
#include "prologdef.h"
#include "builtins.h"
#include "plread.h"
#include "streams.h"
#include "af-area.h"

term *operator_list;
FILE *read_file = stdin;

static void check_operator_list PROTO((void));
static void int_to_term PROTO((term *, int));

void
built_in(arg)
  int arg;
{
  switch (arg)
    {
    case P_READ:
#ifdef DEBUG
      if (TAG(X_REG(2)) != T_VAR)
	internal_error("nonvar bei READ");
#endif
      operator_list = &X_REG(1);
      check_operator_list();
      read_file = stdin;

      if (!prolog_read(&REF(X_REG(2))))
	SET_CONST(REF(X_REG(2)),READERROR_SYM);
      clearerr(stdin);
      break;

    case P_WRITE:
      operator_list = &X_REG(1);
      check_operator_list();
      write_alf_term(stdout, &X_REG(2));
      fflush(stdout);
      break;

    case P_READS:
#ifdef DEBUG
      if (TAG(X_REG(1)) != T_VAR)
	internal_error("nonvar bei READS");
#endif
      SET_CONST(REF(X_REG(1)), read_line(stdin));
      clearerr(stdin);
      break;

    case P_WRITES:
      {
	term *name = &X_REG(1);

	DEREF_TERMP(name);

	if (ATOM(*name))
	  printf("%s", AF_NAME(VAL(*name)));
	else
	  write_term(stdout, &X_REG(1));
	fflush(stdout);
      }
      break;

    case P_FOPEN:
      {
	term *name = &X_REG(1);
	term *mode = &X_REG(2);

#ifdef DEBUG
	if (TAG(X_REG(3)) != T_VAR)
	  internal_error("nonvar bei FOPEN");
#endif
	DEREF_TERMP(name);
	DEREF_TERMP(mode);

	if (ATOM(*name) && ATOM(*mode))
	  REF(X_REG(3)) = open_stream(name, mode);
	else
	  runtime_error("wrong argument to FOPEN");
      }
      break;

    case P_FCLOSE:
      {
	term *stream = &X_REG(1);

	close_stream(stream);
      }
      break;

    case P_FREAD:
      {
	term *stream = &X_REG(1);

#ifdef DEBUG
	if (TAG(X_REG(3)) != T_VAR)
	  internal_error("nonvar bei FREAD");
#endif
	operator_list = &X_REG(2);
	check_operator_list();
	read_file = get_file(stream, 0);

	if (!prolog_read(&REF(X_REG(3))))
	  SET_CONST(REF(X_REG(3)),READERROR_SYM);
      }
      break;

    case P_FWRITE:
      {
	term *stream = &X_REG(1);
	FILE *file = get_file(stream, 1);

	operator_list = &X_REG(2);
	check_operator_list();
	write_alf_term(file, &X_REG(3));
      }
      break;

    case P_FREADS:
      {
	term *stream = &X_REG(1);
	FILE *file = get_file(stream, 0);

#ifdef DEBUG
	if (TAG(X_REG(2)) != T_VAR)
	  internal_error("nonvar bei FREADS");
#endif
	SET_CONST(REF(X_REG(2)), read_line(file));
      }
      break;

    case P_FWRITES:
      {
	term *stream = &X_REG(1);
	term *name = &X_REG(2);
	FILE *file = get_file(stream, 1);

	DEREF_TERMP(name);
	if (ATOM(*name))
	  fprintf(file, "%s", AF_NAME(VAL(*name)));
	else
	  write_term(file, name);
      }
      break;

    case P_PUT:
      {
	term *number = &X_REG(1);
	int c = get_integer(number);

	putchar(c);
	fflush(stdout);
      }
      break;

    case P_FPUT:
      {
	term *stream = &X_REG(1);
	term *number = &X_REG(2);
	FILE *file = get_file(stream, 1);
	int c = get_integer(number);

	fputc(c, file);
      }
      break;

    case P_GET:
      {
	int c = getchar();

#ifdef DEBUG
	if (TAG(X_REG(2)) != T_VAR)
	  internal_error("nonvar bei GET");
#endif
	if (c == EOF)
	  fail();
	else
	  int_to_term(&REF(X_REG(1)), c);
      }
      break;

    case P_FGET:
      {
	term *stream = &X_REG(1);
	FILE *file = get_file(stream, 0);
	int c = fgetc(file);

#ifdef DEBUG
	if (TAG(X_REG(2)) != T_VAR)
	  internal_error("nonvar bei FGET");
#endif
	if (c == EOF)
	  fail();
	else
	  int_to_term(&REF(X_REG(2)), c);
      }
      break;

    case P_CALLUNIX:
      {
	term *string = &X_REG(1);

	DEREF_TERMP(string);

	if (ATOM(*string))
	  {
	    char *cmd = AF_NAME(VAL(*string));
	    int ret = system(cmd);

	    if (ret == -1)
	      error("system() failed");

	    if (ret != 0)
	      fail();

	  }
	else
	  runtime_error("wrong argument to 'CALLUNIX'");
      }
      break;

    case P_CONCAT:
      {
	term *string1 = &X_REG(1);
	term *string2 = &X_REG(2);

	DEREF_TERMP(string1);
	DEREF_TERMP(string2);

	if (ATOM(*string1) && ATOM(*string2))
	  {
	    char *s1 = AF_NAME(VAL(*string1));
	    char *s2 = AF_NAME(VAL(*string2));
	    char *new_string;
	    int len1 = strlen(s1), len2 = strlen(s2);

	    new_string = xmalloc(len1 + len2 + 1);
	    strcpy(new_string, s1);
	    strcpy(new_string + len1, s2);
#ifdef DEBUG
	    if (TAG(X_REG(3)) != T_VAR)
	      internal_error("nonvar bei CONCAT");
#endif
	    SET_CONST(REF(X_REG(3)), lookup_symbol(new_string, 0));
	    free(new_string);
	  }
	else
	  runtime_error("wrong argument to 'CONCAT'");
      }
      break;

    case P_STRINGNAT:
      {
	term *string_arg, *int_arg, int_term;
	int int_val;
	char *str_end, *number_string;

	string_arg = &X_REG(1);
	int_arg = &X_REG(2);
	DEREF_TERMP(string_arg);
	DEREF_TERMP(int_arg);

	if (ATOM(*string_arg))
	  {
	    if (TAG(*int_arg) == T_UNDEF && !trail_undef(&REF(*int_arg)))
	      return;
	    int_val = strtol(AF_NAME(VAL(*string_arg)), &str_end, 10);
	    while (isspace(*str_end))
	      str_end++;
	    if (*str_end != '\0' || int_val < 0)
	      /* fail(); */
	      runtime_error("illegal string argument to 'STRINGNAT'");
	    else
	      {
		int_to_term(&int_term, int_val);
		if (TAG(*int_arg) == T_UNDEF)
		  *int_arg = int_term;
		else
		  unify(X_REG(2), int_term);
	      }
	  }
	else if (TAG(*string_arg) == T_UNDEF)
	  {
	    if (!trail_undef(&REF(*string_arg)))
	      return;
	    int_val = get_integer(int_arg);
	    number_string = xmalloc(10);
	    sprintf(number_string, "%d", int_val);
	    SET_CONST(*string_arg, lookup_symbol(number_string, 0));
	  }
	else
	  runtime_error("bad arguments to 'STRINGNAT'");
      }
      break;

    case P_STRCHARS:
      {
	term *string_arg, *list_arg, list_term, string_term;

	string_arg = &X_REG(1);
	list_arg = &X_REG(2);
	DEREF_TERMP(string_arg);
	DEREF_TERMP(list_arg);

	if (ISLIST(*list_arg))
	  {
	    if (TAG(*string_arg) == T_UNDEF)
	      {
		if (!trail_undef(&REF(*string_arg)))
		  return;
	      }
	    else if (!ATOM(*string_arg))
	      runtime_error("invalid arguments to 'STRCHARS'");

	    SET_CONST(string_term, implode_string(list_arg));
	    if (TAG(*string_arg) == T_UNDEF)
	      *string_arg = string_term;
	    else
	      unify(X_REG(1), string_term);
	  }
	else if (ATOM(*string_arg))
	  {
	    if (TAG(*list_arg) == T_UNDEF)
	      {
		if (!trail_undef(&REF(*list_arg)))
		  return;
	      }
	    else /* if (!ISLIST(*list_arg)) */
	      runtime_error("invalid arguments to 'STRCHARS'");

	    list_term = explode_string(AF_NAME(VAL(*string_arg)));
	    if (TAG(*list_arg) == T_UNDEF)
	      *list_arg = list_term;
	    else
	      unify(X_REG(2), list_term);
	  }
	else
	  runtime_error("invalid arguments to 'STRCHARS'");
      }
      break;

    case P_COMPSTR:
      {
	term *left, *right;

	left = &X_REG(1);
	right = &X_REG(2);
	DEREF_TERMP(left);
	DEREF_TERMP(right);

	if (ATOM(*left) && ATOM(*right))
	  {
	    if (strcmp(AF_NAME(VAL(*left)), AF_NAME(VAL(*right))) >= 0)
	      fail();
	  }
	else
	  runtime_error("illegal arguments to 'COMPSTR'");
      }
      break;

    default:
      internal_error("built_in");
      break;
    }
}

static void
check_operator_list()
{
  DEREF_TERMP(operator_list);
  if (!ISLIST(*operator_list))
    runtime_error("invalid operator list");
}

void
create_term(result, name, symbol)
     term *result;
     char *name;
     int symbol;
{
  CHECK_HEAP(2);

  SET_CONST(s_regs.h[0], symbol);
  SET_CONST(s_regs.h[1], lookup_symbol(name, 0));
  SET_TERM(*result, T_STRUCT, s_regs.h);
  s_regs.h += 2;
}

void
create_list(result, head, tail)
     term *result;
     term *head;
     term *tail;
{
  CHECK_HEAP(3);

  SET_CONST(s_regs.h[0], LIST_SYM);
  s_regs.h[1] = *head;
  s_regs.h[2] = *tail;
  SET_TERM(*result, T_STRUCT, s_regs.h);
  s_regs.h += 3;
}

void
create_structure(result, name, arity, args)
     term *result;
     char *name;
     int arity;
     term args[];
{
  int i;

  CHECK_HEAP(arity*2 + 3);

  s_regs.h[0] = args[arity-1];
  SET_CONST(s_regs.h[1], NIL);
  s_regs.h += 2;
  for (i = arity - 2; i >= 0; i--)
    {
      s_regs.h[0] = args[i];
      SET_TERM(s_regs.h[1], T_LIST, s_regs.h-2);
      s_regs.h += 2;
    }
  SET_CONST(s_regs.h[0], COMP_SYM);
  SET_CONST(s_regs.h[1], lookup_symbol(name, 0));
  SET_TERM(s_regs.h[2], T_LIST, s_regs.h-2);

  SET_TERM(*result, T_STRUCT, s_regs.h);
  s_regs.h += 3;
}

void
create_integer(result, sval)
     term *result;
     char *sval;
{
  int val;
  term int_term;

  val = atoi(sval);
  int_to_term(&int_term, val);
  CHECK_HEAP(2);
  SET_CONST(s_regs.h[0], INT_SYM);
  s_regs.h[1] = int_term;
  SET_TERM(*result, T_STRUCT, s_regs.h);
  s_regs.h += 2;
}

static void
int_to_term(result, val)
     term *result;
     int val;
{
  int i;

  CHECK_HEAP(2 * val);

  if (val == 0)
    SET_CONST(*result, ZERO_SYM);
  else
    {
      SET_CONST(s_regs.h[0], SUCC_SYM);
      SET_CONST(s_regs.h[1], ZERO_SYM);
      s_regs.h += 2;

      for (i = 1; i < val; i++)
	{
	  SET_CONST(s_regs.h[0], SUCC_SYM);
	  SET_TERM(s_regs.h[1], T_STRUCT, s_regs.h-2);
	  s_regs.h += 2;
	}
      SET_TERM(*result, T_STRUCT, s_regs.h-2);
    }
}

void
lookup_operator(symbol, precedence, optypes)
     int symbol;
     int *precedence;
     int *optypes;
{
  int type;
  term *t, *ot, *op;

  *optypes = *precedence = 0;
  t = operator_list;
  while (TAG(*t) == T_LIST)
    {
      ot = &REF(*t);
      DEREF_TERMP(ot);
      if (TAG(*ot) != T_AF)
	runtime_error("invalid operator declaration");
      switch (VAL(*ot))
	{
	case YFX_SYM:
	  type = OP_YFX;
	  break;
	case XFY_SYM:
	  type = OP_XFY;
	  break;
	case XFX_SYM:
	  type = OP_XFX;
	  break;
	case FY_SYM:
	  type = OP_FY;
	  break;
	case YF_SYM:
	  type = OP_YF;
	  break;
	default:
	  runtime_error("invalid operator type");
	}

      op = ot + 2;
      ot++;
      DEREF_TERMP(ot);
      if (TAG(*ot) != T_AF)
	runtime_error("invalid operator");

      if (VAL(*ot) == symbol)
	{
	  if (*optypes & type || type & OP_INFIX && *optypes & OP_INFIX)
	    runtime_error("duplicate operator type");

	  *optypes |= type;

	  if (*precedence == 0)
	    {
	      DEREF_TERMP(op);
	      for (; TAG(*op) == T_AF && VAL(*op) == SUCC_SYM; ++*precedence)
		{
		  ++op;
		  DEREF_TERMP(op);
		}
	      if (TAG(*op) != T_AF || VAL(*op) != ZERO_SYM)
		runtime_error("invalid operator precedence");
	    }
	}
      t = &TAIL(*t);
      DEREF_TERMP(t);
    }
  if (!ISNIL(*t))
    runtime_error("invalid operator list");
}

int
get_integer(t)
     term *t;
{
  int i;

  DEREF_TERMP(t);
  for (i = 0; TAG(*t) == T_AF && VAL(*t) == SUCC_SYM; i++)
    {
      ++t;
      DEREF_TERMP(t);
    }
  if (TAG(*t) != T_AF || VAL(*t) != ZERO_SYM)
    runtime_error("illegal number argument");

  return i;
}
