%% Prolog-Programm zur Erstellung der A/F-area und der Text-area
%% aus den vorhandenen Daten.
%% Dabei werden gleichzeitig die Adressen fuer Strukturen und
%% gebunden.
%% <c> Andreas Schwab, Berthold Josephs

%% b_make_AF_area/7
%%
%% ermittelt die AF-area und die textuelle Darstellung der Symbole,
%% die Zeiger NafP (New-Atom/Functor-Pointer) und
%% NTextP (New Text Pointer) zeigen nach Abarbeitung 
%% auf das erste freie Byte bzw. auf der naechsten zu vergebenen Zahl
%%         ----------------          --------------------------------
%%
 

b_make_AF_area( LabelList, GSymTab, CSymTab, AF_area, NAfP,
	        Text_area, NTextP) :-
    b_make_AF_area(LabelList, GSymTab, CSymTab,
		   AF_area, 0, NAfP,
		   Text_area, 0, NTextP).

%% b_make_AF_area/9
%% Eingabe:
%%    LabelList, GSymTab, CSymTab als Tabellen
%%    AfP und TextP fuer `Atom/functor-Pointer` und `Text-Pointer`
%%
%% Ein-/Ausgabe:
%%    AF_area : Liste von Listen
%%              Jede Teilliste enthaelt die Informationen fuer einen Eintrag.
%%    Text_area : Liste von Bytelisten, wobei eine Unterliste die Bytes der ASCII-Darstellung
%%		  eines Symbols mit 0 abgeschlossen enthaelt.
%%

b_make_AF_area([],_,_,[],AfP,AfP,[],TextP,TextP).
b_make_AF_area([Symbol1|Symbolrest],GSymTab,CSymTab,
	       [AF_1|AF_Rest],AfP,NAfP,[Text_1|Text_Rest],TextP,NTextP) :-
    b_make_entry(Symbol1, GSymTab, CSymTab,
		 AF_1,AfP,NNAfP,
		 Text_1,TextP,NNTextP),
    b_make_AF_area(Symbolrest, GSymTab, CSymTab, 
		   AF_Rest, NNAfP, NAfP, Text_Rest, NNTextP, NTextP).


%% Die b_make_entry Praedikate ermitteln einen Texteintrag und einen
%% AF-Eintrag und aendern entsprechend die globalen Pointer
%% 
%% Eingabe:
%%	Eintrag aus der LabelList
%% 	GSymTab : globale Symboltabelle
%%	AfP, TextP : Zeiger 'auf die erste freie Stelle'
%% 
%% Ein-/Ausgabe : 
%%	CSymTab : die Adressen werden eingetragen
%%
%% Ausgabe :
%%	falls AF_entry nicht gebunden wird, wird das Symbol nicht verwendet
%%	AF_Entry : Liste von 16 Bytes
%%	Text_entry : Liste der Buchstaben des Symbols mit 0 als Endmarkierung
%%	NAfP, NTextP : die neuen Zeiger

%% Fallunterscheidung :
%% a) Konstruktor:

b_make_entry(entry(Symbol/Stell,cons,Operator,Prec),
	     _,CSymTab,
	     AF_entry,AfP,NAfP,
	     Text_entry,TextP,NTextP) :-
    AF_entry = entry(-1,			% Narrow-Codeadresse
		     -1,			% Rewrite-Codeadresse
		     TextP,			% Zeiger in Textbereich
		     Stell,			% Stelligkeit
		     Operatorcode,		% Operatordefinition
		     Prec),			% Praezedenz
    NAfP is AfP + 1,
    b_code_operator(Operator,Operatorcode), 	% Operatortext codieren
    b_code_text(Symbol,Text_entry,TextP,NTextP), % aus dem Symbol eine Liste von Buchstaben erstellen
    b_lookup(CSymTab,const(Symbol,Stell),AfP).	% Adr. eintragen

%% b) Praedikate
b_make_entry(entry(Symbol/Stell,pred(Label),Operator,Prec),
	     GSymTab,CSymTab,
	     AF_entry,AfP,NAfP,
	     Text_entry,TextP,NTextP) :-
    AF_entry = entry(Val,
		     -1,
		     TextP,
		     Stell,
		     Operatorcode,
		     Prec),
    NAfP is AfP + 1,
    b_lookup(GSymTab,Label,Val),
    b_code_operator(Operator,Operatorcode),
    b_code_text(Symbol,Text_entry,TextP,NTextP),
    b_lookup(CSymTab,const(Symbol,Stell),AfP).	% Adr. eintragen

%% c) rewrite-Regeln
b_make_entry(entry(Symbol/Stell,rewr(Label),Operator,Prec),
	     GSymTab,CSymTab,
	     AF_entry,AfP,NAfP,
	     Text_entry,TextP,NTextP) :-
    AF_entry = entry(0,				% Narrow-Codeadresse -> fail
		     Val,			% Rewrite-Codeadresse
		     TextP,			% Zeiger in Textbereich
		     Stell,			% Stelligkeit
		     Operatorcode,		% Operatordefinition
		     Prec),			% Praezedenz
    NAfP is AfP + 1,
    b_lookup(GSymTab,Label,Val),
    b_code_operator(Operator,Operatorcode), 	% Operatortext codieren
    b_code_text(Symbol,Text_entry,TextP,NTextP), % aus dem Symbol eine Liste von Buchstaben erstellen
    b_lookup(CSymTab,const(Symbol,Stell),AfP).	% Adr. eintragen

%% d) Funktionen
b_make_entry(entry(Symbol/Stell,func(NLabel,RLabel),Operator,Prec),
	     GSymTab,CSymTab,
	     AF_entry,AfP,NAfP,
	     Text_entry,TextP,NTextP) :-
    AF_entry = entry(Nval,			% Narrow-Codeadresse
		     Rval,			% Rewrite-Codeadresse
		     TextP,			% Zeiger in Textbereich
		     Stell,			% Stelligkeit
		     Operatorcode,		% Operatordefinition
		     Prec ),			% Praezedenz
    NAfP is AfP + 1,
    b_lookup(GSymTab,NLabel,Nval),
    b_lookup(GSymTab,RLabel,Rval),
    b_code_operator(Operator,Operatorcode), 	% Operatortext codieren
    b_code_text(Symbol,Text_entry,TextP,NTextP), % aus dem Symbol eine Liste von Buchstaben erstellen
    b_lookup(CSymTab,const(Symbol,Stell),AfP).	% Adr. eintragen


%% --------------------------------------------------------------------------
%% weitere Hilfspraedikate :
%% --------------------------------------------------------------------------


%% b_code_operator/2
%%
%% ermittelt zu jedem Operatortyp die entsprechenden Codierung:

b_code_operator(nofix, 0).
b_code_operator(prefix, 1).
b_code_operator(postfix, 2).
b_code_operator(infixnot, 3).
b_code_operator(infixleft, 4).
b_code_operator(infixright, 5).


%% b_code_text/4
%%
%% erstellt aus einem eingegebenen Symbol eine Liste der Bytes des Textes mit 
%% angehaengter 0

b_code_text(Symbol, TextListe, TextP, NTextP) :-
    name(Symbol,HListe),
    b_append(HListe,[0], TextListe),
    length(TextListe, N),
    NTextP is TextP + N.
