%%% Uebersetzung AbsySOWAM -> Bytecode
%%% Berthold Josephs, Andreas Schwab
%%%
%%% hier: Uebersetzung einer Funktion


%%% b_func_to_bc/7
%%
%% Aufgabe:
%%   Uebersetzung einer Funktion
%%
%% b_func_to_bc(InstrList, SymbolList, GSymTab, CSymTab, BCList, PC, NPC)
%%
%% Eingabe:
%%   InstrList:  Liste der SOWAM-Instruktionen
%%   SymbolList: Liste der Symbole
%%   PC:         Programmcounter vorher
%%
%% Ein-/Ausgabe:
%%   GSymTab: Symboltabelle fuer die globalen Labels (offene Liste)
%%   CSymTab: Symboltabelle fuer die Konstanten (offene Liste)
%%
%% Ausgabe:
%%   BCList: Liste von Bytecodelisten fuer jede SOWAM-Instruktion
%%   NPC:    Programmcounter nachher
%%
%% Benutzte Praedikate:
%%   b_func_to_bc/8

b_func_to_bc(InstrList, SymbolList, GSymTab, CSymTab, BCList, PC, NPC) :-
    b_lookup(LSymTab, fail, 0),
    b_func_to_bc(InstrList, SymbolList, LSymTab, 
		 GSymTab, CSymTab, BCList, PC, NPC),
    b_check_symtab(LSymTab).


%%% b_func_to_bc/8
%%
%% Aufgabe:
%%   Hilfspraedikat fuer b_func_to_bc/7
%%
%% b_func_to_bc(InstrList, SymbolList, LSymTab, GSymTab, CSymTab, BCList, PC, NPC)
%%
%% Argumente wie b_func_to_bc/7
%% zusaetzlich:
%%   LSymTab: Symboltabelle fuer die lokalen Labels (offene Liste)
%%	      Der erste Eintrag ist immer `fail -> 0'
%%
%% benutzte Praedikate:
%%   b_label_bytecode/7

b_func_to_bc([], _, _, _, _, [], PC, PC).
b_func_to_bc([Instr | IRest], SymbolList, LSymTab, GSymTab, CSymTab,
	     [BC | BCRest], PC, NPC) :-
    b_label_bytecode(Instr, SymbolList, LSymTab, GSymTab, CSymTab, BC, PC, NNPC),
    b_func_to_bc(IRest, SymbolList, LSymTab, GSymTab, CSymTab, BCRest, NNPC, NPC).

%%% b_label_bytecode/8
%%

b_label_bytecode(Label : Instr, SymbolList, LSymTab, GSymTab, CSymTab,
		 BC, PC, NPC) :-
    b_new_label(Label, PC, LSymTab),
    b_label_bytecode(Instr, SymbolList, LSymTab, GSymTab, CSymTab, BC, PC, NPC).
b_label_bytecode(Instr, SymbolList, LSymTab, GSymTab, CSymTab, BC, PC, NPC) :-
    b_bytecode(Instr, SymbolList, LSymTab, GSymTab, CSymTab, BC, Length),
    NPC is PC + Length.


%%% b_rewrite_label/4
%%
%% Aufgabe:
%%   Bestimmung des Rewrite-Labels einer Funktion
%%
%% b_rewrite_label(Name, Stell, SymbolList, Label)
%%
%% Eingabe:
%%   Name:       Name der Funktion
%%   Stell:      Stelligkeit der Funktion
%%   SymbolList: Liste der Symbole
%%
%% Ausgabe:
%%   Label:     Rewrite-Label der Funktion
%%
%% benutzte Praedikate:
%%   b_find_spec/4

b_rewrite_label(Name, Stell, SymbolList, Label) :-
    b_find_spec(SymbolList, Name, Stell, func(_, Label)).
b_rewrite_label(_, _, _, 'MISSING LABEL').


%%% b_narrow_label/4
%%
%% Aufgabe:
%%   Bestimmung des Narrow-Labels einer Funktion
%%
%% b_narrow_label(Name, Stell, SymbolList, Label)
%%
%% Eingabe:
%%   Name:       Name der Funktion
%%   Stell:      Stelligkeit der Funktion
%%   SymbolList: Liste der Symbole
%%
%% Ausgabe:
%%   Label:     Narrow-Label der Funktion
%%
%% benutzte Praedikate:
%%   b_find_spec/4

b_narrow_label(Name, Stell, SymbolList, Label) :-
    b_find_spec(SymbolList, Name, Stell, func(Label, _)).
b_narrow_label(_, _, _, 'MISSING LABEL').


%%% b_pred_label/4
%%
%% Aufgabe:
%%   Bestimmung des Labels eines Praedikats
%%
%% b_pred_label(Name, Stell, SymbolList, Label)
%%
%% Eingabe:
%%   Name:       Name des Praedikats
%%   Stell:      Stelligkeit des Praedikats
%%   SymbolList: Liste der Symbole
%%
%% Ausgabe:
%%   Label:     Label des Praedikats
%%
%% benutzte Praedikate:
%%   b_find_spec/4

b_pred_label(Name, Stell, SymbolList, Label) :-
    b_find_spec(SymbolList, Name, Stell, pred(Label)).
b_pred_label(_, _, _, 'MISSING LABEL').


%%% b_find_spec/4
%%
%% Aufgabe:
%%   Suchen der Typspezifikation eines Funktors
%%
%% b_find_spec(SymbolList, Name, Stell, Spec)
%%
%% Eingabe
%%   SymbolList: Liste der Symbole
%%   Name:       Name des Funktors
%%   Stell:      Stelligkeit des Funktors
%%
%% Ausgabe:
%%   Spec: Spezifikation (cons | pred(Label) | func(NLab, RLab))

b_find_spec([entry(Name/Stell, Spec, _, _) | _], Name, Stell, Spec).
b_find_spec([_ | Rest], Name, Stell, Spec) :-
    b_find_spec(Rest, Name, Stell, Spec).
b_find_spec([], Name, Stell, _) :-
    write('Fehler: Spezifikation fuer `'),
    writeq(Name/Stell),
    write(''' nicht gefunden!'), nl,
    !, fail.

%%% b_lookup/3
%%
%% Aufgabe:
%%   Suchen in der Symboltabelle SymTab nach Schluessel Key. Ist die Symboltabelle
%%   eine offene Liste, wird ein ggf. ein neuer Eintrag erzeugt.
%%
%% b_lookup(SymTab, Key, Val)
%%
%% Eingabe:
%%   SymTab: Symboltabelle (offene oder geschlossene Liste)
%%   Key:    zu suchender Schluessel
%%
%% Ausgabe:
%%   SymTab: falls bei offener Liste der Eintrag nicht gefunden wurde
%%   Val:    Wert, der zum Schluessel gespeichert war, falls gefunden, sonst
%%	     nicht veraendert
%%

b_lookup([entry(Key, Val) | _], Key, Val) :- !.
b_lookup([_ | Rest], Key, Val) :- !,
    b_lookup(Rest, Key, Val).


%%% b_new_label/3
%%

b_new_label(Label, PC, SymTab) :-
    b_lookup(SymTab, Label, PC).


%%% b_const_switch_table/5
%%

b_const_switch_table([], _, _, _, []).
b_const_switch_table([entry(Name, Label) | TableRest], SymbolList,
		     LSymTab, CSymTab, [entry(AF_Index,0,PC) | STableRest]) :-
    b_lookup(LSymTab, Label, PC),
    b_lookup(CSymTab, const(Name, 0), AF_Index),
    b_const_switch_table(TableRest, SymbolList, LSymTab, CSymTab, STableRest).


%%% b_struct_switch_table/5
%%

b_struct_switch_table([], _, _, _, []).
b_struct_switch_table([entry(Name/Stell,Label) | TableRest], SymbolList,
		      LSymTab, CSymTab, [entry(AF_Index,Stell,PC) | STableRest]) :-
    b_lookup(LSymTab, Label, PC),
    b_lookup(CSymTab, const(Name,Stell), AF_Index),
    b_struct_switch_table(TableRest, SymbolList, LSymTab, CSymTab, STableRest).
