;;  $Id: pce-user-defined-classes.lisp,v 1.5 1993/02/26 11:55:05 anjo Exp $
;;  
;;  File	pce-user-defined-classes.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden (anjo@swi.psy.uva.nl)
;;		Gertjan van Heijst (gertjan@swi.psy.uva.nl)
;;  Purpose	Syntactic sugar for user defined classes
;;  Works with	PCE 4.4
;;  
;;  Notice	Copyright (c) 1992  University of Amsterdam
;;  
;;  History	10/04/92  (Created)
;;  		25/03/93  (Added delegates argument to defpceclass (GvH))
;;  		12/05/93  (Last modified)


(in-package "PCE")


(export '(defpceclass
	  defpcesendmethod
	  defpcegetmethod
	  !let
	  !setq
	  !store
	  !fetch
	  !new
	  !send
	  !send-self
	  !send-super
	  !send-class
	  !get
	  !get-self
	  !get-super
	  !get-class
	  !get
	  !if
	  !when
	  !while
	  !block
	  !progn
	  !or
	  !and
	  !not
	  !eq
	  !neq
	  !> 
	  !< 
	  !>= 
	  !<= 
	  !+ 
	  !- 
	  !* 
	  !/))


;;  ------------------------------------------------------
;;  Description
;;  ------------------------------------------------------

;;  This file contains the functions that can be used to
;;  define pce-methods. Although defining pce methods is
;;  in many respects similar to defining lisp functions
;;  but there are some important differences:
;;  
;;  - only the functions defined in this file can be used
;;    (they all have the prefix !)
;;  - Function abstraction is not allowed. (pce method
;;    abstraction allowed usually not needed).
;;  - local variables (defined within the !let construct)
;;    may not have names beginning with : or !.


;;  ------------------------------------------------------
;;  Variables
;;  ------------------------------------------------------

(defvar *method-type* nil)
(defvar *print-expansions* nil)


;;  ------------------------------------------------------
;;  Frames
;;  ------------------------------------------------------

;;!m !let vars &rest body ==> {pce-block | nil}
;;
;;  !let is similar to lisp let. There is one difference: if one of
;;  the forms in the body fails the procedure is exited with nil. vars
;;  is a list with var-names or with list with two elements. The first
;;  element is the var-name and the second the initial value.

(defmacro !let (vars &rest body)
  (let ((bindings nil)
	(assigns nil)
	pcevar)
    (dolist (var vars)
	    (cond ((symbolp var)
		   (push (cons var 
			       (setf pcevar (pce-new :var)))
			 bindings)
		   (push (pce-new :assign pcevar @nil) assigns))
		  (t (push (cons (first var)
				 (setf pcevar (pce-new :var)))
			   bindings)
		     (push (pce-new :assign pcevar 
				    (eval 
				     (pce::recursive-macroexpand
				      (second var))))
			   assigns))))
    `(pce-new ,(if (eq *method-type* :get-method) :progn :block)
	      ,@assigns 
	      ,@(sublis bindings body))))


(defmacro !progn (&rest args)
  `(pce-new :progn ,@args))


;;!m !block &rest body ==> {pce-block | nil}
;;
;;  !block is similar to lisp progn. There is one difference: if
;;  one of the forms in the body fails the procedure is exited
;;  with nil.

(defmacro !block (&rest args)
  `(pce-new :block ,@args))
  

;;  ------------------------------------------------------
;;  Assignment
;;  ------------------------------------------------------

;;!m !setq var value ==> pce-assignment
;;
;;  !setq is similar to lisp setq. !setq may only be used
;;  within the body of a !let.

(defmacro !setq (var value)
  `(pce-new :assign ,var ,value))


;;  ------------------------------------------------------
;;  Slot accessors
;;  ------------------------------------------------------

(defmacro !store (slot value)
  `(pce-new :message @receiver :slot ,slot ,value))


(defmacro !fetch (slot)
  `(pce-new :? @receiver :slot ,slot))


;;  ------------------------------------------------------
;;  Method invocation
;;  ------------------------------------------------------

(defmacro !send (object selector &rest args)
  (let ((all-args (append (list object selector) args)))
    `(pce-new :message ,@all-args)))


(defmacro !send-self (selector &rest args)
  (let ((all-args (append (list @receiver selector) args)))
    `(pce-new :message ,@all-args)))


(defmacro !send-super (selector &rest args)
  (let ((all-args (append (list @receiver :send-super selector) args)))
    `(pce-new :message ,@all-args)))


(defmacro !send-class (selector &rest args)
  (let ((all-args (append (list @receiver :send-class selector) args)))
    `(pce-new :message ,@all-args)))


(defmacro !get (object selector &rest args)
  (let ((all-args (append (list object selector) args)))
    `(pce-new :? ,@all-args)))


(defmacro !get-self (selector &rest args)
  (let ((all-args (append (list @receiver selector) args)))
    `(pce-new :? ,@all-args)))


(defmacro !get-super (selector &rest args)
  (let ((all-args (append (list @receiver :get-super selector) args)))
    `(pce-new :? ,@all-args)))


(defmacro !get-class (selector &rest args)
  (let ((all-args (append (list @receiver :get-class selector) args)))
    `(pce-new :? ,@all-args)))


;;  ------------------------------------------------------
;;  Creators
;;  ------------------------------------------------------

(defmacro !new (class-name &rest args)
  `(pce-new :? @pce :instance ,class-name ,@args))


;;  ------------------------------------------------------
;;  Conditionals
;;  ------------------------------------------------------

;;!m !if condition then &optional else ==> pce-object
;;
;;  !if is a procedural switch: it is different from the lisp
;;  macro if because it can not be used to return values. 

(defmacro !if (condition then &optional else)
  (if else
      `(pce-new :if ,condition ,then ,else)
    `(pce-new :if ,condition ,then)))


(defmacro !when (condition then else)
  `(pce-new :when ,condition ,then ,else))
  

;;  ------------------------------------------------------
;;  Iteration
;;  ------------------------------------------------------

(defmacro !while (condition body)
  `(pce-new :while ,condition ,body))
  

;;  ------------------------------------------------------
;;  Logical operators
;;  ------------------------------------------------------

(defmacro !not (condition)
  `(pce-new :not ,condition))

  
(defmacro !or (&rest args)
  `(pce-new :or ,@args))


(defmacro !and (&rest args)
  `(pce-new :and ,@args))


;;  ------------------------------------------------------
;;  Predicates
;;  ------------------------------------------------------

(defmacro !eq (left right)
  `(pce-new :== ,left ,right))


(defmacro !neq (left right)
  `(pce-new :\\== ,left ,right))


(defmacro !> (left right)
  `(pce-new :> ,left ,right))


(defmacro !< (left right)
  `(pce-new :< ,left ,right))


(defmacro !>= (left right)
  `(pce-new :>= ,left ,right))


(defmacro !<= (left right)
  `(pce-new :=< ,left ,right))


;;  ------------------------------------------------------
;;  Arithmetic functions
;;  ------------------------------------------------------

(defmacro !+ (left right)
  `(pce-new :+ ,left ,right))


(defmacro !- (left right)
  `(pce-new :- ,left ,right))


(defmacro !* (left right)
  `(pce-new :* ,left ,right))


(defmacro !/ (left right)
  `(pce-new :/ ,left ,right))



;;  ------------------------------------------------------
;;  Defining classes
;;  ------------------------------------------------------

(defun defpceclass (name &key
			 (super :object)
			 (delegates nil)
			 (slots nil)
			 (locals nil)
			 (keep-instances nil)
			 (create-method nil)
			 (delete-method nil)
			 (resources nil)); '((:name :type :default) ...)
  (if (> (length slots) 10)
      (error "[ DEFPCECLASS: At most 10 arguments allowed ]")
    (let ((class (or (pce-get @pce :convert name :class)
		     (pce-new :class name super)))
	  (names (mapcar #'first slots))
	  (args (list @arg1 @arg2 @arg3 @arg4 @arg5
		      @arg6 @arg7 @arg8 @arg9 @arg10))
	  (types (mapcar #'second slots)))
      (pce-define-class-function name)
      (mapc #'(lambda(argument)
		(pce-send class :instance-variable
			  (apply #'pce-instance-variable argument)))
	    (append slots locals))
      (when slots
	    (pce-send class :send-method
		      (pce-new :send-method :initialise
			       (pce-make :vector types)
			       (pce-make :block
					 (mapcar
					  #'(lambda(name arg)
					      (pce-new :message
						       @receiver
						       :slot
						       name
						       arg))
					  names
					  args)))))
      (when (or keep-instances create-method)
	    (let ((ch (pce-get class :created-messages)))
	      (unless (eq ch @nil)
		      (pce-send ch :clear))))
      (when (or keep-instances delete-method)
	    (let ((ch (pce-get class :freed-messages)))
	      (unless (eq ch @nil)
		      (pce-send ch :clear))))
      (when (or keep-instances create-method delete-method)
	    (pce-send class :attribute
		      (pce-new :attribute :instances (pce-new :chain))))
      (when keep-instances
	    (pce-keep-instances class))
      (when create-method
	    (pce-send class :created-message
		      (pce-new :message @arg2 create-method)))
      (when delete-method
	    (pce-send class :freed-message
		      (pce-new :message @arg2 delete-method)))
      (dolist (resource resources)
	      (pce-send class :resource
			(pce-new :resource (first resource)
				 @default
				 (second resource)
				 (third resource)
				 class)))
      (when delegates
	    (dolist (delegate delegates)
		    (pce-send class :delegate delegate)))
      class)))



;;  ------------------------------------------------------
;;  Defining methods
;;  ------------------------------------------------------

(defun defpcesendmethod (class-name selector &key
				    arguments
				    (documentation @default)
				    body)
  (defpcemethod :send-method 
                 class-name
		 selector
		 arguments
		 documentation
		 body))



(defun defpcegetmethod (class-name selector &key
				   (documentation @default)
				   (returns :any)
				   arguments
				   body)
  (defpcemethod :get-method 
                 class-name 
		 selector
		 arguments
		 documentation
		 body
		 returns))



(defun defpcemethod (type class-name selector arguments documentation body
			  &optional returns)
  (setf *method-type* type)
  (let* ((bindings (mapcar #'(lambda (x y)
			       (cons (first x) y))
			   arguments
			   (list @arg1 @arg2 @arg3 @arg4 @arg5
				 @arg6 @arg7 @arg8 @arg9 @arg10)))
	 (recipient (if (keywordp class-name)
			(pce-get @pce :convert class-name :class)
		      class-name))	; An object
	 (types (mapcar #'second arguments))
	 (message (mapcar #'macro-eval
			  (sublis bindings body)))
	 code)
    (cond ((eql (length message) 1)
	   (setq code (first message)))
	  ((eq type :send-method)
	   (setq code (pce-new :block))
	   (pce-send-list code :append message))
	  ((eq type :get-method)
	   (setq code (pce-new :progn))	; REPLACED :block BY :PROGN -- AA
	   (pce-send-list code :append message)))
    (setf *method-type* nil)
    (pce-send recipient type
	      (if (eq type :send-method)
		  (pce-new :send-method
			   selector
			   (pce-make :vector types)
			   code
			   documentation)
		(pce-new :get-method
			 selector
			 returns
			 (pce-make :vector types)
			 code
			 documentation)))))



(defun macro-eval (object)
  (let ((expansion (recursive-macroexpand object)))
    (when *print-expansions*
	  (pprint expansion))
    (eval expansion)))


(defun recursive-macroexpand (form)
  ;; recursively expands all the macros in form
  (if (and form (listp form))
      (apply #'macroexpand 
	     (list (cons (first form)
			 (mapcar #'recursive-macroexpand 
				 (rest form)))))
    form))



(defun pce-instance-variable (name pce-type
				   &key
				   (access :both)
				   (documentation @nil)
				   (default nil))
  (let ((v (pce-new :variable name pce-type access documentation)))
    (when default
	  (pce-send v :initial-value default))
    v))


;;  ------------------------------------------------------
;;  Keeping instances
;;  ------------------------------------------------------

(defun pce-keep-instances (&rest classes)
  (dolist (class classes)
          (pce-send class :created-message
                    (pce-new :message (pce-new :? class :instances)
                             :append @arg2))
          (pce-send class :freed-message
                    (pce-new :message (pce-new :? class :instances)
                             :delete @arg2)))
  t)
