;;  $Id: pce-macros.lisp,v 1.2 1993/01/07 10:38:23 anjo Exp $
;;  
;;  File	pce-macros.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Definition of macros
;;  Works with	PCE 4.5,  Lisp
;;  
;;  Notice	Copyright (c) 1992, 1993  University of Amsterdam
;;  
;;  History	30/06/92  (Created)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")


(export '(pce-gc			; Expression -> Value
	  pce-at			; ObjectID -> Object
	  pce-memory-usage		; Expression -> Value
	  *standard-readtable*
	  *pce-readtable*
	  ))


;;  ------------------------------------------------------
;;  Lisp readtable management
;;  ------------------------------------------------------

(defvar *standard-readtable* (copy-readtable nil) "Lisp readtable without @")
(defvar *pce-readtable* *readtable* "Lisp readtable including @")


;;  ------------------------------------------------------
;;  @ Macro
;;  ------------------------------------------------------

;;  Expands the form @<atomic> to '(pce-at <atomic>).
;;  Used to denote PCE object references.

(set-macro-character '#\@
		     #'(lambda (fl ch)
			 (declare (ignore ch))
			 `(pce-at ',(read fl t nil t)))
		     nil
		     *pce-readtable*)


(set-macro-character '#\$
		     #'(lambda (fl ch)
			 (declare (ignore ch))
			 `(pce-new := ',(read fl t nil t)
				   ',(read fl t nil t)))
		     nil
		     *pce-readtable*)


;;  ------------------------------------------------------
;;  Memory management macros
;;  ------------------------------------------------------

;;!M pce-gc expression ==> value
;;
;;  Evaluates expression and returns the value it computes.  The macro
;;  pce-gc/# takes care of inserting special instructions such that
;;  PCE cleans up garbage created while evaluating expression.
;;  pce-gc/# is especially useful during initialisation of you
;;  program, always put a pce-gc/# around it.
;;
;;  Examples:
;;=	(pce-gc (pce-send (pce-dialog) :open))
;;=	(pce-gc (initialise-my-application))

(defmacro pce-gc (expression)
  `(let ((mark (pcelisp-mark)))
     (prog1
	 ,expression
       (pcelisp-rewind mark))))


(defmacro pce-memory-usage (expression)
  `(let ((old (pce-get @pce :core-usage)))
     (prog1
	 ,expression
       (format t "~A bytes of memory used~%"
	       (- (pce-get @pce :core-usage) old)))))


;;  ------------------------------------------------------
;;  Manipulation of the pce-object defstruct
;;  ------------------------------------------------------

;;! defstruct pce-object id class
;;
;;  Representation of PCE objects in Lisp is achieved through
;;  the pce-object structure.  The id slot holds the identifier
;;  of the object (e.g. pce in @pce).
;;
;;  The id is as a Lisp symbol interned in the PCE package.

(defstruct (pce-object (:print-function
			(lambda (obj stream depth)
			  (declare (ignore depth))
			  (format stream "@~A / ~A"
				  (pce-object-id obj)
				  (if (pce-exists-p obj)
				      (intern (symbol-name (pce-class-name obj))
					      "PCE")
				    "freed")))))
  id)


;;! defstruct pce-special id
;;
;;  Structure for special objects (e.g. @on, @default, ...).

(defstruct (pce-special (:include pce-object)
			(:print-function
			 (lambda (obj stream depth)
			   (declare (ignore depth))
			   (format stream "@~A"
				   (pce-object-id obj))))))


;;! defstruct pce-assoc id
;;
;;  Structure for named objects (e.g. @pce).

(defstruct (pce-assoc (:include pce-object)))


;;!  defstruct pce-reference id
;;
;;  Structure for numbered objects (e.g. @123456).

(defstruct (pce-reference (:include pce-object)))


;;  defconstant *pce-object-refs*
;;  defconstant *pce-object-assocs*
;;
;;  A Lisp hash-table mapping PCE object names onto defstructs.
;;  For all id's in the hash-table, the PCE/Lisp interface returns
;;  the same defstruct for the same id (i.e. they compare eq/# in
;;  Lisp).  In order to avoid incompatibility with the contents of
;;  PCE itself and the hash-table only special objects (e.g. @on),
;;  and named objects (e.g. @pce) are inserted in the hash-table
;;  by default.
;;
;;  Hash tables are created during initialisation of PCE.

(defvar *pce-object-refs* nil)
(defvar *pce-object-assocs* nil)


;;! pce-at id ==> defstruct
;;
;;  Returns the defstruct corresponding to PCE object with identifier
;;  ^id^, or creates a new defstruct for that object.

(defun pce-at (id)
  (if (integerp id)
      (or (gethash id *pce-object-refs*)
	  (make-pce-reference :id id))
    (if (or (symbolp id) (stringp id))
	(let ((name (intern (string id) "PCE")))
	  (or (gethash name *pce-object-assocs*)
	      (setf (gethash name *pce-object-assocs*)
		    (#+LUCID with-static-area
		     #+LISPWORKS system:in-static-area
		     (make-pce-assoc :id name)))))
      (error "PCE-AT  Argument (~A) not an integer, symbol or string" id))))


;;  ------------------------------------------------------
;;  Marking
;;  ------------------------------------------------------

(defun pcelisp-mark ()
  (pcelisp-c-mark))


(defun pcelisp-rewind (mark)
  (pcelisp-c-rewind mark))
