/* xltvec.c - typed vectors */
/* Copyright (c) 1989, by David Michael Betz.                            */
/* You may give out copies of this software; for conditions see the file */
/* COPYING included with this distribution.                              */

#include "xlisp.h"

#define gettvectype(x) ((unsigned char *) (gettvecdata(x)))[gettlength(x)]
#define settvectype(x,t) \
  (((unsigned char *) (gettvecdata(x)))[gettlength(x)] = (unsigned char) (t))

enum {
  CD_CHARACTER = 0,
  CD_FIXTYPE,
  CD_FLOTYPE,
  CD_CXFIXTYPE,
  CD_CXFLOTYPE,
  CD_CHAR,
  CD_SHORT,
  CD_INT,
  CD_LONG,
  CD_FLOAT,
  CD_DOUBLE,
  CD_COMPLEX,
  CD_DCOMPLEX,
  CD_TRUE
};

LOCAL int typecode _((LVAL x));
LOCAL int typesize _((int code));

LOCAL int typecode(x)
     LVAL x;
{
  x = xlparsetype(x);
  if (x == a_char) return(CD_CHARACTER);
  else if (x == a_fixnum) return(CD_FIXTYPE);
  else if (x == a_flonum) return(CD_FLOTYPE);
  else if (consp(x) && consp(cdr(x)) && car(x) == a_complex) {
    x = xlparsetype(car(cdr(x)));
    if (x == a_fixnum) return(CD_CXFIXTYPE);
    else if (x == a_flonum) return(CD_CXFLOTYPE);
    else return(CD_TRUE);
  }
  else if (x == xlenter("C-CHAR")) return(CD_CHAR);
  else if (x == xlenter("C-SHORT")) return(CD_SHORT);
  else if (x == xlenter("C-INT")) return(CD_INT);
  else if (x == xlenter("C-LONG")) return(CD_LONG);
  else if (x == xlenter("C-FLOAT")) return(CD_FLOAT);
  else if (x == xlenter("C-DOUBLE")) return(CD_DOUBLE);
  else if (x == xlenter("C-COMPLEX")) return(CD_COMPLEX);
  else if (x == xlenter("C-DCOMPLEX")) return(CD_DCOMPLEX);
  else return(CD_TRUE);
}

LOCAL int typesize(code)
     int code;
{
  switch(code) {
  case CD_CHARACTER: return(sizeof(char));
  case CD_FIXTYPE:   return(sizeof(FIXTYPE));
  case CD_FLOTYPE:   return(sizeof(FLOTYPE));
  case CD_CXFIXTYPE: return(2 * sizeof(FIXTYPE));
  case CD_CXFLOTYPE: return(2 * sizeof(FIXTYPE));
  case CD_CHAR:      return(sizeof(char));
  case CD_SHORT:     return(sizeof(short));
  case CD_INT:       return(sizeof(int));
  case CD_LONG:      return(sizeof(long));
  case CD_FLOAT:     return(sizeof(float));
  case CD_DOUBLE:    return(sizeof(double));
  case CD_COMPLEX:   return(2 * sizeof(float));
  case CD_DCOMPLEX:  return(2 * sizeof(float));
  default:           return(1);
  }
}

LVAL mktvec(n, etype)
     int n;
     LVAL etype;
{
  LVAL val;
  int type;

  type = typecode(etype);
  if (type == CD_TRUE)
    val = newvector(n);
  else if (type == CD_CHARACTER) {
    int i;
    val = newstring(n);
    for (i = 0; i < n; i++)
      setstringch(val, i, ' ');
  }
  else {
    val = newtvec(n, typesize(type));
    settvectype(val, type);
  }
  return(val);
}

#define settvecdataelt(c, t, i, v) (((t *) (c))[i] = ((t) (v)))
#define gettvecdataelt(c, t, i) (((t *) (c))[i])

#define CVFIX(x) cvfixnum((FIXTYPE) (x))
#define CVFLO(x) cvflonum((FLOTYPE) (x))

#define u_char unsigned char
#define u_short unsigned short
#define u_int unsigned int
#define u_long unsigned long

int gettvecsize(x)
     LVAL x;
{
  switch(ntype(x)) {
  case VECTOR: return(getsize(x));
  case STRING: return(getslength(x));
  case TVEC:   return(gettlength(x) / typesize(gettvectype(x)));
  default: xlbadtype(x);
  }
  /* not reacched */
  return 0;
}

LVAL gettvecelement(x, i)
     LVAL x;
     int i;
{
  double rval, ival;
  FIXTYPE irval, iival;
  ALLOCTYPE *v;
  int type;

  switch (ntype(x)) {
  case VECTOR: return(getelement(x, i));
  case STRING: return(cvchar(getstringch(x, i)));
  case TVEC:
    type = gettvectype(x);
    v = gettvecdata(x);
  
    switch (type) {
    case CD_CHARACTER:  return(cvchar(gettvecdataelt(v, char, i)));
    case CD_FIXTYPE:    return(CVFIX(gettvecdataelt(v, FIXTYPE, i)));
    case CD_FLOTYPE:    return(CVFLO(gettvecdataelt(v, FLOTYPE, i)));
    case CD_CXFIXTYPE:
      irval = gettvecdataelt(v, FIXTYPE, 2 * i);
      iival = gettvecdataelt(v, FIXTYPE, 2 * i + 1);
      return(newicomplex(irval, iival));
    case CD_CXFLOTYPE:
      rval = gettvecdataelt(v, FLOTYPE, 2 * i);
      ival = gettvecdataelt(v, FLOTYPE, 2 * i + 1);
      return(newicomplex(rval, ival));
    case CD_CHAR:       return(CVFIX(gettvecdataelt(v, char, i)));
    case CD_SHORT:      return(CVFIX(gettvecdataelt(v, short, i)));
    case CD_INT:        return(CVFIX(gettvecdataelt(v, int, i)));
    case CD_LONG:       return(CVFIX(gettvecdataelt(v, long, i)));
    case CD_FLOAT:      return(CVFLO(gettvecdataelt(v, float, i)));
    case CD_DOUBLE:     return(CVFLO(gettvecdataelt(v, double, i)));
    case CD_COMPLEX:
      rval = gettvecdataelt(v, float, 2 * i);
      ival = gettvecdataelt(v, float, 2 * i + 1);
      return(newdcomplex(rval, ival));
    case CD_DCOMPLEX:
      rval = gettvecdataelt(v, double, 2 * i);
      ival = gettvecdataelt(v, double, 2 * i + 1);
      return(newdcomplex(rval, ival));
    default:
      xlbadtype(x);
    }
  default:
    xlbadtype(x);
  }
  /* not reached */
  return(NIL);
}

VOID settvecelement(x, i, item)
     LVAL x, item;
     int i;
{
  ALLOCTYPE *v;
  int type;

  switch (ntype(x)) {
  case VECTOR: setelement(x, i, item); break;
  case STRING:
    if (! charp(item)) xlbadtype(item);
    setstringch(x, i, getchcode(item));
    break;
  case TVEC:
    type = gettvectype(x);
    v = gettvecdata(x);

    switch (type) {
    case CD_CHARACTER:
      if (! charp(item)) xlbadtype(item);
      settvecdataelt(v, char, i, getchcode(item));
      break;
    case CD_FIXTYPE:
      if (! fixp(item)) xlbadtype(item);
      settvecdataelt(v, FIXTYPE, i, getfixnum(item));
      break;
    case CD_FLOTYPE:
      settvecdataelt(v, FLOTYPE, i, makefloat(item));
      break;
    case CD_CXFIXTYPE:
      if (! fixp(realpart(item))) xlbadtype(item);
      if (! fixp(imagpart(item))) xlbadtype(item);
      settvecdataelt(v, FIXTYPE, 2 * i, getfixnum(realpart(item)));
      settvecdataelt(v, FIXTYPE, 2 * i + 1, getfixnum(imagpart(item)));
      break;
    case CD_CXFLOTYPE:
      settvecdataelt(v, FLOTYPE, 2 * i, makefloat(realpart(item)));
      settvecdataelt(v, FLOTYPE, 2 * i + 1, makefloat(imagpart(item)));
      break;
    case CD_CHAR:
      if (! fixp(item)) xlbadtype(item);
      settvecdataelt(v, char, i, getfixnum(item));
      break;
    case CD_SHORT: 
      if (! fixp(item)) xlbadtype(item);
      settvecdataelt(v, short, i, getfixnum(item));
      break;
    case CD_INT:
      if (! fixp(item)) xlbadtype(item);
      settvecdataelt(v, int, i, getfixnum(item));
      break;
    case CD_LONG:
      if (! fixp(item)) xlbadtype(item);
      settvecdataelt(v, long, i, getfixnum(item));
      break;
    case CD_FLOAT:
      settvecdataelt(v, float, i, makefloat(item));
      break;
    case CD_DOUBLE:
      settvecdataelt(v, double, i, makefloat(item));
      break;
    case CD_COMPLEX:
      settvecdataelt(v, float, 2 * i, makefloat(realpart(item)));
      settvecdataelt(v, float, 2 * i + 1, makefloat(imagpart(item)));
      break;
    case CD_DCOMPLEX:
      settvecdataelt(v, double, 2 * i, makefloat(realpart(item)));
      settvecdataelt(v, double, 2 * i + 1, makefloat(imagpart(item)));
      break;
    default:
      xlbadtype(x);
    }
    break;
  default:
    xlbadtype(x);
  }
}

LVAL gettvecetype(x)
     LVAL x;
{
  switch (ntype(x)) {
  case VECTOR: return(s_true);
  case STRING: return(a_char);
  case TVEC:
    switch (gettvectype(x)) {
    case CD_CHARACTER: return(a_char);
    case CD_FIXTYPE:   return(a_fixnum);
    case CD_FLOTYPE:   return(a_flonum);
    case CD_CXFIXTYPE: return(cons(a_complex, consa(a_fixnum)));
    case CD_CXFLOTYPE: return(cons(a_complex, consa(a_flonum)));
    case CD_CHAR:      return(s_c_char);
    case CD_SHORT:     return(s_c_short);
    case CD_INT:       return(s_c_int);
    case CD_LONG:      return(s_c_long);
    case CD_FLOAT:     return(s_c_float);
    case CD_DOUBLE:    return(s_c_double);
    case CD_COMPLEX:   return(s_c_complex);
    case CD_DCOMPLEX:  return(s_c_dcomplex);
    default: xlbadtype(x);
    }
  default: xlbadtype(x);
  }
  /* not reached */
  return(NIL);
}

int gettveceltsize(x)
     LVAL x;
{
  switch (ntype(x)) {
  case VECTOR: return(sizeof(LVAL));
  case STRING: return(1);
  case TVEC:   return(typesize(gettvectype(x)));
  default:     xlbadtype(x);
  }
  /* not reached */
  return(0);
}
