/* xlread - xlisp expression input routine */
/* Copyright (c) 1989, by David Michael Betz.                            */
/* You may give out copies of this software; for conditions see the file */
/* COPYING included with this distribution.                              */

#include "xlisp.h"

/* symbol parser modes */
#define DONE	0
#define NORMAL	1
#define ESCAPE	2

/* string constants */
#define WSPACE "\t \f\r\n"
#define CONST1 "!$%&*+-./0123456789:<=>?@[]^_{}~"
#define CONST2 "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"

/* forward declarations */
LOCAL LVAL callmacro _((LVAL fptr, int ch));
LOCAL LVAL psymbol _((LVAL fptr));
LOCAL LVAL punintern _((LVAL fptr));
LOCAL LVAL pnumber _((LVAL fptr, int radix));
LOCAL LVAL pquote _((LVAL fptr, LVAL sym));
LOCAL LVAL plist _((LVAL fptr));
LOCAL LVAL pvector _((LVAL fptr));
LOCAL LVAL pstruct _((LVAL fptr));
LOCAL LVAL readlist _((LVAL fptr, int *plen));
LOCAL VOID pcomment _((LVAL fptr));
LOCAL VOID badeof _((void));/* TAA MOD to remove unnecessary arg 11/92 */
LOCAL VOID upcase _((char *str));
LOCAL VOID storech _((int *c, int ch));
LOCAL int  nextch _((LVAL fptr));
LOCAL int  checkeof _((LVAL fptr));
LOCAL int  readone _((LVAL fptr, LVAL *pval));
#ifdef PACKAGES
LOCAL int  pname _((LVAL fptr, int *pescflag, int *ppackindex));
#else
LOCAL int  pname _((LVAL fptr, int *pescflag));
#endif /* PACKAGES */
LOCAL VOID defmacro _((int ch, LVAL type, int offset));
#ifdef PRINTCIRCLE
LOCAL LVAL findcircval _((int n, LVAL data));
LOCAL VOID cleancircle _((LVAL val, LVAL data));
LOCAL VOID circpush _((LVAL val, LVAL table, LVAL *ptodo));
LOCAL VOID registercirc _((LVAL entry, LVAL table));
#endif /* PRINTCIRCLE */

/* xlload - load a file of xlisp expressions */
int xlload(fname,vflag,pflag)
  char *fname; int vflag,pflag;
{
    char fullname[STRMAX+1];
    LVAL fptr,expr;
    CONTEXT cntxt;
    FILEP fp;
    int sts, mask;
    LVAL oldrtable = getvalue(s_rtable);
#ifdef PACKAGES
    LVAL oldpack = getvalue(s_package);
#endif /* PACKAGES */

#if (defined(XLISP_ONLY) && ! defined(BYTECODE))
    /* protect some pointers */
    xlstkcheck(3);
    xlsave(fptr);
    xlsave(expr);
    xlprotect(oldrtable);
#ifdef PACKAGES
    xlprot1(oldpack);
#endif /* PACKAGES */

    /* default the extension */
    if (needsextension(fname)) {
	strcpy(fullname,fname);
	strcat(fullname,".lsp");
	fname = fullname;
    }

    /* allocate a file node */
    fptr = cvfile(CLOSED,S_FORREADING);

    /* open the file */
#ifdef PATHNAMES
    if ((fp = ospopen(fname,TRUE)) == CLOSED)
#else
    if ((fp = OSAOPEN(fname,OPEN_RO)) == CLOSED)
#endif
    {
	xlpopn(3);
#ifdef PACKAGES
	xlpop();
#endif /* PACKAGES */
	return (FALSE);
    }
    setfile(fptr,fp);
#else
    {
      char origname[STRMAX+1];
      int extend = needsextension(fname);
      int done, try;
#ifndef XLISP_ONLY
      extern LVAL s_default_path;
      LVAL dp = getvalue(s_default_path);
#endif /* XLISP_ONLY */

      strcpy(origname, fname);
      fname = fullname;
      fp = CLOSED;

      for (try = 1, done = FALSE; ! done; try++) {
	switch (try) {
	case 1:
	  if (extend) {
	    strcpy(fullname, origname);
	    strcat(fullname, ".fsl");
	    break;
	  }
	  else try++;
	  /* fall through */
	case 2:
	  strcpy(fullname, origname);
	  if (extend)
	    strcat(fullname, ".lsp");
	  break;
#ifndef XLISP_ONLY
	case 3:
	  if (extend && stringp(dp)) {
	    strcpy(fullname, getstring(dp));
	    strcat(fullname, origname);
	    strcat(fullname, ".fsl");
	    break;
	  }
	  else try++;
	  /* fall through */
	case 4:
	  if (stringp(dp)) {
	    strcpy(fullname, getstring(dp));
	    strcat(fullname, origname);
	    if (extend)
	      strcat(fullname, ".lsp");
	    break;
	  }
	  else try++;
	  /* fall through */
#endif /* XLISP_ONLY */
	default: done = TRUE;
	}
	if (! done)
	  if ((fp = OSAOPEN(fname,OPEN_RO)) != CLOSED)
	    done = TRUE;
      }
    }

    if (fp == CLOSED)
	return(FALSE);

    /* protect some pointers */
    xlstkcheck(3);
    xlsave(fptr);
    xlsave(expr);
    xlprotect(oldrtable);
#ifdef PACKAGES
    xlprot1(oldpack);
#endif /* PACKAGES */

    /* allocate a file node */
    fptr = cvfile(fp,S_FORREADING);
#endif /* XLISP_ONLY */

    /* print the information line */
    if (vflag)  /* TAA MOD -- changed from printing to stdout */
	{ sprintf(buf,"; loading \"%s\"\n",fname); dbgputstr(buf); }

    /* read, evaluate and possibly print each expression in the file */
    xlbegin(&cntxt,CF_ERROR|CF_UNWIND,s_true); /*TAA mod so file gets closed*/
#ifdef CRAYCC
    mask = setjmp(cntxt.c_jmpbuf); /* TAA mod -- save mask */
    if (mask != 0)
#else
    if ((mask = setjmp(cntxt.c_jmpbuf)) != 0) /* TAA mod -- save mask */
#endif /* CRAYCC */
	sts = FALSE;
    else {
	while (xlread(fptr,&expr,FALSE)) {
	    expr = xleval(expr);
	    if (pflag)
		stdprint(expr);
	}
	sts = TRUE;
    }
    xlend(&cntxt);

    /* restore the readtable and package */
    setvalue(s_rtable, oldrtable);
#ifdef PACKAGES
    setvalue(s_package, oldpack);
#endif /* PACKAGES */

    /* close the file */
    OSCLOSE(getfile(fptr));
    setfile(fptr,CLOSED);

    /* restore the stack */
    xlpopn(3);
#ifdef PACKAGES
    xlpop();
#endif /* PACKAGES */

    /* check for unwind protect TAA MOD */
    if ((mask & ~CF_ERROR) != 0)
        xljump(xltarget, xlmask, xlvalue);

    /* return status */
    return (sts);
}

#ifdef PRINTCIRCLE
#define PCHSIZE 31

#define circvalp(x) (consp(x) && car(x) == car(data))
#define circindex(x) (getfixnum(cdr(x)))

LOCAL LVAL findcircval(n, data)
     int n;
     LVAL data;
{
  LVAL next;
  for (next = cdr(cdr(data)); consp(next); next = cdr(next)) {
    if (consp(car(next)) && fixp(car(car(next)))) {
      if (getfixnum(car(car(next))) == n)
	return cdr(car(next));
    }
  }
  xlerror("bad circle read index", cvfixnum((FIXTYPE) n));
  return(NIL);
}

LOCAL VOID circpush(val, table, ptodo)
     LVAL val, table, *ptodo;
{
  LVAL next;
  int i;

  switch (ntype(val)) {
  case CONS:
  case DARRAY:
  case RNDSTATE:
  case ARRAY:
  case OBJECT:
  case VECTOR:
  case STRUCT:
#ifdef BYTECODE
  case CPSNODE:
  case BCODE:
#endif /* BYTECODE */
#ifdef HASHFCNS
    if (structp(val) && getelement(val,0) == a_hashtable)
	break;
#endif
    i = (int) (CVPTR(val) % PCHSIZE);
    for (next = getelement(table, i); consp(next); next = cdr(next))
      if (car(next) == val)
	return;
    *ptodo = cons(val, *ptodo);
  }
}
	 
LOCAL VOID registercirc(entry, table)
     LVAL entry, table;
{  
  int i = (int) (CVPTR(car(entry)) % PCHSIZE);
  rplacd(entry, getelement(table, i));
  setelement(table, i, entry);
}

LOCAL VOID cleancircle(val, data)
     LVAL val, data;
{
  LVAL todo, table, entry, next;
  int i, changed;

  if (null(car(cdr(data)))) return;

  switch (ntype(val)) {
  case SUBR:
  case FSUBR:
  case FIXNUM:
  case FLONUM:
  case STREAM:
  case CHAR:
  case USTREAM:
  case COMPLEX:
#ifdef BYTECODE
  case BCCLOSURE:
#endif /* BYTECODE */
  case CLOSURE:
  case STRING:
  case ADATA:
  case TVEC:
  case SYMBOL:
#ifdef PACKAGES
  case PACKAGE:
#endif /* PACKAGES */
    return;
  }

  xlstkcheck(2);
  xlsave(todo);
  xlsave(table);

  table = newvector(PCHSIZE);

  do {
    changed = FALSE;

    todo = consa(val);
    for (i = 0; i < PCHSIZE; i++)
      setelement(table, i, NIL);

    while (consp(todo)) {
      entry = todo;
      next = car(todo);
      todo = cdr(todo);
      switch (ntype(next)) {
      case CONS:
      case DARRAY:
      case RNDSTATE:
	registercirc(entry, table);
	if (circvalp(car(next))) {
	  rplaca(next, findcircval(circindex(car(next)), data));
	  changed = TRUE;
	}
	circpush(car(next), table, &todo);
	if (circvalp(cdr(next))) {
	  rplacd(next, findcircval(circindex(cdr(next)), data));
	  changed = TRUE;
	}
	circpush(cdr(next), table, &todo);
	break;
      case ARRAY:
      case OBJECT:
      case VECTOR:
      case STRUCT:
#ifdef BYTECODE
      case CPSNODE:
    case BCODE:
#endif /* BYTECODE */
#ifdef HASHFCNS
	if (structp(next) && getelement(next,0) == a_hashtable)
	  break;
#endif
	{
	  int i, n;
	  registercirc(entry, table);
	  for (i = 0, n = getsize(next); i < n; i++) {
	    if (circvalp(getelement(next, i))) {
	      setelement(next, i,
			 findcircval(circindex(getelement(next, i)), data));
	      changed = TRUE;
	    }
	    circpush(getelement(next, i), table, &todo);
	  }
	}
	break;
      }
    }
  } while (changed);

  xlpopn(2);
}
#endif /* PRINTCIRCLE */

/* xlread - read an xlisp expression */
int xlread(fptr,pval,recursive)
  LVAL fptr,*pval;
  int recursive;
{
    int sts;
#ifdef PRINTCIRCLE
    LVAL olddenv = xldenv;
    if (! recursive)
      xldbind(s_rdcircdat, cons(consa(NIL),consa(NIL)));
#endif /* PRINTCIRCLE */

    /* read an expression */
    while ((sts = readone(fptr,pval)) == FALSE)
	;
#ifdef PRINTCIRCLE
    if (! recursive) {
      cleancircle(*pval, getvalue(s_rdcircdat));
      xlunbind(olddenv);
    }
#endif /* PRINTCIRCLE */

    /* return status */
    return (sts == EOF ? FALSE : TRUE);
}

/* readone - attempt to read a single expression */
LOCAL int readone(fptr,pval)
  LVAL fptr, *pval;
{
    LVAL val,type;
    int ch;

#ifdef STSZ
    /* check the stack */
    stchck();
#endif

    /* get a character and check for EOF */
    if ((ch = xlgetc(fptr)) == EOF)
	return (EOF);

    /* handle white space */
    if ((type = tentry(ch)) == k_wspace)
	return (FALSE);

    /* handle symbol constituents */
    /* handle single and multiple escapes */  /* combined by TAA MOD */
    else if (type == k_const ||
	     type == k_sescape || type == k_mescape) {
	xlungetc(fptr,ch);
	*pval = psymbol(fptr);
	return (TRUE);	    
    }

    /* handle read macros */
    else if (consp(type)) {
	if (((val = callmacro(fptr,ch)) != NIL) && consp(val)) {
	    *pval = car(val);
	    return (TRUE);
	}
	else
	    return (FALSE);
    }

    /* handle illegal characters */
    else {
/*      xlerror("illegal character",cvfixnum((FIXTYPE)ch)); */
        xlerror("illegal character",cvchar(ch));    /* friendlier TAA MOD*/
        return (0);  /* compiler warning */
    }
}

/* rmhash - read macro for '#' */
LVAL rmhash()
{
    LVAL fptr,val;
    char *bufp;         /* TAA fix to allow control character literals */
        int i;
    int ch;
#ifdef __SASC__
    int testch;
#endif

    /* protect some pointers */
    xlsave1(val);

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* make the return value */
    val = consa(NIL);

    /* check the next character */
    switch (ch = xlgetc(fptr)) {
    case '\'':
		rplaca(val,pquote(fptr,s_function));
		break;
    case '(':
		xlungetc(fptr,ch);
		rplaca(val,pvector(fptr));
		break;
    case '.':
		if (! null(getvalue(s_read_suppress))) {
		  rplaca(val,NIL);
		  break;
		}
		readone(fptr,&car(val));
		rplaca(val,xleval(car(val)));
		break;
    case 'b':
    case 'B':
		rplaca(val,pnumber(fptr,2));
		break;
    case 'o':
    case 'O':
		rplaca(val,pnumber(fptr,8));
		break;
    case 'x':
    case 'X':
    		rplaca(val,pnumber(fptr,16));
		break;
    case 's':
    case 'S':
		rplaca(val,pstruct(fptr));
		break;
    case '\\':
		for (i = 0; i < STRMAX-1; i++) {
		  ch = xlgetc(fptr);  /* TAA fix to scan at end of file */
		  if (ch == EOF || 
		      ((tentry((unsigned char)(buf[i] = ch))  != k_const) &&
		       (i > 0) &&      /* TAA fix for left and right paren */
		       buf[i] != '\\' && buf[i] != '|')) {
		    xlungetc(fptr, buf[i]);
		    break;
		  }
		}
		if (! null(getvalue(s_read_suppress))) {
		  rplaca(val,NIL);
		  break;
		}
		buf[i] = 0;
		ch = (unsigned char)buf[0];
#ifdef __SASC__
	testch = etoa(ch);
#endif
		if (strlen(buf) > (unsigned)1) {  /* TAA Fixed */
		  i = buf[strlen(buf)-1]; /* Value of last character */
		  upcase(buf);
		  bufp = &buf[0];
#ifdef __SASC__ /* EBCDIC */
	    testch = 0;
	    if (strncmp(bufp,"M-",2) == 0) {
		testch = 128;
		bufp += 2;
	    }
	    if (strcmp(bufp,"NEWLINE") == 0)
		testch += 0x0a;
	    else if (strcmp(bufp,"SPACE") == 0)
		testch += 0x20;
	    else if (strcmp(bufp,"RUBOUT") == 0)
		ch += 127;
	    else if (strlen(bufp) == 1)
		ch += i;
	    else if (strncmp(bufp,"C-",2) == 0 && strlen(bufp) == 3)
		testch += etoa(bufp[2]) & 31;
	    else xlerror("unknown character name",cvstring(buf));
	    ch = testch;
#else
		  ch = 0;
		  if (strncmp(bufp,"M-",2) == 0) {
		    ch = 128;
		    bufp += 2;
		  }
		  if (strcmp(bufp,"NEWLINE") == 0)
		    ch += '\n';
		  else if (strcmp(bufp,"SPACE") == 0)
		    ch += ' ';
		  else if (strcmp(bufp,"RUBOUT") == 0)
		    ch += 127;
		  else if (strcmp(bufp,"TAB") == 0)
		    ch += '\t';
		  else if (strlen(bufp) == 1) 
		    ch += i;
		  else if (strncmp(bufp,"C-",2) == 0 && strlen(bufp) == 3) 
		    ch += bufp[2] & 31;
#ifdef MACINTOSH /* lines added by Luke Tierney, March 12, 1988 */
		  else if (strcmp(buf,"CHECK") == 0) ch =  0x12;
		  else if (strcmp(buf,"APPLE") == 0) ch =  0x14;
#endif /* MACINTOSH */ /* lines added by Luke Tierney, March 12, 1988 */
		  else xlerror("unknown character name",cvstring(buf));
#endif
		}
#ifdef __SASC__
		rplaca(val, cvchar(atoe(testch)));
#else
		rplaca(val,cvchar(ch));
#endif
		break;
    case ':':
	        rplaca(val,punintern(fptr));
		break;
    case '|':
    		pcomment(fptr);
		val = NIL;
		break;
    case 'c':
    case 'C':  /* From XLISP-STAT, Copyright (c) 1988, Luke Tierney */
      {
        LVAL list;
        readone(fptr, &list);
        if (! consp(list) || ! consp(cdr(list)) || cdr(cdr(list)) != NIL)
          xlerror("bad complex number specification", list);
        rplaca(val, newcomplex(car(list), car(cdr(list))));
        break;
      }
    case '+': /* From XLISP-STAT, Copyright (c) 1988, Luke Tierney */
    case '-':  
      {
        LVAL arg;
	LVAL olddenv;

        xlsave1(arg);

	olddenv = xldenv;
	xldbind(s_package, xlkeypack);
        while (! readone(fptr, &arg));
	xlunbind(olddenv);

        if (null(getvalue(s_read_suppress)) && checkfeatures(arg, ch)) {
          while (! readone(fptr, &arg));
          rplaca(val, arg);
        }
        else {
	  olddenv = xldenv;
	  xldbind(s_read_suppress, s_true);
          while (! readone(fptr, &arg));
          val = NIL;
	  xlunbind(olddenv);
        }
        xlpop();
        break;
      }
/*************************************************************************/
/*      Lines below added to allow for common lisp arrays                */
/*      Luke Tierney, March 1, 1988                                      */
/*************************************************************************/
	case '0':
	case '1':	
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		{
		  int rank = 0;

		  while (isdigit(ch)) {
		    rank = 10 * rank + ch - '0';
		    ch = xlgetc(fptr);
		  }
#ifdef PRINTCIRCLE
		  if (ch == '=') {
		    LVAL data;
		    if (! boundp(s_rdcircdat)) xlfail("no top level read");
		    if (!xlread(fptr,&val,TRUE))
		      badeof();
		    data = getvalue(s_rdcircdat);
		    rplacd(cdr(data), cons(cons(cvfixnum((FIXTYPE) rank),val),
					   cdr(cdr(data))));
		    val = consa(val);
		  }
		  else if (ch == '#') {
		    LVAL next;
		    int found = FALSE;
		    if (! boundp(s_rdcircdat)) xlfail("no top level read");
		    for (next = cdr(cdr(getvalue(s_rdcircdat)));
			 consp(next);
			 next = cdr(next)) {
		      if (consp(car(next)) && fixp(car(car(next)))) {
			if (getfixnum(car(car(next))) == rank) {
			  found = TRUE;
			  val = consa(cdr(car(next)));
			  break;
			}
		      }
		    }
		    if (! found) {
		      rplaca(cdr(getvalue(s_rdcircdat)), s_true);
		      val = consa(cons(car(getvalue(s_rdcircdat)),
				       cvfixnum((FIXTYPE) rank)));
		    }
		  }
		  else			
#endif /* PRINTCIRCLE */
		  if ((ch == 'A') || (ch == 'a')) {
		    readone(fptr, &val);
		    val = nested_list_to_array(val, rank);
		    val = consa(val);
		  }
		  else
		    xlfail("incomplete array specification");
		}
		break;
/*************************************************************************/
/*      Lines above added to allow for common lisp arrays                */
/*      Luke Tierney, March 1, 1988                                      */
/*************************************************************************/
#ifdef BYTECODE
	      case 'k':
	      case 'K':
		{
		  LVAL olddenv = xldenv;
		  LVAL arg;

		  xlsave1(arg);
		  xldbind(s_rtcase, k_upcase);
		  xldbind(s_rtable, getvalue(s_stdrtable));
		  xldbind(s_read_suppress, NIL);
		  readone(fptr, &arg);
		  xlunbind(olddenv);
		  rplaca(val, xlapplysubr(xlmakebcode, arg));
		  xlpop();
		}
		break;
#endif /* BYTECODE */
	      case '$':
		{
		  LVAL arg, *oldargv, *oldsp;
		  int oldargc;
		  
		  xlsave1(arg);
		  oldargv = xlargv;
		  oldargc = xlargc;
		  oldsp = xlsp;
		  xlargv = xlsp;
		  readone(fptr, &arg);
		  pusharg(s_true);
		  xlargc = 1;
		  for (; consp(arg); arg = cdr(arg)) {
		    pusharg(car(arg));
		    xlargc++;
		  }
		  rplaca(val, xmkrndstate());
		  xlsp = oldsp;
		  xlargc = oldargc;
		  xlargv = oldargv;
		  xlpop();
		}
		break;		  
    default:
/*      xlerror("illegal character after #",cvfixnum((FIXTYPE)ch)); */
	xlerror("illegal character after #",cvchar(ch)); /*TAA Mod */
    }

    /* restore the stack */
    xlpop();

    /* return the value */
    return (val);
}

/* rmquote - read macro for '\'' */
LVAL rmquote()
{
    LVAL fptr;

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* parse the quoted expression */
    return (consa(pquote(fptr,s_quote)));
}

/* rmdquote - read macro for '"' */
LVAL rmdquote()
{
    char buf[STRMAX+1],*p, *sptr;
    LVAL fptr,str,newstr;
    int len,blen,ch,d2,d3;

    /* protect some pointers */
    xlsave1(str);

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* loop looking for a closing quote */
    len = blen = 0; p = buf;
    while ((ch = checkeof(fptr)) != '"') {

	/* handle escaped characters */
	switch (ch) {
	case '\\':
		switch (ch = checkeof(fptr)) {
		case 't':
			ch = '\011';
			break;
		case 'n':
			ch = '\012';
			break;
		case 'f':
			ch = '\014';
			break;
		case 'r':
			ch = '\015';
			break;
		default:
			if (ch >= '0' && ch <= '7') {
			    d2 = checkeof(fptr);
			    d3 = checkeof(fptr);
			    if (d2 < '0' || d2 > '7'
			     || d3 < '0' || d3 > '7')
				xlfail("invalid octal digit");
			    ch -= '0'; d2 -= '0'; d3 -= '0';
			    ch = (ch << 6) | (d2 << 3) | d3;
			}
			break;
		}
	}

	/* check for buffer overflow */
	if (blen >= STRMAX) {
	    newstr = newstring(len + STRMAX);
	    sptr = getstring(newstr);
	    if (str != NIL)
		MEMCPY(sptr, getstring(str), len);
	    *p = '\0';
	    MEMCPY(sptr+len, buf, blen+1);
	    p = buf;
	    blen = 0;
	    len += STRMAX;
	    str = newstr;
	}

	/* store the character */
	*p++ = ch; ++blen;
    }

    /* append the last substring */
    if (str == NIL || blen) {
	newstr = newstring(len + blen);
	sptr = getstring(newstr);
	if (str != NIL) MEMCPY(sptr, getstring(str), len);
	*p = '\0';
	MEMCPY(sptr+len, buf, blen+1);
	str = newstr;
    }

    /* restore the stack */
    xlpop();

    /* return the new string */
    return (consa(str));
}

/* rmbquote - read macro for '`' */
LVAL rmbquote()
{
    LVAL fptr;

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* parse the quoted expression */
    return (consa(pquote(fptr,s_bquote)));
}

/* rmcomma - read macro for ',' */
LVAL rmcomma()
{
    LVAL fptr,sym;
    int ch;

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* check the next character */
    if ((ch = xlgetc(fptr)) == '@')
	sym = s_comat;
    else {
	xlungetc(fptr,ch);
	sym = s_comma;
    }

    /* make the return value */
    return (consa(pquote(fptr,sym)));
}

/* rmlpar - read macro for '(' */
LVAL rmlpar()
{
    LVAL fptr;

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* make the return value */
    return (consa(plist(fptr)));
}

/* rmrpar - read macro for ')' */
LVAL rmrpar()
{
    xlfail("misplaced close paren");
    return (NIL);   /* never returns */
}

/* rmsemi - read macro for ';' */
LVAL rmsemi()
{
    LVAL fptr;
    int ch;

    /* get the file and macro character */
    fptr = xlgetarg();  /* internal -- don't bother with error checks */

    /* skip to end of line */
    while ((ch = xlgetc(fptr)) != EOF && ch != '\n')
	;

    /* return nil (nothing read) */
    return (NIL);
}

/* pcomment - parse a comment delimited by #| and |# */
LOCAL VOID pcomment(fptr)
  LVAL fptr;
{
    int lastch,ch,n;

    /* look for the matching delimiter (and handle nesting) */
    for (n = 1, lastch = -1; n > 0 && (ch = xlgetc(fptr)) != EOF; ) {
	if (lastch == '|' && ch == '#')
	    { --n; ch = -1; }
	else if (lastch == '#' && ch == '|')
	    { ++n; ch = -1; }
	lastch = ch;
    }
}

/* pnumber - parse a number */
LOCAL LVAL pnumber(fptr,radix)
  LVAL fptr; int radix;
{
    int digit,ch;
    long num;
    
    for (num = 0L; (ch = xlgetc(fptr)) != EOF; ) {
	if (ISLOWER7(ch)) ch = toupper(ch);
	if (!('0' <= ch && ch <= '9') && !('A' <= ch && ch <= 'F'))
	    break;
	if ((digit = (ch <= '9' ? ch - '0' : ch - 'A' + 10)) >= radix)
	    break;
	num = num * (long)radix + (long)digit;
    }
    xlungetc(fptr,ch);
    return (cvfixnum((FIXTYPE)num));
}

/* plist - parse a list */
LOCAL LVAL plist(fptr)
  LVAL fptr;
{
    LVAL val,expr,lastnptr,nptr;

    /* protect some pointers */
    xlstkcheck(2);
    xlsave(val);
    xlsave(expr);

    /* keep appending nodes until a closing paren is found */
    for (lastnptr = NIL; nextch(fptr) != ')'; )

	/* get the next expression */
	switch (readone(fptr,&expr)) {
	case EOF:
	    badeof();
	case TRUE:

	    /* check for a dotted tail */
	    if (expr == s_dot) {

		/* make sure there's a node */
		if (lastnptr == NIL)
		    xlfail("invalid dotted pair");

		/* parse the expression after the dot */
		if (!xlread(fptr,&expr,TRUE))
		    badeof();
		rplacd(lastnptr,expr);

		/* make sure its followed by a close paren */
		if (nextch(fptr) != ')')
		    xlfail("invalid dotted pair");
	    }

	    /* otherwise, handle a normal list element */
	    else {
		nptr = consa(expr);
		if (lastnptr == NIL)
		    val = nptr;
		else
		    rplacd(lastnptr,nptr);
		lastnptr = nptr;
	    }
	    break;
	}

    /* skip the closing paren */
    xlgetc(fptr);

    /* restore the stack */
    xlpopn(2);

    /* return successfully */
    return (val);
}

/* pvector - parse a vector */
LOCAL LVAL pvector(fptr)
  LVAL fptr;
{
    LVAL list,val;
    int len,i;

    /* protect some pointers */
    xlsave1(list);

    /* read the list */
    list = readlist(fptr,&len);

    /* make a vector of the appropriate length */
    val = newvector(len);

    /* copy the list into the vector */
    for (i = 0; i < len; ++i, list = cdr(list))
	setelement(val,i,car(list));

    /* restore the stack */
    xlpop();

    /* return successfully */
    return (val);
}

/* pstruct - parse a structure */
LOCAL LVAL pstruct(fptr)
  LVAL fptr;
{
    LVAL list,val;
    int len;

    /* protect some pointers */
    xlsave1(list);

    /* read the list */
    list = readlist(fptr,&len);

    /* make the structure */
    val = xlrdstruct(list);

    /* restore the stack */
    xlpop();

    /* return successfully */
    return (val);
}

/* pquote - parse a quoted expression */
LOCAL LVAL pquote(fptr,sym)
  LVAL fptr,sym;
{
    LVAL val,p;

    /* protect some pointers */
    xlsave1(val);

    /* allocate two nodes */
    val = consa(sym);
    rplacd(val,consa(NIL));

    /* initialize the second to point to the quoted expression */
    if (!xlread(fptr,&p,TRUE))
	badeof();
    rplaca(cdr(val),p);

    /* restore the stack */
    xlpop();

    /* return the quoted expression */
    return (val);
}

/* psymbol - parse a symbol name */
#ifdef PACKAGES
LOCAL LVAL psymbol(fptr)
  LVAL fptr;
{
    int escflag, packindex;
    LVAL val, pack;
    int colons;
    char *p;

    pname(fptr,&escflag,&packindex);
    if (! null(getvalue(s_read_suppress))) return(NIL);
    if (escflag || packindex >= 0 || !isnumber(buf,&val)) {
      if (packindex >= 0) {
	/* check for zero-length name */
	if (buf[packindex+1] == 0) xlfail("zero length name after ':'");

	if (packindex == 0) {
	  /* count the colons */
	  for (p = buf + packindex + 1, colons = 1; *p == ':'; p++, colons++);
	  if (colons > 2) xlfail("too many :'s");
	  val = xlintern(p, xlkeypack);
	}
	else {
	  /* find the package */
	  buf[packindex] = 0;
	  pack = xlfindpackage(buf);
	  if (! packagep(pack))
	    xlerror("package not found", cvstring(buf));
	  
	  /* count the colons and switch */
	  for (p = buf + packindex + 1, colons = 1; *p == ':'; p++, colons++);
	  switch (colons) {
	  case 1:
	    if (xlfindsymbol(p, pack, &val) != SYM_EXTERNAL)
	      xlerror("external symbol not found", cvstring(p));
	    break;
	  case 2:
	    val = xlintern(p, pack);
	    break;
	  default: xlfail("too many :'s");
	  }
	}
      }
      else {
	pack = getvalue(s_package);
	return(goodpackagep(pack) ? xlintern(buf, pack) : NIL);
      }
    }
    return(val);
}
#else
LOCAL LVAL psymbol(fptr)
  LVAL fptr;
{
    int escflag;
    LVAL val;
    pname(fptr,&escflag);
    if (! null(getvalue(s_read_suppress))) return(NIL);
    return (escflag || !isnumber(buf,&val) ? xlenter(buf) : val);
}
#endif /* PACKAGES */

/* punintern - parse an uninterned symbol */
#ifdef PACKAGES
LOCAL LVAL punintern(fptr)
  LVAL fptr;
{
    int escflag,packindex;
    pname(fptr,&escflag,&packindex);
    return (xlmakesym(buf));
}
#else
LOCAL LVAL punintern(fptr)
  LVAL fptr;
{
    int escflag;
    pname(fptr,&escflag);
    return (xlmakesym(buf));
}
#endif /* PACKAGES */

/* pname - parse a symbol/package name */
#ifdef PACKAGES
LOCAL int pname(fptr,pescflag,ppackindex)
     LVAL fptr; int *pescflag, *ppackindex;
#else
LOCAL int pname(fptr,pescflag)
     LVAL fptr; int *pescflag;
#endif /* PACKAGES */
{
    int mode,ch,i;
    LVAL type;
#ifdef READTABLECASE
    LVAL rtcase = getvalue(s_rtcase);
    int low=0, up=0;
#endif

    /* initialize */
    *pescflag = FALSE;
#ifdef PACKAGES
    *ppackindex = -1;
#endif /* PACKAGES */
    mode = NORMAL;
    i = 0;

    /* accumulate the symbol name */
    while (mode != DONE) {

	/* handle normal mode */
	while (mode == NORMAL)
	    if ((ch = xlgetc(fptr)) == EOF)
		mode = DONE;
	    else if ((type = tentry(ch)) == k_sescape) {
		storech(&i,checkeof(fptr));
		*pescflag = TRUE;
	    }
	    else if (type == k_mescape) {
		*pescflag = TRUE;
		mode = ESCAPE;
	    }
	    else if (type == k_const
		 ||  (consp(type) && car(type) == k_nmacro))
#ifdef PACKAGES
	      {
		if (ch == ':') {
		  if (*ppackindex < 0) *ppackindex = i;
		  storech(&i,ch);
		}
		else
#endif /* PACKAGES */
#ifdef READTABLECASE
            {
                if (rtcase == k_preserve)
                    storech(&i,ch);
                else if (rtcase == k_downcase)
                    storech(&i,ISUPPER(ch) ? TOLOWER(ch) : ch);
                else if (rtcase == k_invert)
                    storech(&i,ISLOWER(ch) ? (low++, TOUPPER(ch)) : 
                        (ISUPPER(ch) ? (up++, TOLOWER(ch)) : ch));
                else   /*  default upcase  */
                    storech(&i,ISLOWER(ch) ? TOUPPER(ch) : ch);
            }
#else
		storech(&i,ISLOWER(ch) ? TOUPPER(ch) : ch);
#endif
#ifdef PACKAGES
	    }
#endif /* PACKAGES */
	    else
		mode = DONE;

	/* handle multiple escape mode */
	while (mode == ESCAPE)
	    if ((ch = xlgetc(fptr)) == EOF)
		badeof();
	    else if ((type = tentry(ch)) == k_sescape)
		storech(&i,checkeof(fptr));
	    else if (type == k_mescape)
		mode = NORMAL;
	    else
		storech(&i,ch);
    }
    buf[i] = 0;

#ifdef READTABLECASE    /* TAA Mod, sorta fixing a bug */
    if (rtcase == k_invert && low != 0 && up != 0) {
        /* must undo inversion (ugh!). Unfortunately, we don't know if
           any characters are quoted, so we'll just label this bug as
           a feature in the manual. The problem will only occur in symbols
           with mixed case characters outside of quotes and at least one
           quoted alpha character -- not very likely, I hope. */
        int cnt, c;
        for (cnt = 0; cnt < i; cnt++ ) {
            c = buf[cnt];
            if (ISUPPER(c)) buf[cnt] = TOLOWER(c);
            else if (ISLOWER(c)) buf[cnt] = TOUPPER(c);
        }
    }
#endif

    /* check for a zero length name */
    if (i == 0)
        xlfail("zero length name");     /* TAA fix, Jeff Prothero improved*/

    /* unget the last character and return it */
    xlungetc(fptr,ch);
    return (ch);
}

/* readlist - read a list terminated by a ')' */
LOCAL LVAL readlist(fptr,plen)
  LVAL fptr; int *plen;
{
    LVAL list,expr,lastnptr,nptr;
    int ch;

    /* protect some pointers */
    xlstkcheck(2);
    xlsave(list);
    xlsave(expr);

    /* get the open paren */
    if ((ch = nextch(fptr)) != '(')
	xlfail("expecting an open paren");
    xlgetc(fptr);

    /* keep appending nodes until a closing paren is found */
    for (lastnptr = NIL, *plen = 0; (ch = nextch(fptr)) != ')'; ) {

	/* check for end of file */
	if (ch == EOF)
	    badeof();

	/* get the next expression */
	switch (readone(fptr,&expr)) {
	case EOF:
	    badeof();
	case TRUE:
	    nptr = consa(expr);
	    if (lastnptr == NIL)
		list = nptr;
	    else
		rplacd(lastnptr,nptr);
	    lastnptr = nptr;
	    ++(*plen);
	    break;
	}
    }

    /* skip the closing paren */
    xlgetc(fptr);

    /* restore the stack */
    xlpopn(2);

    /* return the list */
    return (list);
}

/* storech - store a character in the print name buffer */
/* TAA MOD -- since buffer is always global buf, it is no longer passed
   as argument. also return value is stored in i, so i is now address of
   the int rather than its value */
LOCAL VOID storech(i,ch)
  int *i,ch;
{
    if (*i < STRMAX)
	buf[(*i)++] = ch;
}

/* tentry - get a readtable entry */
LVAL tentry(ch)
  int ch;
{
    LVAL rtable;
    rtable = getvalue(s_rtable);
    if (!vectorp(rtable) || ch < 0 || ch >= getsize(rtable))
	return (NIL);
    return (getelement(rtable,ch));
}

/* nextch - look at the next non-blank character */
LOCAL int nextch(fptr)
  LVAL fptr;
{
    int ch;

    /* return and save the next non-blank character */
    while ((ch = xlgetc(fptr)) != EOF && isspace(ch))
	;
    xlungetc(fptr,ch);
    return (ch);
}

/* checkeof - get a character and check for end of file */
LOCAL int checkeof(fptr)
  LVAL fptr;
{
    int ch;

    if ((ch = xlgetc(fptr)) == EOF)
	badeof();
    return (ch);
}

/* badeof - unexpected eof */
LOCAL VOID badeof()
{
    xlfail("EOF reached before expression end");
}

/* isnumber - check if this string is a number */
int isnumber(str,pval)
  char *str; LVAL *pval;
{
    int dl=0, dr=0;
    char *p = str;
#ifdef RATIOS
    int ratio=0;
    FIXTYPE denom=0;
#endif
    char *dp = NULL; /* 'D' added - L. Tierney */

    /* check for a sign */
    if (*p == '+' || *p == '-')
	p++;

    /* check for a string of digits */
    while (isdigit(*p))
	p++, dl++;

    /* check for a decimal point */
    if (*p == '.') {
	p++;
	while (isdigit(*p))
	    p++, dr++;
    }
#ifdef RATIOS
    else if (*p == '/') {
	if (dl == 0) return FALSE;
	p++;
	while (isdigit(*p)) {
	    denom = denom * 10 + *p - '0';
	    p++, dr++;
	}
	if (dr == 0) return FALSE;
	if (denom == 0)
	    xlerror ("invalid rational number", cvstring (str));
	ratio = 1;
    }
#endif

    /* check for an exponent */
#ifdef RATIOS
#ifdef READTABLECASE
    if ((dl || dr) && !ratio && (*p == 'E' || *p == 'e' || *p == 'D' || *p == 'd'))
#else
    if ((dl || dr) && !ratio && (*p == 'E' || *p == 'D'))
#endif
#else
#ifdef READTABLECASE
    if ((dl || dr) && (*p == 'E' || *p == 'e' || *p == 'D' || *p == 'd'))
#else
    if ((dl || dr) && (*p == 'E' || *p == 'D')) /* 'D' added - L. Tierney */
#endif
#endif
    {
	dp = p; /* 'D' added - L. Tierney */
	p++;

	/* check for a sign */
	if (*p == '+' || *p == '-')
	    p++;

	/* check for a string of digits */
	while (isdigit(*p))
	    p++, dr++;
    }

    /* make sure there was at least one digit and this is the end */
    if ((dl == 0 && dr == 0) || *p) return (FALSE);

    /* convert the string to an integer and return successfully */
    if (pval != NULL) {
      if (dp != NULL) *dp = 'E'; /* 'D' added - L. Tierney */
      if (*str == '+') ++str;
      if (str[strlen(str)-1] == '.') str[strlen(str)-1] = 0;
#ifdef RATIOS
	if (ratio) {
	    *pval = cvratio(ICNV(str), denom);
	}
	else
#endif
      *pval = (dr ? cvflonum(atof(str)) : cvfixnum(ICNV(str)));
    }
    return (TRUE);
}

/* defmacro - define a read macro */
LOCAL VOID defmacro(ch,type,offset)
     int ch; LVAL type; int offset;
{
    LVAL subr;
    subr = cvsubr(funtab[offset].fd_subr,funtab[offset].fd_type,offset);
    setelement(getvalue(s_rtable),ch,cons(type,subr));
}

/* callmacro - call a read macro */
LOCAL LVAL callmacro(fptr,ch)
  LVAL fptr; int ch;
{
    FRAMEP newfp;

    /* create the new call frame */
    newfp = xlsp;
    pusharg(cvfixnum((FIXTYPE)(newfp - xlfp)));
    pusharg(cdr(getelement(getvalue(s_rtable),ch)));
    pusharg(cvfixnum((FIXTYPE)2));
    pusharg(fptr);
    pusharg(cvchar(ch));
    xlfp = newfp;
    return (xlapply(2));
}

/* upcase - translate a string to upper case */
LOCAL VOID upcase(str)
  char *str;
{
    for (; *str != '\0'; ++str)
	if (ISLOWER7(*str))
	    *str = toupper(*str);
}

/* xlrinit - initialize the reader */
VOID xlrinit()
{
    LVAL rtable;
    char *p;
    int ch;

    /* create the read table */
    rtable = newvector(256);
    setsvalue(s_rtable,rtable);

    /* initialize the readtable */
    for (p = WSPACE; (ch = *p++) != 0; )
	setelement(rtable,ch,k_wspace);
    for (p = CONST1; (ch = *p++) != 0; )
	setelement(rtable,ch,k_const);
    for (p = CONST2; (ch = *p++) != 0; )
	setelement(rtable,ch,k_const);

#ifdef ASCII8
/* TAA MOD (8/92) to make extended ASCII character constituent */
    for (ch=128; ch < 255; ch++)
        setelement(rtable,ch,k_const);
#endif

    /* setup the escape characters */
    setelement(rtable,'\\',k_sescape);
    setelement(rtable,'|', k_mescape);

    /* install the read macros */
    defmacro('#', k_nmacro,FT_RMHASH);
    defmacro('\'',k_tmacro,FT_RMQUOTE);
    defmacro('"', k_tmacro,FT_RMDQUOTE);
    defmacro('`', k_tmacro,FT_RMBQUOTE);
    defmacro(',', k_tmacro,FT_RMCOMMA);
    defmacro('(', k_tmacro,FT_RMLPAR);
    defmacro(')', k_tmacro,FT_RMRPAR);
    defmacro(';', k_tmacro,FT_RMSEMI);
#ifdef BYTECODE
    defconstant(s_stdrtable,copyvector(rtable));
#endif /* BYTECODE */
}
