/* xllist.c - xlisp built-in list functions */
/* Copyright (c) 1989, by David Michael Betz.                            */
/* You may give out copies of this software; for conditions see the file */
/* COPYING included with this distribution.                              */

#include "xlisp.h"

struct nsubargs { /* TAA added 7/93 */
    LVAL to;    /* and ALIST */
    LVAL from;
    LVAL fcn;
#ifdef KEYARG
    LVAL kfcn;
#endif
    int tresult;
    int expr;
    int subst;
};

struct substargs {  /* TAA MOD - 7/93 to reduce stack usage */
    LVAL to;
    LVAL from;
    LVAL fcn;
#ifdef KEYARG
    LVAL kfcn;
#endif
    int tresult;
};

struct sublargs {   /* TAA MOD - 7/93 to reduce stack usage */
    LVAL alist;
    LVAL fcn;
#ifdef KEYARG
    LVAL kfcn;
#endif
    int tresult;
};

/* forward declarations */
LOCAL LVAL cxr _((char *adstr));
LOCAL LVAL nth _((int charflag));
LOCAL LVAL subst _((LVAL expr, struct substargs *args));
LOCAL LVAL sublis _((LVAL expr, struct sublargs *args));
#ifdef KEYARG
LOCAL LVAL assoc _((LVAL expr, LVAL alist, LVAL fcn, LVAL kfcn, int tresult));
LOCAL LVAL membr _((LVAL expr,LVAL list,LVAL fcn,LVAL kfcn,int tresult));
#else
LOCAL LVAL assoc _((LVAL expr, LVAL alist, LVAL fcn, int tresult));
LOCAL LVAL membr _((LVAL expr,LVAL list,LVAL fcn,int tresult));
#endif
LOCAL VOID splitlist _((LVAL pivot,LVAL list, LVAL *psmaller, LVAL *plarger, LVAL fcn));
LOCAL LVAL nsub _((int subst, int tresult, int expr));
LOCAL VOID nsub1 _((LVAL * tree, struct nsubargs *args));
LOCAL LVAL map _((int carflag, int valflag));
LOCAL LVAL sortlist _((LVAL list, LVAL fcn));
LOCAL LVAL gluelists _((LVAL smaller, LVAL pivot, LVAL larger));
LOCAL LVAL set_op _((int which));

/* xlcircular -- circular list error */
VOID xlcircular()
{
    xlfail("circular list");
}

/* xcar - take the car of a cons cell */
LVAL xcar()
{
    LVAL list;
    list = xlgalist();
    xllastarg();
    return (null(list) ? NIL : car(list));
}

/* xcdr - take the cdr of a cons cell */
LVAL xcdr()
{
    LVAL list;
    list = xlgalist();
    xllastarg();
    return (null(list) ? NIL : cdr(list));
}

/* cxxr functions */
LVAL xcaar() { return (cxr("aa")); }
LVAL xcadr() { return (cxr("da")); }
LVAL xcdar() { return (cxr("ad")); }
LVAL xcddr() { return (cxr("dd")); }

/* cxxxr functions */
LVAL xcaaar() { return (cxr("aaa")); }
LVAL xcaadr() { return (cxr("daa")); }
LVAL xcadar() { return (cxr("ada")); }
LVAL xcaddr() { return (cxr("dda")); }
LVAL xcdaar() { return (cxr("aad")); }
LVAL xcdadr() { return (cxr("dad")); }
LVAL xcddar() { return (cxr("add")); }
LVAL xcdddr() { return (cxr("ddd")); }

/* cxxxxr functions */
LVAL xcaaaar() { return (cxr("aaaa")); }
LVAL xcaaadr() { return (cxr("daaa")); }
LVAL xcaadar() { return (cxr("adaa")); }
LVAL xcaaddr() { return (cxr("ddaa")); }
LVAL xcadaar() { return (cxr("aada")); }
LVAL xcadadr() { return (cxr("dada")); }
LVAL xcaddar() { return (cxr("adda")); }
LVAL xcadddr() { return (cxr("ddda")); }
LVAL xcdaaar() { return (cxr("aaad")); }
LVAL xcdaadr() { return (cxr("daad")); }
LVAL xcdadar() { return (cxr("adad")); }
LVAL xcdaddr() { return (cxr("ddad")); }
LVAL xcddaar() { return (cxr("aadd")); }
LVAL xcddadr() { return (cxr("dadd")); }
LVAL xcdddar() { return (cxr("addd")); }
LVAL xcddddr() { return (cxr("dddd")); }

/* cxr - common car/cdr routine */
LOCAL LVAL cxr(adstr)
  char *adstr;
{
    LVAL list;

    /* get the list */
    list = xlgalist();
    xllastarg();

    /* perform the car/cdr operations */
    while (*adstr && consp(list))
	list = (*adstr++ == 'a' ? car(list) : cdr(list));

    /* make sure the operation succeeded */
    if (*adstr && !null(list))
	xlfail("bad argument");

    /* return the result */
    return (list);
}

/* xcons - construct a new list cell */
LVAL xcons()
{
    LVAL arg1,arg2;

    /* get the two arguments */
    arg1 = xlgetarg();
    arg2 = xlgetarg();
    xllastarg();

    /* construct a new list element */
    return (cons(arg1,arg2));
}

/* xlist - built a list of the arguments */
/* Rewritten by TAA for compactness and speed */
LVAL xlist()
{
    LVAL val;
    int i=xlargc;

    /* protect a pointer */
    xlsave1(val);

    /* do the work */
    while (i-- > 0)
        val = cons(xlargv[i],val);

    /* restore the stack */
    xlpop();

    /* return the list */
    return (val);
}

/* xliststar - built a list of the arguments */
/* by TAA */
LVAL xliststar()
{
    LVAL val;
    int i=xlargc;

    if (i==0) xltoofew();   /* must have at least one argument */

    /* protect a pointer */
    xlprot1(val);

    /* last argument is list tail */

    val = xlargv[--i];

    /* do the work */
    while (i-- > 0)
        val = cons(xlargv[i],val);

    /* restore the stack */
    xlpop();

    /* return the list */
    return (val);
}

/* xbutlast -- copy list for all but last n */
/* Added function TAA */

LVAL xbutlast()
{
    LVAL val,list,last,next;
    FIXTYPE n=1,l=0;

    /* get argument(s) */
    list = xlgalist();
    if (moreargs()) {
        n = getfixnum(next=xlgafixnum());
        if (n<0) xlerror("bad index",next);
        xllastarg();
    }

    /* get length */
    for (next=list; consp(next);) {
        next=cdr(next);
        l++;
        if (l > nnodes) xlcircular();
    }

    /* calc final length */
    l-=n;
    if (l <= 0) return (NIL);   /* nothing left */

    /* do the first cons */

    val = consa(car(list));
    if (l-- == 1) return val;

    /* protect a pointer */
    xlprot1(val);

    /* do remaining conses */
    last = val;
    while (l-- > 0) {
        list = cdr(list);
        next = consa(car(list));
        rplacd(last,next);
	last = next;
    }


    /* restore the stack */
    xlpop();

    /* return the list */
    return (val);
}


/* xappend - built-in function append */
LVAL xappend()
{
    LVAL list,last=NIL,next,val;
    long n;

    /* protect some pointers */
    xlsave1(val);

    /* append each argument */
    if (moreargs()) {
	while (xlargc > 1) {
            /* check for circular list (Added 5/6/94) */
            next = list = nextarg();
            for (n = 0; consp(next); next=cdr(next)) {
                if (n++ > nnodes) xlcircular(); /*DIRTY, but we loose anyway!*/
            }
	    /* append each element of this list to the result list */
	    for (; consp(list); list = cdr(list)) {
		next = consa(car(list));
		if (!null(val)) rplacd(last,next);
		else val = next;
		last = next;
	    }
	    if (!null(list)) xlbadtype(*--xlargv);  /*TAA added errormessage*/
	}

	/* handle the last argument */
	if (!null(val)) rplacd(last,nextarg());
	else val = nextarg();
    }

    /* restore the stack */
    xlpop();

    /* return the list */
    return (val);
}

/* xlast - return the last cons of a list */
LVAL xlast()
{
    LVAL list;
    long l=0;

    /* get the list */
    list = xlgalist();
    xllastarg();

    /* find the last cons */
    if (consp(list))            /* TAA fix */
      while (consp(cdr(list))) {
	list = cdr(list);
	if (l++ > nnodes) xlcircular();
      }

    /* return the last element */
    return (list);
}

/* xmember - built-in function 'member' */
LVAL xmember()
{
  LVAL x,list,slist,fcn,val;
  int tresult;
#ifdef KEYARG
  LVAL kfcn;

  /* protect some pointers */
  xlstkcheck(2);
  xlsave(fcn);
  xlsave(kfcn);
#else
  /* protect some pointers */
  xlsave1(fcn);
#endif

  /* get the expression to look for and the list */
  x = xlgetarg();
  slist = list = xlgalist();
  xltest(&fcn,&tresult);

#ifdef KEYARG
  kfcn = xlkey();
#endif

  xllastkey();

  /* look for the expression */
  for (val = NIL; consp(list); list = cdr(list), slist = cdr(slist)) {
    /* do a pair per iteration */

#ifdef KEYARG
    if (dotest2(x,car(list),fcn,kfcn) == tresult)
#else
    if (dotest2(x,car(list),fcn) == tresult)
#endif
      {
	val = list;
	break;
      }

    if (!consp(list = cdr(list))) break;
        
#ifdef KEYARG
    if (dotest2(x,car(list),fcn,kfcn) == tresult)
#else
    if (dotest2(x,car(list),fcn) == tresult)
#endif
      {
	val = list;
	break;
      }

    if (list == slist) break;   /* list must be circular, and no match */
  }

  /* restore the stack */
#ifdef KEYARG
  xlpopn(2);
#else
  xlpop();
#endif

  /* return the result */
  return (val);
}

/* xassoc - built-in function 'assoc' */
LVAL xassoc()
{
  LVAL x,alist,slist,fcn,pair,val;
  int tresult;
#ifdef KEYARG
  LVAL kfcn;

  /* protect some pointers */
  xlstkcheck(2);
  xlsave(fcn);
  xlsave(kfcn);
#else
  /* protect some pointers */
  xlsave1(fcn);
#endif

  /* get the expression to look for and the association list */
  x = xlgetarg();
  slist = alist = xlgalist();
  xltest(&fcn,&tresult);

#ifdef KEYARG
  kfcn = xlkey();
#endif

  xllastkey();

  /* look for the expression */
  for (val = NIL; consp(alist); alist = cdr(alist), slist = cdr(slist)) {
    /* do two iterations per loop */
    if ((!null(pair = car(alist))) && consp(pair))
#ifdef KEYARG
      if (dotest2(x,car(pair),fcn,kfcn) == tresult)
#else
      if (dotest2(x,car(pair),fcn) == tresult)
#endif
	{
	  val = pair;
	  break;
	}

    if (!consp(alist = cdr(alist))) break;

    if ((!null(pair = car(alist))) && consp(pair))
#ifdef KEYARG
      if (dotest2(x,car(pair),fcn,kfcn) == tresult)
#else
      if (dotest2(x,car(pair),fcn) == tresult)
#endif
	{
	  val = pair;
	  break;
	}

    if (slist == alist) break;  /* circular alist */
  }

  /* restore the stack */
#ifdef KEYARG
  xlpopn(2);
#else
  xlpop();
#endif

  /* return result */
  return (val);
}

/* xnsubst,xnsublis - destructive versions of subst and sublis */
/* ADDED 7/93 */
LOCAL VOID nsub1(tree, args)
     LVAL *tree; struct nsubargs *args;
{
  LVAL pair;
  FIXTYPE n=0;

tailrecursion:

#ifdef KEYARG
  if (args->subst? 
      (args->expr?
       (dotest2(args->from,*tree,args->fcn,args->kfcn)==args->tresult): 
       (dotest1(*tree, args->fcn, args->kfcn)==args->tresult)) :
      !null(pair=assoc(args->kfcn!=NIL?xlapp1(args->kfcn,*tree):*tree,args->to,args->fcn,NIL,args->tresult)))
#else
  if (args->subst? 
      (args->expr?
       (dotest2(args->from,*tree,args->fcn)==args->tresult): 
       (dotest1(*tree, args->fcn)==args->tresult)) :
      !null(pair=assoc(*tree,args->to,args->fcn,args->tresult)))
#endif
    {
      *tree = (args->subst ? args->to : cdr(pair));
    }
    else if (consp(*tree)) {
#ifdef STSZ	    /* This function is a good candidate for stack ov */
      stchck();
#endif
      nsub1(&car(*tree), args);
      tree = &cdr(*tree);
      if (++n > nnodes) 
	xlfail("circular list");    /* only the tip of the iceburg */
      goto tailrecursion;
    }
    else return;
}

LOCAL LVAL nsub(subst, tresult, expr)
     int subst, tresult, expr;
{
  struct nsubargs args;
  LVAL tree;
  /* protect some pointers */
#ifdef KEYARG
  xlstkcheck(2);
  xlsave(args.fcn);
  xlsave(args.kfcn);
#else
  xlsave1(args.fcn);
#endif

  args.subst = subst;
  args.tresult = tresult;
  args.expr = expr;

  if (expr) { /* get the expressions and the tree */
    args.to = xlgetarg();
    if (subst) args.from = xlgetarg();
    tree = xlgetarg();
    xltest(&args.fcn, &args.tresult);
  }
  else {
    /* get the result expression, the function and the tree */
    args.to = xlgetarg();
    args.fcn = xlgetarg();
    tree = xlgetarg();
  }
    
#ifdef KEYARG
  args.kfcn = xlkey();
#endif

  xllastkey();

  nsub1(&tree, &args);
    
#ifdef KEYARG
  xlpopn(2);
#else
  xlpop();
#endif

  return (tree);
}

LVAL xnsubst() { return nsub(TRUE, TRUE, TRUE);}
LVAL xnsubstif() { return nsub(TRUE, TRUE, FALSE); }
LVAL xnsubstifnot() { return nsub(TRUE, FALSE, FALSE); }
LVAL xnsublis() { return nsub(FALSE, TRUE, TRUE);}

/* xsubst - substitute one expression for another */
LVAL xsubst()
{
    struct substargs args;
    LVAL expr;

    /* protect some pointers */
#ifdef KEYARG
    xlstkcheck(2);
    xlsave(args.fcn);
    xlsave(args.kfcn);
#else
    xlsave1(args.fcn);
#endif

    /* get the to value, the from value and the expression */
    args.to = xlgetarg();
    args.from = xlgetarg();
    expr = xlgetarg();
    xltest(&args.fcn,&args.tresult);

#ifdef KEYARG
    args.kfcn = xlkey();
#endif

    xllastkey();

    /* do the substitution */
    expr = subst(expr,&args);

    /* restore the stack */
#ifdef KEYARG
    xlpopn(2);
#else
    xlpop();
#endif

    /* return the result */
    return (expr);
}

/* subst - substitute one expression for another */
LOCAL LVAL subst(expr,args)
  LVAL expr; struct substargs *args;
{
    LVAL carval,cdrval;

#ifdef KEYARG
    if (dotest2(args->from,expr,args->fcn,args->kfcn) == args->tresult)
#else
    if (dotest2(args->from,expr,args->fcn) == args->tresult)
#endif
	return (args->to);
    else if (consp(expr)) {
#ifdef STSZ	    /* This function is a good candidate for stack ov */
        stchck();
#endif
	xlsave1(carval);
        carval = subst(car(expr),args);
        cdrval = subst(cdr(expr),args);
	xlpop();

	/* the following TAA mod makes subst like COMMON LISP */
        if ((carval == car(expr)) && (cdrval == cdr(expr)))
            return expr; /* no change */
        else
	return (cons(carval,cdrval));
    }
    else
	return (expr);
}

/* xsublis - substitute using an association list */
LVAL xsublis()
{
    struct sublargs args;
    LVAL expr;

    /* protect some pointers */
#ifdef KEYARG
    xlstkcheck(2);
    xlsave(args.fcn);
    xlsave(args.kfcn);
#else
    xlsave1(args.fcn);
#endif

    /* get the assocation list and the expression */
    args.alist = xlgalist();
    expr = xlgetarg();
    xltest(&args.fcn,&args.tresult);

#ifdef KEYARG
    args.kfcn = xlkey();
#endif

    xllastkey();

    /* do the substitution */
    expr = sublis(expr,&args);

    /* restore the stack */
#ifdef KEYARG
    xlpopn(2);
#else
    xlpop();
#endif

    /* return the result */
    return (expr);
}

/* sublis - substitute using an association list */
LOCAL LVAL sublis(expr,args)
  LVAL expr; struct sublargs *args;
{
    LVAL carval,cdrval,pair;

#ifdef KEYARG
    if (!null(pair = assoc(args->kfcn!=NIL?
			   xlapp1(args->kfcn,expr):
			   expr,
			   args->alist,
			   args->fcn,
			   NIL,
			   args->tresult)))
#else
    if (!null(pair = assoc(expr,args->alist,args->fcn,args->tresult)))
#endif
	return (cdr(pair));
    else if (consp(expr)) {
#ifdef STSZ	    /* This function is a good candidate for stack ov */
	stchck();
#endif
	xlsave1(carval);
        carval = sublis(car(expr),args);
        cdrval = sublis(cdr(expr),args);
	xlpop();
	/* TAA MOD for making like common lisp */
        if ((car(expr) == carval) && (cdr(expr) == cdrval))
            return (expr);
        else
	return (cons(carval,cdrval));
    }
    else
	return (expr);
}

/* assoc - find a pair in an association list */
#ifdef KEYARG
LOCAL LVAL assoc(expr,alist,fcn,kfcn,tresult)
  LVAL expr,alist,fcn,kfcn; int tresult;
#else
LOCAL LVAL assoc(expr,alist,fcn,tresult)
  LVAL expr,alist,fcn; int tresult;
#endif
{
    LVAL pair;

    for (; consp(alist); alist = cdr(alist))
	if ((!null((pair = car(alist)))) && consp(pair))
#ifdef KEYARG
            if (dotest2(expr,car(pair),fcn,kfcn) == tresult)
#else
	    if (dotest2(expr,car(pair),fcn) == tresult)
#endif
		return (pair);
    return (NIL);
}

/* xnth - return the nth element of a list */
LVAL xnth()
{
    return (nth(TRUE));
}

/* xnthcdr - return the nth cdr of a list */
LVAL xnthcdr()
{
    return (nth(FALSE));
}

/* nth - internal nth function */
LOCAL LVAL nth(carflag)
  int carflag;
{
    LVAL list,num;
    FIXTYPE n;

    /* get n and the list */
    num = xlgafixnum();
/*  list = xlgacons(); */
    list = xlgalist();      /* TAA fix */
    xllastarg();

    /* make sure the number isn't negative */
    if ((n = getfixnum(num)) < 0)
	xlfail("bad argument");

    /* find the nth element */
    while (consp(list) && --n >= 0)
	list = cdr(list);

    /* return the list beginning at the nth element */
    return (carflag && consp(list) ? car(list) : list);
}

/* xlength - return the length of a list or string */
LVAL xlength()
{
    FIXTYPE n;
    LVAL arg;

    /* get the list or string */
    arg = xlgetarg();
    xllastarg();

    /* find the length of a list */
    if (listp(arg))
	for (n = 0; consp(arg);) {
	    arg = cdr(arg);
	    n++;
	    if (n > nnodes) xlcircular();   /*DIRTY, but we loose anyway!*/
	}

    /* find the length of a string */
    else if (stringp(arg))
	n = (FIXTYPE)getslength(arg);

    /* find the length of a typed vector */
    else if (tvecp(arg))
        n = (FIXTYPE)gettvecsize(arg);

    /* find the length of a vector */
    else if (vectorp(arg))
	n = (FIXTYPE)getsize(arg);

    /* otherwise, bad argument type */
    else
	xlbadtype(arg);

    /* return the length */
    return (cvfixnum(n));
}

/* xlistlength -- return the length of a list */
LVAL xlistlength()
{
    FIXTYPE n = 0;
    LVAL arg, sarg;
    
    /* get the list */
    arg = sarg = xlgalist();
    xllastarg();
    
    while (consp(arg)) {
        arg = cdr(arg);
        if (!consp(arg)) { n++; break; }
        if (sarg == arg) return NIL;    /* circular list */
        arg = cdr(arg);
        sarg = cdr(sarg);
        n += 2;
    }
    
    /* return the length */
    return (cvfixnum(n));
}


/* map - internal mapping function */
#define CONCAT 2    /* third choice for valflag */

LOCAL LVAL map(carflag,valflag)
     int carflag,valflag;
{
    FRAMEP newfp;
    LVAL fun,lists,val,last,p,x,y;
    int argc;
    long n=0, nmax=nnodes;

    /* protect some pointers */
    xlstkcheck(3);
    xlsave(fun);
    xlsave(lists);
    xlsave(val);

    /* get the function to apply and the first list */
    fun = xlgetarg();
    lists = xlgalist();

    /* initialize the result list */
    val = (valflag ? NIL : lists);

    /* build a list of argument lists */
    argc = 1;
    for (lists = last = consa(lists); moreargs(); last = cdr(last)) {
	argc++;
	rplacd(last,cons(xlgalist(),NIL));
    }

    /* loop through each of the argument lists */
    for (;;) {

        if (n++ > nmax) xlcircular();

	/* build an argument list from the sublists */
	newfp = xlsp;
	pusharg(cvfixnum((FIXTYPE)(newfp - xlfp)));
	pusharg(fun);
	pusharg(cvfixnum((FIXTYPE)argc));
	for (x = lists; (consp(x)) && (consp(y = car(x))); x = cdr(x)) {
	    pusharg(carflag ? car(y) : y);
	    rplaca(x,cdr(y));
	}

	/* quit if any of the lists were empty */
	if (!null(x)) {
	    xlsp = newfp;
	    break;
	}

	/* apply the function to the arguments */
	xlfp = newfp;
        switch (valflag) {
        case CONCAT:
            p = xlapply(argc);
            if (!null(p)) {
                if (!consp(p)) xlerror("non-list to concatenate", p);
                if (null(val)) val = p;
                else rplacd(last, p);
                while (consp(cdr(p))) p = cdr(p); /* find end--no circular check */
                last = p;
            }
            break;

        case TRUE:
	    p = consa(xlapply(argc));
            if (!null(val)) rplacd(last,p);
	    else val = p;
	    last = p;
            break;

        case FALSE:
	    xlapply(argc);
            break;
    }
    }

    /* restore the stack */
    xlpopn(3);

    /* return the last test expression value */
    return (val);
}

/* xmapc - built-in function 'mapc' */
LVAL xmapc()
{
    return (map(TRUE,FALSE));
}

/* xmapcar - built-in function 'mapcar' */
LVAL xmapcar()
{
    return (map(TRUE,TRUE));
}

/* xmapl - built-in function 'mapl' */
LVAL xmapl()
{
    return (map(FALSE,FALSE));
}

/* xmaplist - built-in function 'maplist' */
LVAL xmaplist()
{
    return (map(FALSE,TRUE));
}

/* xmapcan - built-in function 'mapcan' */
LVAL xmapcan()
{
    return (map(TRUE,CONCAT));
}

/* xmapcon - built-in function 'mapcon' */
LVAL xmapcon()
{
    return (map(FALSE,CONCAT));
}

/* xrplca - replace the car of a list node */
LVAL xrplca()
{
    LVAL list,newcar;

    /* get the list and the new car */
    list = xlgacons();
    newcar = xlgetarg();
    xllastarg();

    /* replace the car */
    rplaca(list,newcar);

    /* return the list node that was modified */
    return (list);
}

/* xrplcd - replace the cdr of a list node */
LVAL xrplcd()
{
    LVAL list,newcdr;

    /* get the list and the new cdr */
    list = xlgacons();
    newcdr = xlgetarg();
    xllastarg();

    /* replace the cdr */
    rplacd(list,newcdr);

    /* return the list node that was modified */
    return (list);
}

/* xnconc - destructively append lists */
LVAL xnconc()
{
    LVAL next,last=NIL,val=NIL;
    long l; /* TAA MOD */

    /* concatenate each argument */
    if (moreargs()) {
	while (xlargc > 1) {

            /* TAA mod -- give error message if not a list */
	    if ((!null(next = nextarg())) && consp(next)) {

		/* concatenate this list to the result list */
		if (!null(val)) rplacd(last,next);
		else val = next;

		/* find the end of the list */
                l = 0;
		while (consp(cdr(next))) {
		    next = cdr(next);
                    if (l++ > nnodes) xlcircular();
                }
		last = next;
	    }
            else if (!null(next)) xlbadtype(*--xlargv); /* TAA -- oops! */
	}

	/* handle the last argument */
	if (!null(val)) rplacd(last,nextarg());
	else val = nextarg();
    }

    /* return the list */
    return (val);
}

/*
    This sorting algorithm is based on a Modula-2 sort written by
    Richie Bielak and published in the February 1988 issue of
    "Computer Language" magazine in a letter to the editor.
*/

/* gluelists - glue the smaller and larger lists with the pivot */
LOCAL LVAL gluelists(smaller,pivot,larger)
     LVAL smaller,pivot,larger;
{
    LVAL last;

    /* larger always goes after the pivot */
    rplacd(pivot,larger);

    /* if the smaller list is empty, we're done */
    if (null(smaller))
	return (pivot);

    /* append the smaller to the front of the resulting list */
    for (last = smaller; consp(cdr(last)); last = cdr(last))
	;
    rplacd(last,pivot);
    return (smaller);
}

/* sortlist - sort a list using quicksort */
/* TAA Note: this routine suffers from the typical problem of Quicksort
   when data is already sorted. Some eager beaver might want to improve
   the algorithm. much as I did for the xqsort function here */

LOCAL LVAL sortlist(list,fcn)
     LVAL list,fcn;
{
  LVAL smaller,pivot,larger;

  /* protect some pointers */
  xlstkcheck(3);
  xlsave(smaller);
  xlsave(pivot);
  xlsave(larger);

  /* lists with zero or one element are already sorted */
  if (consp(list) && consp(cdr(list))) {
    if (consp(cdr(cdr(list)))) {
      pivot = list; list = cdr(list);
      splitlist(pivot,list,&smaller,&larger,fcn);
      smaller = sortlist(smaller,fcn);
      larger = sortlist(larger,fcn);
      list = gluelists(smaller,pivot,larger);
    }
    else { /* just two elements in the list */
      /* TAA MOD, 7/93 for greater sorting efficiency */
#ifdef KEYARG
      pivot = (null(cdr(fcn))? car(list) : xlapp1(cdr(fcn),car(list)));
      if (dotest2((!null(cdr(fcn)))?
		  xlapp1(cdr(fcn),car(cdr(list))):car(cdr(list)),
		  pivot,car(fcn),NIL) )
#else
      if (dotest2(car(cdr(list)),car(list),fcn))
#endif
	{
	  pivot = car(list);
	  rplaca(list,car(cdr(list)));
	  rplaca(cdr(list),pivot);
	}
    }
  }

  /* cleanup the stack and return the sorted list */
  xlpopn(3);
  return (list);
}

/* splitlist - split the list around the pivot */
LOCAL VOID splitlist(pivot,list,psmaller,plarger,fcn)
  LVAL pivot,list,*psmaller,*plarger,fcn;
{
  LVAL next,smaller,larger;
#ifdef KEYARG
  LVAL tmp=car(pivot);
#endif

  /* In case of garbage collection TAA Mod thanx to Neal Holtz */
#ifdef KEYARG
  xlstkcheck(5);
  xlprotect(tmp);
#else
  xlstkcheck(4);
#endif
  xlprotect(list);
  xlsave(next);
  xlsave(smaller);
  xlsave(larger);

  /* initialize the result lists */
  smaller = larger = NIL;

#ifdef KEYARG
  if (!null(cdr(fcn))) tmp = xlapp1(cdr(fcn),tmp);
#endif

  /* split the list */
  for (; consp(list); list = next) {
    next = cdr(list);
#ifdef KEYARG
    if (dotest2((!null(cdr(fcn)))?xlapp1(cdr(fcn),car(list)):car(list),
		tmp,car(fcn),NIL) )
#else
    if (dotest2(car(list),car(pivot),fcn))
#endif
      {
	rplacd(list, smaller);
	smaller = list;
      }
    else {
      rplacd(list, larger);
      larger = list;
    }
  }

  /* restore the stack */
#ifdef KEYARG
  xlpopn(5);
#else
  xlpopn(4);
#endif
  *psmaller = smaller;
  *plarger = larger;
}

/* xsort - built-in function 'sort' */
LVAL xsort()
{
  LVAL list,fcn;

  /* protect some pointers */
  xlstkcheck(2);
  xlsave(list);
  xlsave(fcn);

  /* get the list to sort and the comparison function */
  list = xlgetarg();

#ifdef KEYARG
  fcn = cons(NIL,NIL);
  rplaca(fcn,xlgetarg());
  rplacd(fcn,xlkey());
#else
  fcn = xlgetarg();
#endif
  xllastkey();

  /* sort the list */
  if (!null(list)) {
    switch (ntype(list)) {
    case VECTOR:
    case STRING:
    case TVEC:
      {
	LVAL tlist;
	int i, n;

	xlsave1(tlist);
	tlist = coerce_to_list(list);
	tlist = sortlist(tlist,fcn);
	for (i = 0, n = gettvecsize(list);
	     i < n && consp(tlist);
	     i++, tlist = cdr(tlist)) {
	  settvecelement(list, i, car(tlist));
	}
	xlpop();
      }
      break;
    case CONS:
      list = sortlist(list,fcn);
      break;
    default: xlbadtype(list);
    }
  }

  /* restore the stack and return the sorted list */
  xlpopn(2);
  return (list);
}

/* These functions have the following copyright notice: */
/* XLISP-STAT 2.0 Copyright (c) 1988, by Luke Tierney                  */
/*      All Rights Reserved                                            */
/*      Permission is granted for unrestricted non-commercial use      */

/* membr - internal MEMBER for set functions TAA */
#ifdef KEYARG
LOCAL LVAL membr(expr,list,fcn,kfcn,tresult)
  LVAL expr,list,fcn,kfcn; int tresult;
{
    xlprot1(expr);
    if (!null(kfcn)) expr = xlapp1(kfcn,expr);
    for (; consp(list); list = cdr(list))
        if (dotest2(expr,car(list),fcn,kfcn) == tresult) {
            xlpop();
            return (list);
        }
    xlpop();
    return (NIL);
}
    
#else
LOCAL LVAL membr(expr,list,fcn,tresult)
  LVAL expr,list,fcn; int tresult;
{
    for (; consp(list); list = cdr(list))
        if (dotest2(expr,car(list),fcn) == tresult)
                return (list);
    return (NIL);
}
#endif

/* Common Lisp ADJOIN function */
LVAL xadjoin()
{
    LVAL x, list, fcn;
    int tresult;
#ifdef KEYARG
    LVAL kfcn;

    /* protect some pointers */
    xlstkcheck(2);
    xlsave(fcn);
    xlsave(kfcn);
#else
    xlsave1(fcn);
#endif

    /* get the lists and key arguements, if any */
    x = xlgetarg();
    list = xlgalist();
    xltest(&fcn,&tresult);
#ifdef KEYARG
    kfcn = xlkey();
#endif

    xllastkey();

#ifdef KEYARG
    if (null(membr(x,list,fcn,kfcn,tresult))) list = cons(x,list) ;
    xlpopn(2);
#else
    if (null(membr(x,list,fcn,tresult))) list = cons(x,list) ;
    xlpop();
#endif

    return list;
}

LOCAL LVAL set_op(which)
     int which;
{
    LVAL x, list1, list2, result, fcn;
    int tresult;
#ifdef KEYARG
    LVAL kfcn;

    /* protect some pointers */
    xlstkcheck(3);
    xlsave(kfcn);
#else
    
    /* protect some pointers */
    xlstkcheck(2);
#endif
    xlsave(fcn);
    xlsave(result);

    /* get the lists and key arguements, if any */
    list1 = xlgalist();
    list2 = xlgalist();
    xltest(&fcn,&tresult);
#ifdef KEYARG
    kfcn = xlkey();
#endif

    xllastkey();

    switch(which) {
        case 'U':
            for (result = list1; consp(list2); list2 = cdr(list2)) {
                x = car(list2);
#ifdef KEYARG
                if (null(membr(x,list1,fcn,kfcn,tresult)))
#else
                if (null(membr(x,list1,fcn,tresult)))
#endif
                    result = cons(x, result);
    }
            break;
        case 'I':
            for (result = NIL; consp(list2); list2 = cdr(list2)) {
                x = car(list2);
#ifdef KEYARG
                if (!null(membr(x,list1,fcn,kfcn,tresult)))
#else
                if (!null(membr(x,list1,fcn,tresult)))
#endif
                    result = cons(x, result);
            }
            break;
        case 'D':
            for (result = NIL; consp(list1); list1 = cdr(list1)) {
                x = car(list1);
#ifdef KEYARG
                if (null(membr(x,list2,fcn,kfcn,tresult)))
#else
                if (null(membr(x,list2,fcn,tresult)))
#endif
                    result = cons(x, result);
            }
            break;
        case 'S':
            for (result = s_true; consp(list1); list1 = cdr(list1)) {
                x = car(list1);
#ifdef KEYARG
                if (null(membr(x,list2,fcn,kfcn,tresult)))
#else
                if (null(membr(x,list2,fcn,tresult)))
#endif
                {
                    result = NIL;
                    break;
                }
            }
            break;
    }

#ifdef KEYARG
    xlpopn(3);
#else
    xlpopn(2);
#endif
    return(result);
}

LVAL xunion()          { return(set_op('U')); }
LVAL xintersection()   { return(set_op('I')); }
LVAL xset_difference() { return(set_op('D')); }
LVAL xsubsetp()        { return(set_op('S')); }


/* HASH TABLES ARE IMPLEMENTED AS STRUCTS, WITHOUT ACCESSING FCNS */

#ifdef HASHFCNS
/* The hash tables have been modified to allow fast EQ, EQL and EQUAL
hashing by using addresses in the hash function. Since addresses are
not preserved accross save/restores, xlimage and dlimage use uflags to
mark hash tables for rehashing on the next access. (xlimage and
dlimage set uflags to TRUE for any STRUCT. If a proper hashtable type
is introduced, this should be changed.) In addition, the
:REHASH-THRESHOLD and :REHASH-SIZE keywords are now supported and hash
tables are resized when new entries are added that push the count over
the threshold. To simplify resizing, the hash table is now a fixed
size structure that contains its data as a vector in one of its slots.
The count is also maintained in a slot.
*/

#define HTABSIZE 6
#define HTAB_REHASH_THRESHOLD 0.8
#define HTAB_REHASH_SIZE 1.3
#define MAXHTABSIZE MAXSLEN

#define hashtablep(x)      (structp(x) && (getelement(x,0) == a_hashtable))
#define xlgahashtable()    (testarg(typearg(hashtablep)))

#define hashtablerehash(x)        nuflags(x)
#define sethashtablerehash(x,y)   setnuflags(x,y)

#define hashtablesize(x)          getsize(hashtabledata(x))
#define hashtablefun(x)           getelement(x,1)
#define sethashtablefun(x,fun)    setelement(x,1,fun)
#define hashtablecount(x)         getfixnum(getelement(x,2))
#define sethashtablecount(x,n)    setelement(x,2,cvfixnum((FIXTYPE)n))
#define hashtabledata(x)          getelement(x,3)
#define sethashtabledata(x,d)     setelement(x,3,d)
#define hashtablerhthresh(x)      getelement(x,4)
#define sethashtablerhthresh(x,v) setelement(x,4,v)
#define hashtablerhsize(x)        getelement(x,5)
#define sethashtablerhsize(x,v)   setelement(x,5,v)

#define hashtablelist(x,i)        getelement(hashtabledata(x),(i))
#define sethashtablelist(x,i,v)   setelement(hashtabledata(x),(i),v)


LOCAL VOID rehash_hashdata _((LVAL old, LVAL new, LVAL fun));
LOCAL VOID rehash_hashtable _((LVAL table));
LOCAL unsigned FIXTYPE eqlhash _((LVAL x));
LOCAL unsigned FIXTYPE equalhash _((LVAL x));
LOCAL int hthash _((LVAL x, int len, LVAL fun));


LOCAL VOID rehash_hashdata(old, new, fun)
     LVAL old, new, fun;
{
  LVAL next;
  int i, j, oldsize, newsize;

  oldsize = getsize(old);
  newsize = getsize(new);

  for (i = 0; i < oldsize; i++) {
    for (next = getelement(old, i); consp(next); next = cdr(next)) {
      j = hthash(car(car(next)), newsize, fun);
      setelement(new, j, cons(car(next), getelement(new, j)));
    }
  }
}

LOCAL VOID rehash_hashtable(table)
     LVAL table;
{
  LVAL new;

  xlsave1(new);
  new = newvector((int) hashtablesize(table));
  rehash_hashdata(hashtabledata(table), new, hashtablefun(table));
  sethashtabledata(table, new);
  sethashtablerehash(table, FALSE);
  xlpop();
}  

LOCAL unsigned FIXTYPE eqlhash(x)
     LVAL x;
{
  union {FIXTYPE i; FLOTYPE j; unsigned FIXTYPE k;} swizzle;
  unsigned FIXTYPE temp;

  switch (ntype(x)) {
  case FIXNUM:
    swizzle.i = getfixnum(x);
    return swizzle.k;
  case FLONUM:
    swizzle.j = getflonum(x);
    return swizzle.k;
  case COMPLEX:
    if (fixp(realpart(x))) {
      swizzle.i = getfixnum(realpart(x));
      temp = swizzle.k;
      swizzle.i = getfixnum(imagpart(x));
    }
    else {
      swizzle.j = getflonum(realpart(x));
      temp = swizzle.k;
      swizzle.j = getflonum(imagpart(x));
    }
    temp = (temp << 2) ^ swizzle.k;
    return temp;
  default:
    return (unsigned FIXTYPE) CVPTR(x);
  }
}

LOCAL unsigned FIXTYPE equalhash(x)
     LVAL x;
{
  unsigned FIXTYPE temp;

  temp = 0;
 hashloop:
  switch (ntype(x)) {
  case STRING:
    {
      char *str = getstring(x);
      while (*str != 0)
	temp = (temp << 2) ^ *str++;
      return temp;
    }
  case CONS:
    temp = (temp << 2) ^ equalhash(car(x));
    x = cdr(x);
    goto hashloop;
  default:
    return (temp << 2) ^ eqlhash(x);
  }
}
    
LOCAL int hthash(x, len, fun)
     LVAL x, fun;
     int len;
{
  if (fun == getfunction(s_eq))
    return (int) (CVPTR(x) % len);
  else if (fun == getfunction(s_eql))
    return (int) (eqlhash(x) % len);
  else if (fun == getfunction(s_equal))
    return (int) (equalhash(x) % len);
  else
    return xlhash(x, len);
}

/* Hash table functions from Ken Whedbee */
LVAL xmakehash()    /* rewritten by TAA */
{
  LVAL size, testfcn, result, temp;
  double rhthresh, rhsize;
  FIXTYPE len;
    
  if (xlgetkeyarg(k_size,&size)) {
    if (!fixp(size) || (len=getfixnum(size)) < 1)
      xlbadtype(size);
  }
  else len = 31;
  if (len % 2 == 0) len++;
  if (len < 1) xlfail("size out of bounds"); /**** check MAXSLEN */

  if (!xlgetkeyarg(k_test,&testfcn)) testfcn = getfunction(s_eql);
  if (symbolp(testfcn) && fboundp(testfcn)) testfcn = getfunction(testfcn);

  if (!xlgetkeyarg(k_rhthresh, &temp)) temp = NIL;
  if (floatp(temp) && getflonum(temp) > 0.0 && getflonum(temp) < 1.0)
    rhthresh = getflonum(temp);
  else
    rhthresh = HTAB_REHASH_THRESHOLD;
  
  if (!xlgetkeyarg(k_rhsize, &temp)) temp = NIL;
  if (fixp(temp) && getfixnum(temp) > 0)
    rhsize = ((double) getfixnum(temp) + len) / len;
  else if (floatp(temp) && getflonum(temp) > 1.0)
    rhsize = getflonum(temp);
  else
    rhsize = HTAB_REHASH_SIZE;

  xllastkey();
    
  xlprot1(testfcn);

  result = newstruct(a_hashtable,HTABSIZE-1);

  sethashtablerehash(result, FALSE);
  sethashtablefun(result, testfcn);
  sethashtablecount(result,0);
  sethashtabledata(result,newvector((int)len));
  sethashtablerhthresh(result, cvflonum((FLOTYPE) rhthresh));
  sethashtablerhsize(result, cvflonum((FLOTYPE) rhsize));

  xlpop();

  return result;
}

LVAL xgethash()
{
  LVAL alist,val,key,fun,table,def=NIL;

  key = xlgetarg();
  table = xlgahashtable();
  if (moreargs()) {
    def = xlgetarg();
    xllastarg();
  }

  if (hashtablerehash(table))
    rehash_hashtable(table);

  fun = hashtablefun(table);
  alist = hashtablelist(table, hthash(key,(int)hashtablesize(table),fun));

#ifdef KEYARG
  val = assoc(key,alist,fun,NIL,TRUE);
#else
  val = assoc(key,alist,fun,TRUE);
#endif

  /* return result */
#ifdef MULVALS
  xlnumresults = 2;
  if (null(val)) {
    xlresults[0] = def;
    xlresults[1] = NIL;
  }
  else {
    xlresults[0] = cdr(val);
    xlresults[1] = s_true;
  }
  return(xlresults[0]);
#else
  return (null(val) ? def : cdr(val));
#endif /* MULVALS */
}

LVAL xremhash()
/* By TAA -- can't use assoc here*/
{
  LVAL alist,key,fun,table,last;

  int idx;

  key = xlgetarg();
  table = xlgahashtable();
  xllastarg();

  if (hashtablerehash(table))
    rehash_hashtable(table);

  fun = hashtablefun(table);
  idx = hthash(key,(int)hashtablesize(table),fun);

  alist = hashtablelist(table,idx);

  if (null(alist))
    return NIL;

#ifdef KEYARG
  else if (dotest2(key,car(car(alist)),fun,NIL)==TRUE)
#else
  else if (dotest2(key,car(car(alist)),fun)==TRUE)
#endif
    {
      sethashtablelist(table,idx,cdr(alist));   /* matches first element */
      sethashtablecount(table,hashtablecount(table)-1);
      return s_true;
    }
  else {
    last = alist;
    alist = cdr(alist);
    while (consp(alist)) {
#ifdef KEYARG
      if (dotest2(key,car(car(alist)),fun,NIL)==TRUE)
#else
      if (dotest2(key,car(car(alist)),fun)==TRUE)
#endif
	{
	  rplacd(last,cdr(alist));
	  sethashtablecount(table,hashtablecount(table)-1);
	  return s_true;
	}
      last = alist;
      alist = cdr(alist);
    }
  }
    
  return NIL;
}
  
VOID xlsetgethash(key,table,value)
     LVAL key,table,value;
{
  LVAL alist,fun,oldval;
  int idx;

  if (! hashtablep(table))
    xlbadtype(table);

  if (hashtablerehash(table))
    rehash_hashtable(table);

  fun = hashtablefun(table);
  idx = hthash(key,(int)hashtablesize(table),fun);

  alist = hashtablelist(table,idx);

#ifdef KEYARG
  if (!null(oldval = assoc(key,alist,fun,NIL,TRUE)))
#else
  if (!null(oldval = assoc(key,alist,fun,TRUE)))
#endif
    rplacd(oldval,value);
  else {
    LVAL new, data, temp;
    double rhthresh, rhsize;
    int size, newsize;

    temp = hashtablerhthresh(table);
    if (floatp(temp) && getflonum(temp) > 0.0 && getflonum(temp) < 1.0)
      rhthresh = getflonum(temp);
    else
      rhthresh = HTAB_REHASH_THRESHOLD;
  
    temp = hashtablerhsize(table);
    if (floatp(temp) && getflonum(temp) > 1.0)
      rhsize = getflonum(temp);
    else
      rhsize = HTAB_REHASH_SIZE;

    alist = cons(cons(key,value),alist);
    sethashtablelist(table,idx,alist);
    sethashtablecount(table,hashtablecount(table)+1);
    if (hashtablecount(table) > rhthresh * hashtablesize(table)) {
      size = hashtablesize(table);
      newsize = (int) (rhsize * (size + 1));
      if (newsize % 2 == 0) newsize++;
      if (newsize < 0) xlfail("bad rehash size");
      if (size < newsize && newsize < MAXHTABSIZE) {
	xlsave1(new);
	new = newvector(newsize);
	data = hashtabledata(table);
	rehash_hashdata(data, new, fun);
	sethashtabledata(table, new);
	xlpop();
      }
    }
  }
}

/* function clrhash  TAA */

LVAL xclrhash()
{
  LVAL table;
  int i;

  table = xlgahashtable();
  xllastarg();

  for (i = hashtablesize(table)-1; i >= 0; i--)
    sethashtablelist(table,i,NIL);
  sethashtablecount(table,0);

  return (table);

}

/* function hash-table-count  TAA */

LVAL xhashcount()
{
  LVAL table;

  table = xlgahashtable();
  xllastarg();

  return (cvfixnum((FIXTYPE) hashtablecount(table)));
}

/* function maphash  TAA */

LVAL xmaphash()
{
  FRAMEP newfp;
  LVAL fun, table, arg, element;
  int i;

  fun = xlgetarg();
  table = xlgahashtable();
  xllastarg();

  xlstkcheck(3);
  xlprotect(fun);
  xlprotect(table);
  xlsave(element);

  for (i = hashtablesize(table)-1; i >= 0; i--)
    for (element=hashtablelist(table,i); consp(element);) {
      arg = car(element);
      element = cdr(element); /* in case element is deleted */
      newfp =xlsp;
      pusharg(cvfixnum((FIXTYPE)(newfp - xlfp)));
      pusharg(fun);
      pusharg(cvfixnum((FIXTYPE) 2));
      pusharg(car(arg));
      pusharg(cdr(arg));
      xlfp = newfp;
      xlapply(2);
    }

  xlpopn(3);

  return (NIL);
}

LVAL xhashtablep()
{
  LVAL x;

  x = xlgetarg();
  xllastarg();
  return(hashtablep(x) ? s_true : NIL);
}

LVAL xhashtablefun()
{
  LVAL x, fun;

  x = xlgahashtable();
  xllastarg();
  
  fun = hashtablefun(x);
  if (fun == getfunction(s_eq))
    return(s_eq);
  else if (fun == getfunction(s_eql))
    return(s_eql);
  else if (fun == getfunction(s_equal))
    return(s_equal);
  else
    return(fun);
}

LVAL xhashtablesize()
{
  LVAL x;

  x = xlgahashtable();
  xllastarg();
  return(cvfixnum((FIXTYPE) hashtablesize(x)));
}

LVAL xhashtablerhthresh()
{
  LVAL x;

  x = xlgahashtable();
  xllastarg();
  return(hashtablerhthresh(x));
}

LVAL xhashtablerhsize()
{
  LVAL x;

  x = xlgahashtable();
  xllastarg();
  return(hashtablerhsize(x));
}

#endif

/* Internal version of MAKE-LIST */
LVAL mklist(n, elem)
     int n;
     LVAL elem;
{
  LVAL result;
  
  xlsave1(result);
  for (result = NIL; n > 0; n--)
    result = cons(elem, result);
  xlpop();
  return(result);
}

/* Common Lisp MAKE-LIST function */
LVAL xmklist()
{
  int n;
  LVAL elem = NIL;
  
  n = getfixnum(xlgafixnum());
  xlgetkeyarg(k_initelem, &elem);
  xllastkey();
  
  return(mklist(n, elem));
}
#ifdef DODO
int geteqhash(key, table, pval)
     LVAL key, table, *pval;
{
  LVAL alist, pair;
  int hindex;

  if (hashtablerehash(table))
    rehash_hashtable(table);

  hindex = ((int) CVPTR(key)) % ((int) hashtablesize(table));
  alist = hashtablelist(table, hindex);

  for (; consp(alist); alist = cdr(alist)) {
    if (consp(pair = car(alist)) && car(pair) == key) {
      *pval = cdr(pair);
      return(TRUE);
    }
  }
  return(FALSE);
}
#endif /* DODO */
