
> (in-package '*lisp)
#<Package "*SIM" 9304B6>
> (*cold-boot :initial-dimensions '(8 4))


Thinking Machines Starlisp Simulator.  Version 16.0

1
(8 4)

;; ** Create a pvar filled with random values
;; ** to manipulate.

> (*defvar x (random!! (!! 10)))
X

;;;; ** NEWS!!, NEWS-BORDER!! and *NEWS


> (ppp x :mode :grid)

     DIMENSION 0 (X)  ----->

4 1 5 1 4 3 6 5 
0 0 6 6 6 8 6 8 
4 5 7 3 1 3 6 6 
3 7 0 9 9 9 9 5 

;; ** Every processor gets a value from the
;; ** processor 1 to its right (east).  NEWS!! returns
;; ** a new pvar, so we can just print it out.
;; ** NEWS!! always wraps around the edge of the grid.

> (ppp (news!! x 1 0) :mode :grid)

     DIMENSION 0 (X)  ----->

1 5 1 4 3 6 5 4 
0 6 6 6 8 6 8 0 
5 7 3 1 3 6 6 4 
7 0 9 9 9 9 5 3 

;; ** Get a value from a processor one below (south).

> (ppp (news!! x 0 1) :mode :grid)

     DIMENSION 0 (X)  ----->

0 0 6 6 6 8 6 8 
4 5 7 3 1 3 6 6 
3 7 0 9 9 9 9 5 
4 1 5 1 4 3 6 5 



;; ** News!! can retrieve values more than 1 unit away
;; ** on the NEWS grid.

> (ppp (news!! x 2 2) :mode :grid)

     DIMENSION 0 (X)  ----->

7 3 1 3 6 6 4 5 
0 9 9 9 9 5 3 7 
5 1 4 3 6 5 4 1 
6 6 6 8 6 8 0 0 

;; ** Negative numbers mean the other way (west and north)

> (ppp (news!! x -1 -2) :mode :grid)

     DIMENSION 0 (X)  ----->

6 4 5 7 3 1 3 6 
5 3 7 0 9 9 9 9 
5 4 1 5 1 4 3 6 
8 0 0 6 6 6 8 6 
NIL

;; ** Use the NEWS-BORDER!! function to prevent automatic
;; ** wrapping.  The second argument is a pvar that is
;; ** used instead of wrapping when a processor tries
;; ** to access off the grid.  So this expression puts
;; ** -17 in those processors which would otherwise have
;; ** wrapped around to get values.

> (ppp (news-border!! x (!! -17) 1 1) :mode :grid :format "~3D ")

     DIMENSION 0 (X)  ----->

  0   6   6   6   8   6   8 -17 
  5   7   3   1   3   6   6 -17 
  7   0   9   9   9   9   5 -17 
-17 -17 -17 -17 -17 -17 -17 -17 
NIL
> 



;; ** NEWS!! can be used to retrieve values from processors which
;; ** are not selected.  *NEWS can be used to put values in
;; ** processors which are not selected.

;; ** Any time a pvar is created in a context where not all the processors
;; ** are active, the simulator puts an ILLEGAL-VALUE into the processors
;; ** which are not selected.  PPP prints this ILLEGAL-VALUE out when it
;; ** attempts to print out pvars which were not defined with all
;; ** processors active.

> (ppp x :mode :grid)

     DIMENSION 0 (X)  ----->

4 1 5 1 4 3 6 5 
0 0 6 6 6 8 6 8 
4 5 7 3 1 3 6 6 
3 7 0 9 9 9 9 5 

> (ppp (*when (evenp!! (self-address-grid!! (!! 0))) (news!! x 1 0)) :mode :grid)

     DIMENSION 0 (X)  ----->

1 #:ILLEGAL-VALUE 1 #:ILLEGAL-VALUE 3 #:ILLEGAL-VALUE 5 #:ILLEGAL-VALUE 
0 #:ILLEGAL-VALUE 6 #:ILLEGAL-VALUE 8 #:ILLEGAL-VALUE 8 #:ILLEGAL-VALUE 
5 #:ILLEGAL-VALUE 3 #:ILLEGAL-VALUE 3 #:ILLEGAL-VALUE 6 #:ILLEGAL-VALUE 
7 #:ILLEGAL-VALUE 9 #:ILLEGAL-VALUE 9 #:ILLEGAL-VALUE 5 #:ILLEGAL-VALUE 

> (*defvar dest)
DEST

;; ** *NEWS takes data and sends is over the NEWS grid, as
;; ** opposed to NEWS!! which retrieves data.  NEWS!! is
;; ** in fact the faster of the two operations.

> (*news x dest 1 0)

> (ppp dest :mode :grid)

     DIMENSION 0 (X)  ----->

5 4 1 5 1 4 3 6 
8 0 0 6 6 6 8 6 
6 4 5 7 3 1 3 6 
5 3 7 0 9 9 9 9 
NIL
> (*set dest (!! -1))
NIL
> (*when (evenp!! (self-address-grid!! (!! 0))) (*news x dest 1 0))
NIL
> (ppp dest :mode :grid)

     DIMENSION 0 (X)  ----->

-1 4 -1 5 -1 4 -1 6 
-1 0 -1 6 -1 6 -1 6 
-1 4 -1 7 -1 1 -1 6 
-1 3 -1 0 -1 9 -1 9 
NIL


;;;; ** Function using NEWS to implement cellular automata.
;;;; ** This is one of the demos that runs on the CM Framebuffer.

;;;; This runs the cellular automaton described
;;;; in the August 1989 issue of Scientific American.
;;;; It assumes an 8 bit color display.

(defun automaton

    (&key
     (grid-width 1024)
     (grid-height 1024)
     (number-of-states 20)
     (number-of-iterations 10000)
     (display-every 20)
     )

  (when (null *current-display-window*)
    (create-display-window)
    )
  (when (not (eql 8 (display-window-bits-per-pixel)))
    (error "the AUTOMATON program assumes an 8 bit display")
    )

  (let-vp-set (grid-vp-set (create-vp-set (list grid-width grid-height)))
    (*with-vp-set grid-vp-set
      (let ((number-of-bits-for-state-variable (+ 3 (integer-length number-of-states)))
	    (last-state (1- number-of-states))
	    )
	(*locally
	 (declare (type fixnum grid-width grid-height number-of-states last-state))
	 (*let ((state (random!! (!! number-of-states)))
		east-state west-state north-state south-state next-state
		)
	   (declare (type (field-pvar number-of-bits-for-state-variable)
			  state east-state west-state north-state south-state next-state
			  ))
	   (dotimes (j number-of-iterations)
	     (*set next-state (if!! (=!! state (!! last-state)) (!! 0) (1+!! state)))
	     (*set east-state (news!! state 1 0))
	     (*set west-state (news!! state -1 0))
	     (*set north-state (news!! state 0 -1))
	     (*set south-state (news!! state 0 1))
	     (*set state 
		   (if!! (or!! (eql!! east-state next-state)
			       (eql!! west-state next-state)
			       (eql!! north-state next-state)
			       (eql!! south-state next-state)
			       )
			 next-state
			 state
			 ))
	     (when (zerop (mod j display-every))
	       (*g:write-display-window (+!! (!! 10) (*!! state (!! 10))))
	       )
	     )))))))


;;;; ** SCAN!!, SPREAD!! and REDUCE-AND-SPREAD!!

> (*defvar y (random!! (!! 5)))
Y
> (ppp y)
0 4 3 2 1 1 3 2 2 1 2 0 1 4 4 3 1 0 0 3 2 2 3 0 2 0 3 0 0 4 2 2 

;; ** Simple add scan.

> (ppp (scan!! y '+!!) :per-line 20)
0 4 7 9 10 11 14 16 18 19 21 21 22 26 30 33 34 34 34 37
39 41 44 44 46 46 49 49 49 53 55 57 

;; ** You can scan backwards as well as forwards.

> (ppp (scan!! y '+!! :direction :backward) :per-line 20)
57 57 53 50 48 47 46 43 41 39 38 36 36 35 31 27 24 23 23 23
20 18 16 13 13 11 11 8 8 8 4 2 

;; ** Scanning in segments.  The pvar is divided into parts and the scan
;; ** is done separately within each contiguous part.  Here then pvar
;; ** begins a new part whenever the pvar is zero.

> (ppp (scan!! y '+!! :segment-pvar (zerop!! y)))
0 4 7 9 10 11 14 16 18 19 21 0 1 5 9 12 13 0 0 3 5 7 10 0 2 0 3 0 0 4 6 8 

> (ppp (scan!! y '+!! :segment-pvar (zerop!! y) :direction :backward))
0 21 17 14 12 11 10 7 5 3 2 0 13 12 8 4 1 0 0 10 7 5 3 0 2 0 3 0 0 8 4 2 

;; ** Scanning can be done with other operators, like logical AND.  Here we
;; ** do a segmented AND scan where the segments are every five processors.

> (ppp (<!! y (!! 3)))
T NIL NIL T T T NIL T T T T T T NIL NIL NIL T T T NIL T T NIL T T T NIL T T NIL T T 

> (ppp (scan!! (<!! y (!! 3)) 'and!! 
       :segment-pvar (zerop!! (mod!! (self-address!!) (!! 5)))) :per-line 20)
T NIL NIL NIL NIL T NIL NIL NIL NIL T T T NIL NIL NIL NIL NIL NIL NIL
T T NIL NIL NIL T NIL NIL NIL NIL T T 

;; ** Here is a segmented MIN scan where the segments are every 4 processors.

> (ppp (scan!! y 'min!! :segment-pvar (zerop!! (mod!! (self-address!!) (!! 4)))))
0 0 0 0 1 1 1 1 2 1 1 0 1 1 1 1 1 0 0 0 2 2 2 0 2 0 0 0 0 0 0 0 

;; ** A very useful type of segmented scanning is a COPY SCAN.
;; ** Here we copy scan out every third processor's value to
;; ** the other two processors.

> (ppp y)
0 4 3 2 1 1 3 2 2 1 2 0 1 4 4 3 1 0 0 3 2 2 3 0 2 0 3 0 0 4 2 2 

> (ppp (scan!! y 'copy!! :segment-pvar (zerop!! (mod!! (self-address!!) (!! 3)))))
0 0 0 2 2 2 3 3 3 1 1 1 1 1 1 3 3 3 0 0 0 2 2 2 2 2 2 0 0 0 2 2 



;; ** You can spread a row or column of values out using SPREAD!!.
;; ** (You can also use SCAN!! along a particular dimension with
;; ** the :dimension argument to SCAN!!).

> (ppp y :mode :grid)

     DIMENSION 0 (X)  ----->

0 1 2 1 1 2 2 0 
4 1 1 4 0 2 0 4 
3 3 2 4 0 3 3 2 
2 2 0 3 3 0 0 2 

;; ** Here we spread column number 3 along dimension 0 (X)

> (ppp (spread!! y 0 3) :mode :grid)

     DIMENSION 0 (X)  ----->

1 1 1 1 1 1 1 1 
4 4 4 4 4 4 4 4 
4 4 4 4 4 4 4 4 
3 3 3 3 3 3 3 3 

;; ** And here we spread column number 2 along dimension 1 (Y)

> (ppp (spread!! y 1 2) :mode :grid)

     DIMENSION 0 (X)  ----->

3 3 2 4 0 3 3 2 
3 3 2 4 0 3 3 2 
3 3 2 4 0 3 3 2 
3 3 2 4 0 3 3 2 

;; ** You can use REDUCE-AND-SPREAD!! to effectively spread the
;; ** final value of a SCAN!! operation back along an entire
;; ** row or column.

;; ** Here we find the max of each row and spread it out
;; ** to every processor in the row.

> (ppp (reduce-and-spread!! y 'max!! 0) :mode :grid)

     DIMENSION 0 (X)  ----->

2 2 2 2 2 2 2 2 
4 4 4 4 4 4 4 4 
4 4 4 4 4 4 4 4 
3 3 3 3 3 3 3 3 



;; ** Program illustrating the use of a segmented copy scan
;; ** to do an addition operation with one bit per processor.


;;; -*- SYNTAX: COMMON-LISP; MODE: LISP; BASE: 10; PACKAGE: *LISP; -*-

(in-package '*lisp)
						
;;;; Author:  JP Massar.


;;;; The object of this excercise is to add two binary numbers,
;;;; represented 1 bit per processor in cube address order, with
;;;; the least signficant bit of each number being in processor 0.

;;;; The algorithm used is to figure out which processors will
;;;; receive a carry bit, using an exclusive scan!!, and then
;;;; figure out the result in that processor as a function of
;;;; the values in that processor of the two binary numbers
;;;; and the carry bit.

;;;; This is a simple example program which does not use much in
;;;; the way of new 5.0 *Lisp features, except for *locally,
;;;; pppdbg and return-pvar-p.

;;;; This text resides in
;;;; /cm/starlisp/interpreter/f5101/very-long-addition-example.lisp
;;;; Ask your systems manager or applications engineer for its
;;;; exact location at your installation.


(defmacro *nocompile (&body body)
  `(compiler-let ((*compilep* nil)) ,@body)
  )


(defun very-long-add!!

       (bit-pvar1 bit-pvar2 length1 length2 &optional (check-overflow t))

  (let ((last-processor (1- *number-of-processors-limit*)))

    (*locally
      (declare (type fixnum length1 length2 last-processor))

      (when check-overflow
	(assert
	  (or (and (< length1 *number-of-processors-limit*)
		   (< length2 *number-of-processors-limit*)
		   )
	      (not (and
		     (plusp (pref bit-pvar1 last-processor))
		     (plusp (pref bit-pvar2 last-processor))
		     )))
	  ()
	  "You are trying to add two numbers who result would be ~@
       a number longer than ~D binary digits, the total size of the ~@
       current vp set."
	  *number-of-processors-limit*
	  ))


      (*all
	(declare (return-pvar-p t))

	;; The input arguments, bit-pvar1 and bit-pvar2
	;; may not be pvars of type (unsigned-byte 1)
	;; (they may be general pvars, for instance).
	;; Allocate two temporarys of the proper type
	;; and coerce the input arguments into them.
	;; The *SET's will error out if the input arguments
	;; do not everywhere contain 1 or 0.

	(*let ((source1 (!! 0))
	       (source2 (!! 0))
	       dest
	       )
	  (declare (type (pvar (unsigned-byte 1)) source1 source2 dest))
	  (declare (return-pvar-p t))

	  (*when (<!! (self-address!!) (!! length1))
	    (declare (return-pvar-p nil))
	    (*nocompile (*set source1 bit-pvar1))
	    )

	  (*when (<!! (self-address!!) (!! length2))
	    (declare (return-pvar-p nil))
	    (*nocompile (*set source2 bit-pvar2))
	    )

	  ;; Any processor which contains a 1 in both input arguments
	  ;; is going to propagate a carry in all subsequent processors
	  ;; until a processor which contains a 0 in both arguments
	  ;; is encountered.

	  ;; Any processor which contains a 0 in both input arguments
	  ;; is going to cause a carry to not be propagated beyond it,
	  ;; and will prevent any carry from being generated until
	  ;; a subsequent processor containing 1 in both arguments
	  ;; is encountered.

	  ;; Thus processors with either two 1's or two 0's define
	  ;; 'segments' of carry propagation or non-propagation.

	  (*let* ((zero-zero (and!! (zerop!! source1) (zerop!! source2)))
		  (one-one (and!! (not!! (zerop!! source1))
				  (not!! (zerop!! source2))
				  ))
		  (zero-one (not!! (or!! zero-zero one-one)))
		  (carry-segment
		    (or!! (zerop!! (self-address!!)) zero-zero one-one))
		  will-receive-carry
		  )
	    (declare (type boolean-pvar zero-zero one-one zero-one)
		     (type boolean-pvar carry-segment will-receive-carry)
		     )
	    (declare (return-pvar-p nil))

	    (*set will-receive-carry
		  (scan!! one-one 'copy!!
			  :segment-pvar carry-segment :include-self nil
			  ))
	    (*setf (pref will-receive-carry 0) nil)

	    (when check-overflow
	      (if (and (pref will-receive-carry last-processor)
		       (not (pref zero-zero last-processor))
		       )
		  (error
		    "You are trying to add two numbers who result ~@
                     would be a number longer than ~D binary digits, ~@
                     the total size of the current vp set."
		    *number-of-processors-limit*
		    )))

	    ;; This implements the three-input binary addition
	    ;; algorithm.

	    (*set dest
		  (cond!!
		    ((and!! (or!! one-one zero-zero) will-receive-carry)
		     (!! 1))
		    ((or!! one-one zero-zero) (!! 0))
		    ((and!! zero-one will-receive-carry) (!! 0))
		    (zero-one (!! 1))
		    )))

	  dest

	  )))))



(defun test-very-long-add ()
  (*warm-boot)
  (let ((length1 (random 20))
	(length2 (random 20))
	)
    (*let (source1 source2)
      (declare (type (field-pvar 1) source1 source2))
      (dotimes (j length1)
	(*setf (pref source1 j) (random 2))
	)
      (dotimes (j length2)
	(*setf (pref source2 j) (random 2))
	)
      (pppdbg source1 :end length1 :format "~1D ")
      (pppdbg source2 :end length2 :format "~1D ")
      (ppp (very-long-add!! source1 source2 length1 length2 t) :end 22 :format "~1D " :title "RESULT ")
      )))

;; SAMPLE OUTPUT

  (test-very-long-add)
SOURCE1: 0 1 1 1 0 0 1 1 1 0 
SOURCE2: 1 0 1 1 1 0 0 1 1 0 0 0 0 1 0 0 0 
RESULT : 1 1 0 1 0 1 1 0 1 1 0 0 0 1 0 0 0 0 0 0 0 0 
NIL
  (test-very-long-add)
SOURCE1: 1 0 1 0 1 0 0 0 1 0 1 1 1 1 1 1 
SOURCE2: 0 1 1 1 1 0 1 1 0 0 0 0 0 1 0 1 0 1 
RESULT : 1 1 0 0 1 1 1 1 1 0 1 1 1 0 0 1 1 1 0 0 0 0 
NIL



;;;; ** *PSET and PREF!!

> (ppp x)
4 0 4 3 1 0 5 7 5 6 7 0 1 6 3 9 4 6 1 9 3 8 3 9 6 6 6 9 5 8 6 5 

> (*defvar dest)
DEST

;; ** 1-1 Send.
;; ** Swap values in each of two consecutive processors.
;; ** Here we do it by using a SEND (*PSET) into a destination pvar.

> (*pset :no-collisions x dest
         (if!! (evenp!! (self-address!!)) 
               (1+!! (self-address!!))
               (1-!! (self-address!!))))

> (ppp dest)
0 4 3 4 0 1 7 5 6 5 0 7 6 1 9 3 6 4 9 1 8 3 9 3 6 6 9 6 8 5 5 6 

;; ** 1-1 get.
;; ** And here we do it by using a GET (PREF!!) which returns a pvar.

> (ppp (pref!! x (if!! (evenp!! (self-address!!)) 
                       (1+!! (self-address!!))
                       (1-!! (self-address!!)))
               :collision-mode :no-collisions
               ))
0 4 3 4 0 1 7 5 6 5 0 7 6 1 9 3 6 4 9 1 8 3 9 3 6 6 9 6 8 5 5 6 

;; ** Many to one send.  Here each processor sends a value to be added
;; ** to one of the first 16 processors.  Note that the value of
;; ** DEST has not changed except in the first 16 processors.

> (ppp x)
4 0 4 3 1 0 5 7 5 6 7 0 1 6 3 9 4 6 1 9 3 8 3 9 6 6 6 9 5 8 6 5 

> (*pset :add x dest (mod!! (self-address!!) (!! 16)))

> (ppp dest)
8 6 5 12 4 8 8 16 11 12 13 9 6 14 9 14 6 4 9 1 8 3 9 3 6 6 9 6 8 5 5 6 

;; ** Many to 1 get.  Here, each processor gets a value from one of the first
;; ** 4 processors.  The first call invokes the backwards routing hardware
;; ** capability of the CM.  The other two calls do the same thing but use
;; ** different algorithms and take different amounts of time.
;; ** The :collisions-allowed algorithm takes time proportional
;; ** to the maximum number of collisions.  The :many-collisions
;; ** algorithm is independent of the number of collisions.

> (ppp (pref!! x (mod!! (self-address!!) (!! 4))))
4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 

> (ppp (pref!! x (mod!! (self-address!!) (!! 4)) :collision-mode :many-collisions))
4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 

> (ppp (pref!! x (mod!! (self-address!!) (!! 4)) :collision-mode :collisions-allowed))
4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 4 0 4 3 


;; ** Simple function which packs values into the first N processors.
;; ** Source values in processors selected by the boolean active-processors
;; ** pvar are sent to a contiguous set of processors at the 'beginning'
;; ** of the machine.

> (defun pack-active-processors (destination source active-processors)
    (*all 
      (*set destination (!! 0.0))
      (*when active-processors (*pset :no-collisions source destination (enumerate!!)))
      ))

PACK-ACTIVE-PROCESSORS

> (*set dest (!! 0.0))

> (pack-active-processors dest x (evenp!! (self-address!!)))

> (ppp dest :per-line 20)
4 4 1 5 5 7 1 3 4 1 3 3 6 6 5 6 0.0 0.0 0.0 0.0
0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 0.0 



;;;; ** VP SETS and COMUNICATION BETWEEN THEM.

;; ** Define a one dimensional vp set and a pvar belonging to it.

> (def-vp-set linear-vp-set '(8))
LINEAR-VP-SET

> (*defvar pvar-in-linear-vp-set (random!! (!! 10)) nil linear-vp-set)
PVAR-IN-LINEAR-VP-SET

;; ** This function takes a pvar in a one dimensional vp set
;; ** and creates a two dimensional vp set big enough to hold
;; ** the cross product of all the processor values in the
;; ** one dimensional vp set.  The cross product of these
;; ** values is formed using inter-vp-set communication and
;; ** spreads, and then the average of the value pairs is
;; ** printed out.

> (defun form-cross-product-and-average (pvar)
    (let ((size (first (vp-set-dimensions (pvar-vp-set pvar)))))
      (let-vp-set (cross-product-vp-set (create-vp-set (list size size)))
        (*with-vp-set cross-product-vp-set
	  (*let (temp1 temp2 average)
	    (*with-vp-set (pvar-vp-set pvar)
	      (*pset :no-collisions pvar temp1 
		     (cube-from-vp-grid-address!!
		       cross-product-vp-set (!! 0) (self-address!!)))
	      (*pset :no-collisions pvar temp2
		     (cube-from-vp-grid-address!! 
                       cross-product-vp-set (self-address!!) (!! 0)))
	      (*with-vp-set cross-product-vp-set
	        (*set temp1 (spread!! temp1 0 0))
	        (*set temp2 (spread!! temp2 1 0))
	        (*set average (/!! (+!! temp1 temp2) (!! 2.0)))
	        (ppp average :mode :grid)
	        )))))))
FORM-CROSS-PRODUCT-AND-AVERAGE
> (form-cross-product-and-average pvar-in-linear-vp-set)

     DIMENSION 0 (X)  ----->

6.0 4.0 5.5 7.5 3.5 4.0 4.0 3.0 
4.0 2.0 3.5 5.5 1.5 2.0 2.0 1.0 
5.5 3.5 5.0 7.0 3.0 3.5 3.5 2.5 
7.5 5.5 7.0 9.0 5.0 5.5 5.5 4.5 
3.5 1.5 3.0 5.0 1.0 1.5 1.5 0.5 
4.0 2.0 3.5 5.5 1.5 2.0 2.0 1.0 
4.0 2.0 3.5 5.5 1.5 2.0 2.0 1.0 
3.0 1.0 2.5 4.5 0.5 1.0 1.0 0.0 

> (ppp pvar-in-linear-vp-set)
6 2 5 9 1 2 2 0 



;;;; **  SORT!! and RANK!!

> (ppp x)
4 0 4 3 1 0 5 7 5 6 7 0 1 6 3 9 4 6 1 9 3 8 3 9 6 6 6 9 5 8 6 5 

> (ppp (sort!! x '<=!!))
0 0 0 1 1 1 3 3 3 3 4 4 4 5 5 5 5 6 6 6 6 6 6 6 7 7 8 8 9 9 9 9 

;; ** Sorting is possible along a single dimension.
;; ** (Also (not shown) within individual segments)

> (ppp x :mode :grid)

     DIMENSION 0 (X)  ----->

4 1 5 1 4 3 6 5 
0 0 6 6 6 8 6 8 
4 5 7 3 1 3 6 6 
3 7 0 9 9 9 9 5 

> (ppp (sort!! x '<=!! :dimension 0) :mode :grid)

     DIMENSION 0 (X)  ----->

1 1 3 4 4 5 5 6 
0 0 6 6 6 6 8 8 
1 3 3 4 5 6 6 7 
0 3 5 7 9 9 9 9 

> (ppp x)
4 0 4 3 1 0 5 7 5 6 7 0 1 6 3 9 4 6 1 9 3 8 3 9 6 6 6 9 5 8 6 5 

;; ** The rank!! function returns the index of the value in the
;; ** sorted order.  SORT!! can thus be implemented using a RANK!!
;; ** and a *PSET.

> (ppp (rank!! x '<=!!))
12 2 11 9 5 1 16 25 15 23 24 0 4 22 8 31 10 21 3 30 7 27 6 29 20 19 18 28 14 26 17 13 

> (*pset :no-collisions x dest (rank!! x '<=!!))

> (ppp dest)
0 0 0 1 1 1 3 3 3 3 4 4 4 5 5 5 5 6 6 6 6 6 6 6 7 7 8 8 9 9 9 9 


> (*cold-boot)

;; Declare and allocate an array pvar 'table' of signed integers, 10 elements long.

> (*proclaim '(type (array-pvar (signed-byte 32) (10)) table))

> (*defvar table)
TABLE

;; ** Put some values into the elements of the array.
;; ** and then print out a few of the arrays to see what we have.

> (dotimes (j 10) (*setf (aref!! table (!! j)) (+!! (self-address!!) (!! j))))
NIL
> (pref table 0)
#(0 1 2 3 4 5 6 7 8 9)
> (pref table 1)
#(1 2 3 4 5 6 7 8 9 10)
> (pref table 23)
#(23 24 25 26 27 28 29 30 31 32)

;; Now turn the array sideways so we can access it using indirect
;; addressing.  You can only turn arrays sideways which have
;; elements which are 2,4,8,16,32 or a multiple of 32 bits long.
;; 32 bit elements are most efficient.  It is not very efficient
;; to use 2 or 4 bit elements.

> (*sideways-array table)
T

;; Retrieve a value from a different index in each processor and print it out.

> (ppp (sideways-aref!! table (mod!! (self-address!!) (!! 10))) :end 20)
0 2 4 6 8 10 12 14 16 18 10 12 14 16 18 20 22 24 26 28 

;; You can, of course, retrieve a value from the same index.  

> (ppp (sideways-aref!! table (!! 0)) :end 20)
0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 

;; Store a value into a different index in each processor and read it
;; out again.

> (*setf (sideways-aref!! table (mod!! (self-address!!) (!! 10))) (!! 17))
NIL
> (ppp (sideways-aref!! table (mod!! (self-address!!) (!! 10))) :end 20)
17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 17 
> (ppp (sideways-aref!! table (!! 0)) :end 20)
17 1 2 3 4 5 6 7 8 9 17 11 12 13 14 15 16 17 18 19 


;; ** Use the *Lisp library CREATE-LOOKUP-TABLE routine to
;; ** create a shared table.  These tables are only replicated
;; ** once per every 32 physical processors.  The array created
;; ** above was replicated once per every virtual processor.

> (setq shared-table
	(sll:create-lookup-table
	 '#(1.1 1.11 2.1 2.11 3.1 3.11 17.23 9.2) '(pvar single-float)))
#S(*LISP-I::LOOKUP-TABLE DIMENSIONS (8) NUMBER-OF-ELEMENTS-STORED 8 ...)
> (ppp (sll:shared-table-aref!!  shared-table (mod!! (self-address!!) (!! 8))) :end 10)
1.1 1.11 2.1 2.11 3.1 3.11 17.23 9.2 1.1 1.11 
> (sll:free-lookup-table shared-table)
