;;; -*- Mode: LISP; Syntax: COMMON-LISP; Package: (*SIM-I COMMON-LISP-GLOBAL); Base: 10; Muser: yes -*-

(in-package '*sim-i :use '(lisp))

;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+
;;;> 
;;;> The Thinking Machines *Lisp Simulator is in the public domain.
;;;> You are free to do whatever you like with it, including but
;;;> not limited to distributing, modifying, and copying.

;;;> Bugs, comments and revisions due to porting can be sent to:
;;;> bug-starlisp@think.com.  Other than to Thinking Machines'
;;;> customers, no promise of support is intended or implied.
;;;>
;;;> *+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+*+

;;; Author:  JP Massar.


;;;
;;; This file contains most of the macros necessary for the *Lisp simulator
;;; Other macros which are implementation dependent can be found in the
;;; file port.lisp

;;;
;;; Common Lisp doesn't have comment!
;;;

(defmacro comment (&rest foo) (declare (ignore foo)) ''comment)

(defmacro pvar-check (x) `(assert (pvar-p ,x)))

(defmacro check-pvar-arglist (arglist) `(dolist (j ,arglist) (pvar-check j)))


(defmacro safety-check (&body body)
  `(when (plusp *interpreter-safety*) ,@body)
  )


;;;; ****************************************************************************
;;;;
;;;;                           CODE FOR *DEFUN
;;;;
;;;; ****************************************************************************


(defun split-declarations-and-body (forms env)

  "Returns two values, a list of declarations and a list of forms"

  (cond

    ;; If there aren't any more forms, we are done.

    ((null forms) (values nil nil))

    ;; If the form is not a non-nil list, then there
    ;; are no more declarations.

    ((null (consp (car forms))) (values nil forms))

    (t
     (let ((form (car forms)))

       (cond

	 ;; If the form is not a DECLARE, and it cannot possibly
	 ;; expand into a DECLARE, then there are no more declarations.

	 ;; Be careful about special forms because special forms
	 ;; can have macro expansions, but they cannot expand
	 ;; into declares.  

	 ((and (not (eq (car form) 'declare))
	       (or (listp (car form))
		   (really-is-a-special-form (car form))
		   (null (macro-function (car form)))
		   ))
	  (values nil forms)
	  )

	 ;; If the form is a declare, grab it and recurse on
	 ;; the rest of the forms.

	 ((eq (car form) 'declare)
	  (multiple-value-bind (other-declares other-forms)
	      (split-declarations-and-body (cdr forms) env)
	    (values (cons form other-declares) other-forms)
	    ))

	 ;; If the form is a macro, macroexpand it once and recurse.
	 ;; We macroexpand only once because things might macroexpand
	 ;; into special forms or certain disgusting Common Lisps
	 ;; may macroexpand DECLARE statements into oblivion!

	 ((macro-function (car form))
	  (let ((macroexpanded-form (macroexpand-1 form env)))
	    (multiple-value-bind (declares other-forms)
		(split-declarations-and-body (cons macroexpanded-form (cdr forms)) env)
	      (values declares other-forms)
	      )))

	 (t (error "Internal error.  You can't get here from Common Lisp"))

	 )))))



(defun split-documentation-and-body (body)
  (if (eql 1 (length body))
      (values nil body)
      (if (stringp (car body))
	  (multiple-value-bind (doc-string-list real-body)
	      (split-documentation-and-body (cdr body))
	    (values (cons (car body) doc-string-list) real-body)
	    )
	  (values nil body)
       )))


(defun parse-body-form-documentation-and-declarations (body env)
  (multiple-value-bind (documentation body)
      (split-documentation-and-body body)
    (multiple-value-bind (declarations body)
	(split-declarations-and-body body env)
      (values documentation declarations body)
      )))


(defun declaration-into-list-of-single-declarations (declaration)
  (mapcar
    #'(lambda (decl-spec) `(declare ,decl-spec))
    (cdr declaration)
    ))


(defun declarations-into-list-of-single-declarations (declarations)
  (mapcan 'declaration-into-list-of-single-declarations declarations)
  )


(defun *lisp-declaration-p (single-declaration)
  (if (return-pvar-p-declaration-p single-declaration)
      t
      (let ((decl-spec (cadr single-declaration)))
	(if (eq 'type (first decl-spec))
	    (if (canonical-pvar-type (second decl-spec)) t nil)
	    nil
	    ))))


(defun return-pvar-p-value-from-list-of-*lisp-declarations (list-of-single-declarations)
  (let ((number-of-return-pvar-p-declarations
	  (count-if
	    'return-pvar-p-declaration-p
	    list-of-single-declarations
	    )))
    (cond
      ;; Allegro 1.3.2 count-if on a nil list returns nil, rather 0, as CLtL states
      #+:ccl-1.3 ((null number-of-return-pvar-p-declarations) :maybe)
      ((eql 0 number-of-return-pvar-p-declarations) :maybe)
      ((eql 1 number-of-return-pvar-p-declarations)
       (let ((return-pvar-p-declaration
	       (find-if
		 'return-pvar-p-declaration-p
		 list-of-single-declarations
		 )))
	 (if (return-pvar-p-declaration-value return-pvar-p-declaration) :yes :no)
	 ))
      (t (error "More than 1 return-pvar-p declaration found"))
      )))

(defun separate-*lisp-declarations-from-other-declarations (list-of-single-declarations)
  (values
    (remove-if-not '*lisp-declaration-p list-of-single-declarations)
    (remove-if '*lisp-declaration-p list-of-single-declarations)
    ))


(defun create-*defun-wrapping-form (function-name arglist)
  (let ((old-temp-pvar-list-symbol (gensym "*DEFUN-TEMP-")))
    `(let ((,old-temp-pvar-list-symbol *temp-pvar-list*))
       (handle-returning-pvar (,function-name ,@arglist) ,old-temp-pvar-list-symbol nil)
       )))


(defmacro *defun
    #-KCL
    (fname arglist &rest body &environment env)
    #+KCL
    (&environment env fname arglist &rest body)
  #+symbolics (declare (zwei:indentation 2 1))
  (let ((internal-symbol (make-*defun-function fname)))
    (multiple-value-bind (documentation declarations body)
	(parse-body-form-documentation-and-declarations body env)
      (setq declarations (declarations-into-list-of-single-declarations declarations))
      (multiple-value-bind (*lisp-declarations other-declarations)
	  (separate-*lisp-declarations-from-other-declarations declarations)
	(let ((return-pvar-p-value (return-pvar-p-value-from-list-of-*lisp-declarations *lisp-declarations))
	      (return-value-symbol (gensym "RETURN-VALUE-"))
	      )
	  `(progn
	     (defmacro ,fname (&rest args)		
	       ,@documentation
	       ,@(hack-declarations-for-symbolics other-declarations arglist)
	       (create-*defun-wrapping-form ',internal-symbol args)
	       )
	     (defun ,internal-symbol ,arglist
	       ,@other-declarations
	       #+symbolics
	       (declare (system:function-parent ,fname defun))
	       (compiler-let ((*inside-a-*defun* t))
		 (let ((,return-value-symbol (block ,fname ,@body)))
		   (check-return-pvar-p ,return-value-symbol ,return-pvar-p-value)
		   ,return-value-symbol
		   )))
	     ',fname 
	     ))))))


(defmacro *locally
    #-KCL
    (&body body &environment env)
    #+KCL
    (&environment env &body body)
  (multiple-value-bind (declarations real-body)
      (split-declarations-and-body body env)
    declarations
    `(progn ,@real-body)
    ))


;;;; ****************************************************************************
;;;;
;;;;                              CODE FOR *SET
;;;;
;;;; ****************************************************************************


(defun even-and-odd-elements (sequence)
  (assert (evenp (length sequence)))
  (let ((even-elements nil) (odd-elements nil) (count 0))
    (map nil
	 #'(lambda (element)
	     (if (evenp count) (push element even-elements) (push element odd-elements))
	     (incf count)
	     )
	 sequence
	 )
    (values (nreverse even-elements) (nreverse odd-elements))
    ))


(defvar *allow-compiled-*set nil)


(defmacro *set
    #-KCL
    (&rest dest-source-pairs &environment env)
    #+KCL
    (&environment env &rest dest-source-pairs)
  (argument-list-declaration
    dest-pvar source-pvar-expression &rest more-dest-source-pvar-pairs
    )

  (when *allow-compiled-*set
    (return-from *set
      (let ((length (length dest-source-pairs)))
	(cond
	  ((zerop length) nil)
	  ((oddp length) (error "Odd number of arguments to *SET"))
	  (t
	   (multiple-value-bind (destinations sources)
	       (even-and-odd-elements dest-source-pairs)
	     `(progn
		,@(mapcar
		    #'(lambda (dest source)
			(let ((code #+*lisp-simulator nil
				    #-*lisp-simulator
				    (*set-can-be-compiled-p `(,dest ,source) env)))
			  (if code
			      code
			      (let ((*allow-compiled-*set nil))
				(macroexpand-1 `(*set ,dest ,source) env)
				))))
		    destinations
		    sources
		    ))))))))

  (let ((length (length dest-source-pairs)))
    (cond
      ((zerop length) nil)
      ((oddp length) (error "Odd number of arguments to *SET"))
      ((and (= 2 length) (symbolp (second dest-source-pairs)) (symbolp (first dest-source-pairs)))
       `(*copy-pvar ,(first dest-source-pairs) ,(second dest-source-pairs))
       )

      (t
       (multiple-value-bind (destinations sources)
	   (even-and-odd-elements dest-source-pairs)
	 (let ((old-temp-pvar-list-symbol (gensym "OLD-TEMP-PVAR-LIST-")))
	   `(let ((,old-temp-pvar-list-symbol *temp-pvar-list*))
	      ,@(mapcar #'(lambda (dest source) `(*copy-pvar ,dest ,source)) destinations sources)
	      (handle-returning-pvar nil ,old-temp-pvar-list-symbol nil)
	      ))))

      )))
		

;;;; ****************************************************************************
;;;;
;;;;                         CODE FOR *LET and *LET*
;;;;
;;;; ****************************************************************************


(defmacro *let
    #-KCL
    (bindings &body body &environment env)
    #+KCL
    (&environment env bindings &body body)
    (*let-1 t bindings body env)
    )

(defmacro *let*
    #-KCL
  (bindings &body body &environment env)
  #+KCL
  (&environment env bindings &body body)
  (*let-1 nil bindings body env)
  )


;;; We want a function which returns 8 values:
;;; 1) A list of variables being bound
;;; 2) A list of initialization forms for those variables
;;; 3) A list of types for each bound variable, obtained from the declarations
;;; 4) A list of any variables declared that are not bound
;;; 5) A list of the types declared for variables that are declared but not bound
;;; 6) A list of any other random decl-specs.
;;; 7) One of :YES :NO or :MAYBE as to (declare (return-pvar-p ...))
;;; 7) The body of the *LET or *LET*

;;; The input is two forms, the list of bindings and the list of forms
;;; that constitute the body.  


(eval-when (compile load eval)
  (defvar *no-binding-for-*let-value*)
  (setq *no-binding-for-*let-value* (gensym "NO-BINDING-"))
  )


(defun parse-*let-bindings-declarations-and-body (bindings body-forms env)
  
  ;; Each body form must be macroexpanded until one of them
  ;; produces a non DECLARE statement.  Further forms in the
  ;; body need not be macroexpanded.

  (multiple-value-bind (bound-variables-list initialization-forms-list)

      (parse-*let-bindings bindings)
    
    (let ((declare-list nil) (real-body-forms body-forms))

      ;; Each time we find a declare we put it on the declare list
      ;; and remove it from the real-body-forms list.

      (block macroexpander
	(dolist (form body-forms)
	  nil
	  (do ((macroexpanded-form form (macroexpand-1 macroexpanded-form env)))
	      ((or (symbolp macroexpanded-form) (not (listp macroexpanded-form))) (return-from macroexpander nil))
	    (cond
	      ((eq 'declare (car macroexpanded-form))
	       (push macroexpanded-form declare-list)
	       (pop real-body-forms)
	       (return)
	       )
	      ((or (listp (car macroexpanded-form))
		   (really-is-a-special-form (car macroexpanded-form))
		   (not (macro-function (car macroexpanded-form)))
		   )
	       (return-from macroexpander nil))
	      ))))

      (multiple-value-bind (variable-types random-variables random-variable-types return-pvar-p other-declarations)
	  
	  (parse-*let-declarations bound-variables-list declare-list)

	(values
	  bound-variables-list
	  initialization-forms-list
	  variable-types
	  random-variables
	  random-variable-types
	  other-declarations
	  return-pvar-p
	  real-body-forms
	  )))))


(defun parse-*let-bindings (bindings)

  (let ((variables-list nil) (initialization-forms-list nil))
    
    (mapc
      #'(lambda (binding)
	  (cond
	    ((symbolp binding)
	     (push binding variables-list)
	     (push *no-binding-for-*let-value* initialization-forms-list)
	     )
	    ((listp binding)
	     (assert (< (length binding) 3) ()
		     "The binding form ~S for a *LET or *LET* has more than two elements" binding)
	     (assert (eql 2 (length binding)) ()
		     "The binding form ~S for a *LET or *LET* has no initial value specified" binding)
	     (assert (symbolp (car binding)) ()
		     "The object ~S is not a symbol and hence is not legal as the car of the binding form ~S for *LET or *LET*"
		     (car binding) binding)
	     (assert (not (member (car binding) '(t!! nil!! t nil))) ()
		     "You cannot bind the symbol ~S !!" (car binding))
	     (push (first binding) variables-list)
	     (push (second binding) initialization-forms-list)
	     )
	    (t
	     (error "The object ~S is not a legal binding form for *LET or *LET*" binding)
	     )))
      bindings
      )

    (values (nreverse variables-list) (nreverse initialization-forms-list))

    ))


(defun parse-*let-declarations (bound-variables-list declare-list)

  (let ((all-decl-specs (apply #'append (mapcar #'cdr declare-list))))
    
    ;; Separate out the type declarations from other declarations.

    (let ((all-type-decl-specs
	    (remove-if
	      #'(lambda (decl-spec) (or (not (listp decl-spec)) (not (eq 'type (car decl-spec)))))
	      all-decl-specs
	      ))
	  (other-decl-specs
	    (remove-if-not
	      #'(lambda (decl-spec) (or (not (listp decl-spec)) (not (eq 'type (car decl-spec)))))
	      all-decl-specs
	      ))
	  )

      (let ((variable-type-pairs-list (parse-type-decl-specs all-type-decl-specs)))

	;; Check for duplicate variable declarations.

	(dolist (pair variable-type-pairs-list)
	  (let ((variable (first pair)))
	    (when (not (eql 1 (count variable variable-type-pairs-list :key #'car)))
	      (error "The variable ~S was declared twice in the same set of *LET or *LET* declaration forms" variable)
	      )))

	(let ((unbound-variables-list nil)
	      (unbound-variables-type-list nil)
	      (bound-variables-type-list nil)
	      )

	  ;; For each bound variable, if it has a type declaration,
	  ;; associate that type with the bound variable.

	  (setq bound-variables-type-list
		(mapcar
		  #'(lambda (bv)
		      (let ((variable-type-pair (assoc bv variable-type-pairs-list)))
			(if variable-type-pair (second variable-type-pair) '(pvar *))
			))
		  bound-variables-list
		  ))

	  ;; For each type / variable pair that was declared,
	  ;; if the variable is not one of the bound *LET or *LET*
	  ;; variables, then push the variable and its associated
	  ;; type onto their own lists.

	  (mapc
	    #'(lambda (variable-type-pair)
		(let ((variable (first variable-type-pair))
		      (type (second variable-type-pair))
		      )
		  (when (not (member variable bound-variables-list))
		    (push variable unbound-variables-list)
		    (push type unbound-variables-type-list)
		    )))
	    variable-type-pairs-list
	    )

	  ;; Look for return-pvar-p declarations.

	  (let ((return-pvar-p :maybe) (found-return-pvar-p nil))

	    (dolist (other-decl other-decl-specs)
	      (when (and (listp other-decl) (eq 'return-pvar-p (car other-decl)))
		(cond
		  ((not (eql 2 (length other-decl)))
		   (error "The return-pvar-p declaration ~S is not syntactically correct" other-decl)
		   )
		  ((not (member (second other-decl) '(t nil)))
		   (error "The return-pvar-p declaration ~S is not syntactically correct.  You must specify T or NIL" other-decl)
		   )
		  (found-return-pvar-p
		   (error "More than one return-pvar-p declaration exists.")
		   )
		  (t
		   (setq found-return-pvar-p t)
		   (if (eq t (second other-decl))
		       (setq return-pvar-p :yes)
		       (setq return-pvar-p :no)
		       )))))

	    (values bound-variables-type-list unbound-variables-list unbound-variables-type-list return-pvar-p other-decl-specs)

	  ))))))


(defun parse-type-decl-specs (type-decl-specs) 

  ;; A type-decl-spec has the form (type <type> &rest symbols)
  ;; This checks that syntax, then returns a list, each of
  ;; whose elements is a 2 element list, the first element being
  ;; a symbol, and the second element being the canonical pvar
  ;; type that that symbol is being declared as.

 (mapcan
    #'(lambda (decl-spec)
	(assert (not (eql 1 (length decl-spec))) () "The type declaration ~S has no type or variables!" decl-spec)
	(assert (not (eql 2 (length decl-spec))) ()
		"The type declaration ~S is lacking any variables.  You're parentheses may be in the wrong place." decl-spec)
	(let ((type (second decl-spec))
	      (symbols (cddr decl-spec))
	      )
	  (assert (every #'symbolp symbols) ()
		  "The type declaration ~S contains non-symbols as objects being declared" decl-spec)
	  (let ((canonical-type (valid-pvar-type-p type)))
	    (mapcar #'(lambda (symbol) (list symbol canonical-type)) symbols)
	    )))
    type-decl-specs
    ))


(defun allocate-array-pvar-form-given-canonical-type (canonical-pvar-type)
  (let ((dimensions (array-pvar-type-dimensions canonical-pvar-type))
	(element-type (array-pvar-type-element-type canonical-pvar-type))
	)
    (cond
      ((eq element-type t)
       (error "*Lisp does not currently allow arrays with element type T (i.e., arrays with general pvars as elements"))
      ((eq element-type '*)
       (error "*Lisp does not currently allow arrays with element type * (i.e., arrays with unspecified pvars as elements)"))
      ((eq '* (length-pvar-type `(pvar ,element-type)))
       (error "*Lisp does not currently allow arrays to be allocated whose element length is not completely specified"))
      )
    (let ((dimensions-form
	    (cond
	      ((eq '* dimensions)
	       (error "*Lisp does not allow arrays to be allocated with unspecified dimensions")
	       )
	      ((symbolp dimensions) dimensions)
	      ((every #'integerp dimensions) `',dimensions)
	      ((some #'(lambda (x) (eq x '*)) dimensions)
	       (error "*Lisp does not allow arrays to be allocated with any dimension size left unspecified")
	       )
	      (t `(list ,@dimensions))
	      ))
	  (dimensions-symbol (gensym "DIMENSIONS-TEMP-"))
	  )
      `(let ((,dimensions-symbol ,dimensions-form))
	 (allocate-temp-array-pvar
	   (make-canonical-pvar-type 'array :dimensions ,dimensions-symbol :element-type ',element-type)
	   )))))

(defun allocate-temp-pvar-form-given-canonical-pvar-type (canonical-pvar-type)
  (cond
    ((null canonical-pvar-type) `(allocate-temp-general-pvar))
    ((array-pvar-type-p canonical-pvar-type)
     (allocate-array-pvar-form-given-canonical-type canonical-pvar-type))
    ((structure-pvar-type-p canonical-pvar-type)
     `(allocate-temp-structure-pvar ',canonical-pvar-type)
     )
    (t `(allocate-temp-general-pvar))
    ))

(defun set-*let-pvar-fields (pvar name canonical-pvar-type)
  (setf (pvar-name pvar) name)
  (setf (pvar-canonical-pvar-type pvar) canonical-pvar-type)
  (setf (pvar-lvalue? pvar) t)
  (setf (pvar-constant? pvar) nil)
  )
  

(defun check-return-pvar-p (value return-pvar-p)
  (ecase return-pvar-p
    (:yes (assert (pvar-p value) ()
		  "You promised that the enclosing *defun, *let or *let* was returning a pvar, but it isn't, it's returning ~S"
		  value
		  ))
    (:no (assert (not (pvar-p value)) ()
		 "You promised that the enclosing *defun, *let or *let* was not returning a pvar, but it is, it's returning ~S"
		 value
		 ))
    (:maybe t)
    ))

(defun *let-1 (*let? bindings body env)
  
  (multiple-value-bind
    (
     bound-variables initialization-forms bound-variable-types unbound-variables
     unbound-variable-types other-decl-specs return-pvar-p body
     )
      (parse-*let-bindings-declarations-and-body bindings body env)
    
    other-decl-specs unbound-variable-types

    (dolist (v unbound-variables)
      (warn "The symbol ~S was declared but not bound in a *LET or *LET*" v)
      )
    
    (let ((bound-variable-types (mapcar #'canonical-pvar-type bound-variable-types)))

      (let* ((return-value-symbol (gensym "*LET-RETURN-VALUE-"))
	     (old-temp-pvar-list-symbol (gensym "OLD-TEMP-PVAR-LIST-"))
	     (is-definitely-pvar-code
	       `(or ,@(mapcar
			#'(lambda (v) `(eq ,v ,return-value-symbol))
			bound-variables
			)))
	     )
	     
	`(let ((,old-temp-pvar-list-symbol *temp-pvar-list*))

	   (,(if *let? 'let 'let*)
	    (,@(mapcar
		 #'(lambda (v i type)
		     (let ((temp (gensym (concatenate 'string (symbol-name v) "-"))))
		       `(,v (let ((,temp ,(allocate-temp-pvar-form-given-canonical-pvar-type type)))
			      (set-*let-pvar-fields ,temp ',v ',type)
			      ,@(when (not (eq i *no-binding-for-*let-value*))
				  `((*set ,temp ,i))
				  )
			      ,temp
			      ))))
		 bound-variables
		 initialization-forms
		 bound-variable-types
		 ))
	    nil

	    (let ((,return-value-symbol (progn ,@body)))
	      ,@(when (not (eq :maybe return-pvar-p))
		  `((check-return-pvar-p ,return-value-symbol ,return-pvar-p))
		  )
	      (handle-returning-pvar
		,return-value-symbol
		,old-temp-pvar-list-symbol
		,is-definitely-pvar-code
		))

	    ))))))





;;;; ****************************************************************************
;;;;
;;;;                         CODE FOR OTHER *LISP MACROS
;;;;
;;;; ****************************************************************************


(defun return-pvar-p-declaration-p (declare-form)
  (and
    (eql 2 (length declare-form))
    (listp (second declare-form))
    (eq 'return-pvar-p (first (second declare-form)))
    (eql 2 (length (second declare-form)))
    (member (second (second declare-form)) '(t nil))
    t
    ))

(defun return-pvar-p-declaration-value (return-pvar-p-declaration)
  (second (second return-pvar-p-declaration))
  )

(defun remove-return-pvar-p-from-body (body function-name)
  (if (and (listp body)
	   (listp (car body))
	   (eq 'declare (caar body))
	   )
      (let ((declare-form (car body)))
	(if (return-pvar-p-declaration-p declare-form)
	    (let ((value (return-pvar-p-declaration-value declare-form)))
	      (if (member value '(t nil))
		  (values (cdr body) (if value :yes :no))
		  (error "Illegal value, ~S, for return-pvar-p declaration.  Only T or NIL are allowed" value)
		  ))
	    (error "Illegal declare form, ~S, in ~S.  Only return-pvar-p declarations are allowed." declare-form function-name)
	    ))
      (values body :maybe)
      ))


(defmacro *all (&body body)
  "Select all processors for the body"
  (multiple-value-bind (body return-pvar-p)
      (remove-return-pvar-p-from-body body '*all)
    (let ((css-index-symbol (gensym "CSS-INDEX-"))
	  (value-symbol (gensym "*ALL-RETURN-VALUE-"))
	  (old-temp-pvar-list-symbol (gensym "OLD-TEMP-PVAR-LIST-"))
	  )
      `(let ((,css-index-symbol *css-current-level*)
	     (,old-temp-pvar-list-symbol *temp-pvar-list*)
	     )
	 (prog2
	   (push-css-select-all)
	   (let ((,value-symbol (progn .,body)))
	     (check-return-pvar-p  ,value-symbol ,return-pvar-p)
	     (handle-returning-pvar ,value-symbol ,old-temp-pvar-list-symbol nil)
	     )
	   (pop-css-to-level ,css-index-symbol)
	   )))))


(defmacro *when (test-pvar &body body)
  "subselect all processors with test-pvar non-nil"
  (multiple-value-bind (body return-pvar-p)
      (remove-return-pvar-p-from-body body '*when)
    (let ((css-index-symbol (gensym "CSS-INDEX-"))
	  (old-temp-pvar-list-symbol (gensym "OLD-TEMP-PVAR-LIST-"))
	  (test-pvar-symbol (gensym "TEST-PVAR-"))
	  (value-symbol (gensym "*WHEN-RETURN-VALUE-"))
	  )
      `(let* ((,css-index-symbol *css-current-level*)
	      (,old-temp-pvar-list-symbol *temp-pvar-list*)
	      (,test-pvar-symbol ,test-pvar)
	      ,value-symbol)
	 (simple-pvar-argument!! ,test-pvar-symbol)
	 (setq ,value-symbol
	       (prog2
		 (push-css ,test-pvar-symbol)
		 (progn ,@body)
		 (pop-css-to-level ,css-index-symbol)
		 ))
	 (check-return-pvar-p ,value-symbol ,return-pvar-p)
	 (handle-returning-pvar
	   ,value-symbol
	   ,old-temp-pvar-list-symbol
	   nil
	   )))))


(defmacro *unless (test-pvar &body body)
  `(*when (not!! ,test-pvar)
     ,@body
     ))


(defmacro *if (condition true-clause &optional else-clause)
  (if (null else-clause)
      `(progn (*when ,condition ,true-clause) (values))
      `(progn
	 ,(let ((condition-symbol (gensym "*IF-CONDITION-")))
	    `(*let ((,condition-symbol ,condition))
	       (*when ,condition-symbol ,true-clause)
	       (*when (not!! ,condition-symbol) ,else-clause)
	       ))
	 (values)
	 )))



(DEFMACRO *COND (&REST CLAUSES)
  "Similar to *IF"
  (COND ((NULL CLAUSES) NIL)
	((EQL (LENGTH CLAUSES) 1)
	 `(*IF ,(FIRST (FIRST CLAUSES)) (PROGN ,@(REST (FIRST CLAUSES))))
	 )
	((EQ (FIRST (FIRST CLAUSES)) 'T!!)
	 ;; if there are more clauses, issue a warning to that effect
	 (when (NOT (EQL (LENGTH CLAUSES) 1))
	   (ERROR "~% *COND: T!! is used in a clause other than the last")
	   ))
	(T
	 `(*IF ,(FIRST (FIRST CLAUSES))
	       (PROGN ., (REST (FIRST CLAUSES)))
	       (*COND ., (REST CLAUSES))))))


(DEFMACRO IF!! (PVAR-EXPRESSION TRUE-PVAR &OPTIONAL (ELSE-PVAR NIL!!))

  "IF!! will return a pvar.  This pvar will contain TRUE-PVAR
   for all processors with PVAR-EXPRESSION 
   true and ELSE-PVAR for all those with PVAR-EXPRESSION false.
   During the execution of TRUE-PVAR, the CSS
   will be set to those processors that passed PVAR-EXPRESSION,
   whereas during execution of ELSE-PVAR, the
   CSS will be set to those processors which failed PVAR-EXPRESSION.
  "
  
  (let ((if!!-result-symbol (gensym "IF!!-RESULT-")))
    `(*let (,if!!-result-symbol)
       (*if ,pvar-expression (*set ,if!!-result-symbol ,true-pvar) (*set ,if!!-result-symbol ,else-pvar))
       ,if!!-result-symbol
       )))

(defmacro cond!! (&rest clauses)
  (let ((len (length clauses)))
    (cond ( (eql 0 len) 'nil!! )
	  ( (eql 1 len)
	    (let* ((clause (car clauses))
		   (test (car clause))
		   (consequents (cdr clause))
		  )
	      (if (null consequents)
		  test
		  `(if!! ,test (progn ,@consequents) nil!!)
	       ))
	  )
	  ( t
	    (let* ((clause1 (car clauses))
		   (test1 (car clause1))
		   (consequents1 (cdr clause1))
		  )
	      (if (null consequents1)
		  (let ((test-symbol (gensym "COND!!-TEMP-TEST-")))
		    `(*let ((,test-symbol ,test1))
		       (if!! ,test-symbol
			     ,test-symbol
			     (cond!! ,@(cdr clauses))
			)))
		  `(if!! ,test1
			 (progn ,@consequents1)
			 (cond!! ,@(cdr clauses))
		    )))
	  )
      )))



(DEFMACRO AND!! (&REST PVARS)
  (COND ((NULL PVARS) 'T!!)
	((= 1 (LENGTH PVARS))
	 (LET ((SYM (GENSYM "AND!!-")))
	   `(let ((,sym ,(FIRST PVARS)))
	      (simple-pvar-argument!! ,sym)
	      (pvar-check ,sym)
	      ,sym
	     )))
	(T
	 `(IF!! ,(FIRST PVARS) (AND!! ., (REST PVARS)) NIL!!))))

;;; (AND!! 1 2)



(DEFMACRO OR!! (&REST PVARS)
  (COND ((NULL PVARS) 'NIL!!)
	((= 1 (LENGTH PVARS))
	 (LET ((SYM (GENSYM "OR!!-")))
	   `(let ((,sym ,(FIRST PVARS)))
	      (simple-pvar-argument!! ,sym)
	      (pvar-check ,sym)
	      ,sym
	     )))
	(T
	 `(LET ((TEMP ,(FIRST PVARS)))
	    (IF!! TEMP TEMP (OR!! ., (REST PVARS)))))))

;;; (or!! 1 2 3)


(defmacro with-css-saved (&rest body)

  "Save the state of the temporary pvar list and the
   currently selected set.  If anyone tries to
   break out of the body of the form we use
   unwind-protect to restore state before allowing
   the exit.
  "

  (let ((foo (gensym "OLD-CSS-LEVEL"))
	(bar (gensym "OLD-TEMP-PVAR-LIST-"))
	(baz (gensym "WITH-CSS-BODY-VALUE-"))
       )
    `(let ((,foo *css-current-level*)
	   (,bar *temp-pvar-list*)
	   (,baz nil)
	  )
       (unwind-protect
	   (setq ,baz (progn ,@ body))
	 (progn
	   (pop-css-to-level ,foo)
	   (handle-returning-pvar ,baz ,bar nil)
	   )))))


(defun *defun-function-symbol-exists? (symbol)
  (let ((*defun-symbol (make-*defun-function symbol)))
    (if (and (macro-function symbol) (fboundp *defun-symbol)) *defun-symbol nil)
    ))

(defun *apply-error (who f-name)
  (error "In ~A: The ~S operator is a macro.~%~
    ~A cannot accept macros other than those defined by *DEFUN."
	 who f-name who))

(defmacro *funcall (f &rest args)
  (let ((actual-function-symbol (gensym "*FUNCALL-ACTUAL-FUNCTION-"))
	(old-temp-pvar-symbol (gensym "*FUNCALL-OLD-TEMP-PVAR-"))
	(f-symbol (gensym "FUNCTION-ARGUMENT-"))
       )
    `(let* ((,old-temp-pvar-symbol *temp-pvar-list*)
	    (,f-symbol ,f)
	    (,actual-function-symbol
	      (if (symbolp ,f-symbol)
		  (or (*defun-function-symbol-exists? ,f-symbol)
		      (when (macro-function ,f-symbol)
			(*apply-error '*funcall ,f-symbol))
		      ,f-symbol)
		  ,f-symbol
		  ))
	    )
       (handle-returning-pvar
	 (funcall ,actual-function-symbol ,@args) ,old-temp-pvar-symbol nil)
       )))


(defmacro *apply (f arg &rest args)
  (let ((actual-function-symbol (gensym "*APPLY-ACTUAL-FUNCTION-"))
	(old-temp-pvar-symbol (gensym "*APPLY-OLD-TEMP-PVAR-"))
	(f-symbol (gensym "FUNCTION-ARGUMENT-"))
	)
    `(let* ((,old-temp-pvar-symbol *temp-pvar-list*)
	    (,f-symbol ,f)
	    (,actual-function-symbol
	      (if (symbolp ,f-symbol)
		  (or (*defun-function-symbol-exists? ,f-symbol)
		      (when (macro-function ,f-symbol)
			(*apply-error '*apply ,f-symbol))
		      ,f-symbol)
		  ,f-symbol
		  ))
	    )
       (handle-returning-pvar
	 (apply ,actual-function-symbol (list* ,arg ,@args)) ,old-temp-pvar-symbol nil)
       )))


;;;; Code for CASE!! and friends.  Written by Jim Salem.


(defmacro once-only (variable-list &body body)
  (cond ((null variable-list) `(progn .,body))
	(t
	 (let* ((var (car variable-list)))
	   `(once-only ,(cdr variable-list)
	      (cond ((or (atom ,var) (member (car ,var) '(quote function)))
		     ;;; Variable is an atom or quoted.
		     ;;; Don't expand differently
		     .,body)
		    (t
		     (let* ((temp (gensym "ONCE-ONLY-VAR"))
			    (actual-arg ,var)
			    (form (gensym "ONCE-ONLY"))
			    (,var temp)
			    )
		       (setq form (progn .,body))
		       `(let ((,temp ,actual-arg))
			  ,form)))))))))


(defmacro def-*lisp-case-form (case-function-name ecase-function-name cond-function-name)

  `(progn

     (defmacro ,case-function-name (pvar-expression &body clauses)
       (once-only (pvar-expression)
	 `(,',cond-function-name
	    .,(let ((last-clause-found-p nil)
		    keys body
		    (return-cond-clauses nil)
		    )
		(dolist (clause clauses)
		  ;; For example, CLAUSE could be  ((:apples :oranges) (eat-fruit :seedsp t))
		  (setq keys (car clause)) ;; E.G. (:apples :oranges)
		  (setq body (cdr clause)) ;; E.G. ((eat-fruit :seedsp t))
		  
		  ;;; WARNINGS ---

		  ;;; Do we have a clause we can never reach ?
		  (when last-clause-found-p
		    (warn "The clause ~S appeared after a T or OTHERWISE clause in ~A"
			  clause ',case-function-name))

		  (when (or (eql keys 't!!) (and (listp keys)
						 (or (eql (car keys) '!!)
						     (and (listp (car keys))
							  (eql (caar keys) '!!))
						     )))
		    (warn "The keywords for ~A shouldn't be pvars (in clause ~S)."
			  ',case-function-name
			  clause)
		    )

		  ;;; GENERATE COND CLAUSES
		  (push (cond 
			  ((listp keys)
			   ;;; Multiple keywords for this clause
			   `((or!! .,(mapcar #'(lambda (x)
						 `(eql!! ,pvar-expression (!! ',x))) keys))
			     .,body))
			  ((or (eql keys 't) (eql keys 'otherwise))
			   ;;; Final clause
			   (setq last-clause-found-p t)
			   `(t!! .,body))
			  (t
			   ;;; A clause with a single keyword
			   `((eql!! ,pvar-expression (!! ',keys)) .,body)
			   )
			  )

			return-cond-clauses))
		(nreverse return-cond-clauses)
		))))

     (defmacro ,ecase-function-name (pvar-expression &body clauses)
       (let ((all-keys nil)
	     keys
	     )
	 ;;; Collect the names of all the keys
	 (dolist (clause clauses)
	   (setq keys  (car clause))
	   (cond ((atom keys) (push keys all-keys))
		 (t (dolist (key keys)
		      (push key all-keys)
		      ))
		 ))
	 `(,',case-function-name ,pvar-expression
	    ,@clauses
	    (t
	      (when (*or t!!)
		(error "For ~A, PVAR-EXPRESSION in some processor was not one of ~{~S~^, ~}."
		       ',',ecase-function-name ',all-keys))

	      nil!!))))
     ))

(def-*lisp-case-form case!! ecase!! cond!!)
(def-*lisp-case-form *case *ecase *cond) 


;;;; ****************************************************************************
;;;;
;;;;                         MISCELLANEOUS MACROS
;;;;
;;;; ****************************************************************************


(defmacro valid-integer-range (x lower-inclusive-limit upper-inclusive-limit)
  `(and (integerp ,x) (>= ,x ,lower-inclusive-limit) (<= ,x ,upper-inclusive-limit))
 )

(defmacro valid-integer-range-exclusive (x lower-limit upper-limit)
  `(and (integerp ,x) (>= ,x ,lower-limit) (< ,x ,upper-limit))
 )

(defmacro valid-integer-range-inclusive (x lower-limit upper-limit)
  `(and (integerp ,x) (>= ,x ,lower-limit) (< x ,upper-limit))
 )

(defmacro valid-start-and-end (start end &key (min-start 0) max-end)
  `(and (integerp ,start) (integerp ,end)
	(>= ,start ,min-start) ,@(if max-end `((< ,end ,max-end)) nil)
	(<= ,start ,end)
    ))

(defmacro valid-start-and-limit (start limit &key (min-start 0) max-limit)
  `(and (integerp ,start)
	(integerp ,limit)
	(>= ,start ,min-start)
	,@(if max-limit `((<= ,limit ,max-limit)) nil)
	(< ,start ,limit)
    ))

(defmacro incf-use-count (function-name)
;  `(incf (get ,function-name 'use-count))
  (declare (ignore function-name))
  nil
 )



(defmacro while (condition &rest body)
   `(loop (if (null ,condition) (return)) ,@body))

(defmacro until (condition &rest body)
   `(loop (if ,condition (return)) ,@body))

(defmacro for++ ((var init end) &rest body)
  `(let ((,var ,init))
     (loop
       (if (<= ,var ,end)
           (progn (progn ,@body) (incf ,var))
           (return)
        ))))

(defmacro for-- ((var init end) &rest body)
  `(let ((,var ,init))
     (loop
       (if (>= ,var ,end)
           (progn (progn ,@body) (decf ,var))
           (return)
        ))))


(defmacro for (((var init &optional result) endtest step) &rest body)
  `(let ((,var ,init))
     (loop
       (and ,endtest (return ,result))
       (progn ,.body)
       ,step
      )))


(defmacro assocv-cadr (value list) `(cadr (assoc ,value ,list)))

(defmacro assocv-cdr (value list) `(cdr (assoc ,value ,list)))


(defmacro no-processors-active () `(*and nil!!))
(defmacro some-processor-active () `(not (*and nil!!)))
(defmacro all-processors-active () `(eql *number-of-processors-limit* (*sum (!! 1))))


(defmacro with-array-elements-iterated (array var &body body)
  (assert (symbolp var))
  (let ((array-symbol (gensym "ARRAY-"))
	(body-function (gensym "WITH-ARRAY-ELEMENTS-ITERATED-BODY-FUNCTION-"))
	)
    `(let ((,array-symbol ,array))
       (flet
	 ((,body-function (,var) ,@body))
	 (if (not (eql 1 (array-rank ,array-symbol)))
	     (let ((displaced-array (make-array (array-total-size ,array-symbol) :displaced-to ,array-symbol)))
	       (dotimes (j (length displaced-array))
		 (,body-function (aref displaced-array j))
		 ))
	     (dotimes (j (length ,array-symbol))
	       (,body-function (aref ,array-symbol j))
	       ))))))


(defmacro with-many-array-elements-iterated ((&rest vars) (&rest arrays) &body body)

  (assert (every #'symbolp vars))
  (assert (eql (length vars) (length arrays)) () "Number of variables not equal to number of arrays to iterate over")

  (let* ((array-symbols (mapcar #'(lambda (var) (gensym (concatenate 'string (symbol-name var) "-ARRAY-"))) vars))
	 (displaced-array-symbols
	   (mapcar #'(lambda (array-symbol) (gensym (concatenate 'string "DISPLACED-" (symbol-name array-symbol)))) array-symbols))
	 (body-function (gensym "WITH-MANY-ARRAY-ELEMENTS-ITERATED-BODY-FUNCTION-"))
	)

    (if (zerop (length vars))

	nil

	`(let
	   (,@(mapcar #'(lambda (array-symbol array) `(,array-symbol ,array)) array-symbols arrays))
	   (flet
	     ((,body-function (,@vars) ,@body))
	     (let* ((first-array ,(first array-symbols))
		    (array-dimensions (array-dimensions first-array))
		    )
	       (assert (every #'(lambda (array) (equal array-dimensions (array-dimensions array))) (list ,@array-symbols))
		       ()
		       "Error.  All the arrays you are iterating over do not have the same dimensions"
		       )
	       (if (eql 1 (array-rank first-array))
		   (dotimes (j (length ,(first array-symbols)))
		     (,body-function ,@(mapcar #'(lambda (array-symbol) `(aref ,array-symbol j)) array-symbols))
		     )
		   (let
		     (,@(mapcar
			  #'(lambda (displaced-array-symbol array-symbol)
			      `(,displaced-array-symbol (make-array (array-total-size ,array-symbol) :displaced-to ,array-symbol))
			      )
			  displaced-array-symbols array-symbols
			  ))
		     (dotimes (j (length ,(first displaced-array-symbols)))
		       (,body-function ,@(mapcar #'(lambda (array-symbol) `(aref ,array-symbol j)) displaced-array-symbols))
		       )))))))))


(defmacro with-structure-elements-iterated
	  (((&rest vars) (&rest structures) slot-accessor-functions &key (aliased? nil)) &body body)
  
  (assert (every #'symbolp vars))
  (assert (eql (length vars) (length structures)) () "Number of variables not equal to number of structures to iterate over")

  (let* ((structure-symbols (mapcar #'(lambda (var) (gensym (concatenate 'string (symbol-name var) "-STRUCTURE-"))) vars))
	 (body-function (gensym "WITH-STRUCTURE-ELEMENTS-ITERATED-BODY-FUNCTION-"))
	 (slot-accessor-functions-list-symbol (gensym "SLOT-ACCESSORS-LIST-"))
	)

    (if (zerop (length vars))

	nil

	`(let
	   (,@(mapcar #'(lambda (structure-symbol structure) `(,structure-symbol ,structure)) structure-symbols structures))
	   (let ((,slot-accessor-functions-list-symbol ,slot-accessor-functions))
	     (flet
	       ((,body-function (,@vars) ,@body))
	       (let* ((first-structure ,(first structure-symbols))
		      (type (type-of first-structure))
		      )
		 (assert (every #'(lambda (structure) (eq type (type-of structure))) (list ,@structure-symbols))
			 ()
			 "Error.  All the structures you are iterating over are not of the same type"
			 )
		 (dolist (slot-accessor
			   ,(if (not aliased?)
				slot-accessor-functions-list-symbol
				`(mapcar #'*defstruct-slot-alias!!-function ,slot-accessor-functions-list-symbol)
				))
		   (,body-function
		    ,@(mapcar #'(lambda (structure) `(funcall slot-accessor ,structure)) structure-symbols)
		    )))))))))






(defmacro do-for-selected-processors-internal ((var) &body body)
  (assert (symbolp var))
  (let ((local-css-symbol (gensym "LOCAL-CSS-")))
    `(let ((,local-css-symbol *css*))
       (bit-vector-array-declaration ,local-css-symbol)
       (dotimes (,var *number-of-processors-limit*)
	 (declare (type fixnum ,var))
	 (when (eql 1 (sbit ,local-css-symbol ,var)) ,.body)
	 ))))

(defmacro with-selected-general-pvar-arrays ((processor-index &key (return-any-set t)) (&rest vars) (&rest pvars) &body body)
  (assert (eql (length vars) (length pvars)))
  (assert (every #'symbolp vars))
  (assert (symbolp processor-index))
  (let ((any-set-symbol (gensym "ANY-SET-")))
    `(let
       (,@(mapcar #'(lambda (symbol pvar) `(,symbol (pvar-array ,pvar))) vars pvars)
	,@(if return-any-set `((,any-set-symbol nil)))
	)
       (1-d-array-declaration ,@vars)
       (do-for-selected-processors-internal (,processor-index)
	 ,@(if return-any-set `((setq ,any-set-symbol t)))
	 ,@body
	 )
       ,@(if return-any-set `(,any-set-symbol))
       )))

(defmacro with-scalar-body-mapped-into-result-pvar ((result-pvar) (&rest source-pvars) function-of-n-scalars)
  (let ((result-pvar-array-symbol (gensym "RESULT-PVAR-ARRAY-"))
	(index-symbol (gensym "PROCESSOR-INDEX-"))
	)
    (let* ((source-pvar-symbols (mapcar #'(lambda (pvar) (declare (ignore pvar)) (gensym "SOURCE-PVAR-ARRAY-")) source-pvars))
	   (aref-forms (mapcar #'(lambda (symbol) `(aref ,symbol ,index-symbol)) source-pvar-symbols))
	   )
      `(with-selected-general-pvar-arrays
	 (,index-symbol) (,@(cons result-pvar-array-symbol source-pvar-symbols)) (,@(cons result-pvar source-pvars))
	 (setf (aref ,result-pvar-array-symbol ,index-symbol)
	       ,(if (symbolp function-of-n-scalars)
		    `(,function-of-n-scalars ,@aref-forms)
		    `(funcall ,function-of-n-scalars ,@aref-forms)
		    ))))))

(defmacro *compile ((&key (safety 1) (warning-level :normal)) &body body)
  (declare (ignore safety warning-level))
  `(progn ,@body)
  )


(defmacro *nocompile (&body body) `(progn ,@body))

(defmacro *compile-blindly (&body body) `(progn ,@body))

(defun *lisp-compiler-hook (form env) form env nil)

(defmacro pref
    #-(OR :CCL :ALLEGRO KCL)
    (&whole form pvar processor &key vp-set &environment macroexpand-environment)
    #+(OR :CCL :ALLEGRO KCL)
    (&whole form &environment macroexpand-environment pvar processor &key vp-set)

  ;; What's going on here?

  ;; We need to look at the form to evaluate to get the pvar being referenced,
  ;; and the form to evaluate to get the processor from which we will do
  ;; the extraction.

  ;; There are two cases for the pvar expression.  Either it is a symbol or
  ;; it is not.  If it is a symbol, life is easy; we can just call
  ;; pref-function, which works with a pvar in any vp set.  If it is not
  ;; a symbol, then two things must happen:  we must evaluate the expression
  ;; in the proper vp set, and we must evaluate the expression in the
  ;; active set which consists solely of the processor being read from.
  ;; The :vp-set argument, if provided, is used to determine the proper
  ;; vp set in expression case.  If it is not provided in the expression
  ;; case the *current-vp-set* is assumed, and the user will lose if
  ;; the expression references pvars in a different vp set.

  ;; There are two major cases for the processor form.  Either it is
  ;; of the form (grid!! &rest args) or it is not.

  ;; If it is a grid!! call, then there are 3 subcases:

  ;; First, if no :vp-set argument is provided and the pvar expression
  ;; is a symbol, we convert the arguments of the grid!! call into
  ;; a cube address using the vp-set of the pvar symbol.

  ;; Second, if a :vp-set argument is provided, we convert the
  ;; arguments using that vp set.

  ;; Finally, if no :vp-set argument is provided and the pvar expression
  ;; is not a symbol we convert using *current-vp-set*.

  ;; If the form is not a grid!! call, then what it evaluates to can
  ;; be one of two things: a cube address or an address object.

  ;; If it is a cube address we just use it.  If it is an address object,
  ;; then we must extract a cube address from it.  To do this we must
  ;; transform the address object into an address object in the proper
  ;; vp-set.  The proper vp set is one of three possibilities, as discussed
  ;; above.

  ;; First check if the *compiler wants to deal with it.  If so, let it!

  (or (*lisp-compiler-hook form macroexpand-environment)     

      (let ((me-pvar-expression (macroexpand pvar macroexpand-environment))
	    (me-processor-expression (macroexpand processor macroexpand-environment))
	    )
	(let* ((is-grid-expression (and (listp me-processor-expression) (eq 'grid (car me-processor-expression))))
	       (processor-form
		 (if is-grid-expression
		     (cond
		       (vp-set `(cube-from-vp-grid-address ,vp-set ,@(cdr me-processor-expression)))
		       ((symbolp me-pvar-expression)
			`(cube-from-vp-grid-address (pvar-vp-set ,me-pvar-expression) ,@(cdr me-processor-expression))
			)
		       (t `(cube-from-grid-address ,@(cdr me-processor-expression)))
		       )
		     me-processor-expression
		     ))
	       )
		       
	  (if (symbolp me-pvar-expression)
	      (let ((vp-set-form (if vp-set vp-set `(if (fast-pvarp ,me-pvar-expression)
							(pvar-vp-set ,me-pvar-expression)
							*current-vp-set*))))
		`(new-pref-function ,me-pvar-expression ,processor-form ,vp-set-form))
	      (let ((vp-set-form (if vp-set vp-set '*current-vp-set*)))
		`(new-pref-function #'(lambda () 
					(let ((pvar-exp ,me-pvar-expression))
					  (simple-pvar-argument!! pvar-exp)
					  pvar-exp)) ,processor-form ,vp-set-form)))))))


(defmacro pref-grid (pvar &rest indices) `(pref ,pvar (grid ,@indices)))

(defmacro all!! (&body body) `(*all (declare (return-pvar-p t)) ,@body))
(defmacro when!! (condition &body body) `(*when ,condition (declare (return-pvar-p t)) ,@body))
(defmacro let!! (bindings &body body) `(*let ,bindings (declare (return-pvar-p t)) ,@body))

(defmacro with-grid-indices-iterated

	  ((iterated-index-list-symbol
	     number-of-grid-indices
	     &key
	     start-index-list
	     end-index-list
	     mask
	     (check-arguments t)
	     (direction :backward)
	     (bind-as :list)
	     )
	   &rest body
	   )

  (assert (symbolp iterated-index-list-symbol))

  (let ((start-temp (gensym "START-INDEX-LIST-"))
	(end-temp (gensym "END-INDEX-LIST-"))
	(mask-temp (gensym "MASK-"))
	(n-dimensions-temp (gensym "N-DIMENSIONS-"))
	(direction-temp (gensym "DIRECTION-"))
	(bind-as-temp (gensym "BIND-AS-"))
	)

    `(let* ((,start-temp ,start-index-list)
	    (,end-temp ,end-index-list)
	    (,mask-temp ,mask)
	    (,n-dimensions-temp ,number-of-grid-indices)
	    (,direction-temp ,direction)
	    (,bind-as-temp ,bind-as)
	    )

       (when ,check-arguments
	 (check-args-for-with-grid-indices-iterated
	   ,direction-temp ,bind-as-temp ,mask-temp ,start-temp ,end-temp ,n-dimensions-temp))

       (when (null ,start-temp)
	 (setq ,start-temp (make-sequence 'vector ,n-dimensions-temp :initial-element 0)))
       (when (null ,end-temp)
	 (setq ,end-temp (concatenate 'vector *current-cm-configuration*)))
       (when (null ,mask-temp)
	 (setq ,mask-temp (make-list ,n-dimensions-temp :initial-element t)))


       (setq ,iterated-index-list-symbol
	     (concatenate (if (eq ,bind-as-temp :list) 'list 'vector) ,start-temp))

       (loop

         (progn ,@body)

	 (when
	   (null
	     (next-grid-coordinates
	       ,start-temp
	       ,end-temp
	       ,iterated-index-list-symbol
	       ,mask-temp
	       ,n-dimensions-temp
	       ,direction-temp
	       ))
	   (return)
	   )

	 ))))



(defmacro with-displaced-arrays ((&rest arrays) (&rest displaced-array-symbols) &body body)
  (assert (eql (length arrays) (length displaced-array-symbols)))
  (assert (every #'symbolp arrays))
  `(let
     ,(mapcar
	#'(lambda (array displaced-array-symbol)
	    `(,displaced-array-symbol
	      (if (vectorp ,array)
		  ,array
		  (make-array (array-total-size ,array) :displaced-to ,array :element-type (array-element-type ,array))
		  )))
	arrays
	displaced-array-symbols
	)
     ,@body
     ))


(defmacro defun-wcefi (name args &body body)
  `(defun ,name ,args ,@body)
  )

(defmacro defun-wco (name args &body body)
  `(defun ,name ,args ,@body)
  )

(defmacro with-paris-from-*lisp (&body body)
  (declare (ignore body))
  (error "The macro with-paris-from-*lisp is obsolete as of Release 5.1.  It is no longer necessary.")
  )

(defmacro with-*lisp-from-paris (&body body)
  (declare (ignore body))
  (error "The macro with-*lisp-from-paris is obsolete as of Release 5.1.  It is no longer necessary.")
  )

(defmacro pref-grid-relative!! (pvar &rest index-pvars)
  (declare (ignore pvar index-pvars))
  (error "The macro pref-grid-relative!! is obsolete as of Release 5.0.  Use NEWS!!, NEWS-BORDER!! or PREF!! instead.")
  )

(defmacro *pset-grid (&rest obsolete-function-args)
  (declare (ignore obsolete-function-args))
  (error "The macro *pset-grid is obsolete as of Release 5.0.  Use *PSET instead.")
  )

(defmacro *pset-grid-relative (&rest obsolete-function-args)
  (declare (ignore obsolete-function-args))
  (error "The macro *pset-grid-relative is obsolete as of Release 5.0.  Use *PSET or *NEWS instead.")
  )


(defmacro address-object-cached-geometry-id (address-object-pvar)
  `(pvar-address-object-geometry-id ,address-object-pvar)
  )
(defmacro set-address-object-cached-geometry-id (address-object-pvar geometry-id)
  `(setf (pvar-address-object-geometry-id ,address-object-pvar) ,geometry-id)
  )

(defmacro coerce-integer-pvar-into-dest (dest source &key dest-type)
  (declare (ignore dest-type))
  `(*set ,dest ,source)
  )

(defmacro without-void-pvars (pvar-list &body body)
  (declare (ignore pvar-list))
  `(progn ,@body)
  )


(defmacro with-*defstruct-accessors-iterated ((accessor-function-var *defstruct-name) &body body)
  `(dolist (,accessor-function-var (list-of-*defstruct-accessors ,*defstruct-name))
     ,@body
     ))

(defmacro with-*defstruct-slot-descriptors-iterated ((slot-descriptor-var *defstruct-name) &body body)
  `(dolist (,slot-descriptor-var (list-of-*defstruct-slot-descriptors ,*defstruct-name))
     ,@body
     ))

(defun list-of-*defstruct-slot-descriptors (*defstruct-name)
  (*defstruct-all-slots-list (get *defstruct-name '*defstruct-structure))
  )

(defun list-of-*defstruct-accessors (*defstruct-name)
  (let* ((*defstruct-structure (get *defstruct-name '*defstruct-structure))
	 (slot-accessor-prefix-name (*defstruct-slot-accessor-prefix-name *defstruct-structure))
	 (slots-list (*defstruct-all-slots-list *defstruct-structure))
	 )
    (mapcar
      #'(lambda (slot)
	  (let ((slot-name (*defstruct-slot-name slot)))
	    (let ((accessor-function-name
		    (intern (concatenate
			      'string
			      (string slot-accessor-prefix-name)
			      (string slot-name)
			      "!!"
			      )
			    (symbol-package *defstruct-name)
			    )))
	      (when (not (fboundp accessor-function-name))
		(error "The accessor function name ~S for *defstruct ~S does not have a function binding"
		       accessor-function-name *defstruct-name
		       ))
	      accessor-function-name
	      )))
      slots-list
      )))
    

(defmacro ppme (form)
  `(let ((*compilep* t) (*compiling* t)
	 (*print-gensym* nil) (*print-case* :downcase)
	 (*print-length* nil) (*print-level* nil)
	 )
     (pprint (macroexpand-1 ',form))
     ))
