;;|=========================================================================|
;;|                         COPYRIGHT NOTICE                                |
;;|                                                                         |
;;|             Copyright 1990, 1991, 1992, 1993, Mark Tarver               |
;;|                                                                         |
;;|        Permission to use, copy, and distribute this software and        |
;;| its documentation for any purpose is hereby granted providing           |
;;| any such use, copying and distribution is not done                      |
;;| for money, securities or any other pecuniary benefit and that both      |
;;| the above copyright and this permission notice appear in all copies     |
;;| and in the supporting documentation.  Any modification of the software  |
;;| or documentation should be accompanied by the name of the author of the |
;;| modification, and Mark Tarver must be formally notified                 |
;;| of this modification before distributing the software.                  |
;;|                                                                         |
;;|       Any commercial use of this software or use of the names "SEQUEL", |
;;| or "Mark Tarver" in connection with any version, modified or            |
;;| unmodified, of this software, through publicity or advertising,         |
;;| requires written permission.  Mark Tarver makes no                      |
;;| representation about the suitability of this software for any purpose.  |
;;| SEQUEL is provided "as is" without express or implied warranty.         |
;;|                                                                         |
;;|       Mark Tarver disclaims all warranties with regard to               |
;;| this software, including all implied warranties of merchantability and  |
;;| fitness. In no event shall Mark Tarver be liable for any                |
;;| special, indirect or consequential damages or any damages whatsoever    |
;;| resulting from loss of use, data or profits, whether in an action of    |
;;| contract, negligence or other tortious action, arising out of or in     |
;;| connection with the use or performance of this software.                |
;;|                                                                         |
;;|=========================================================================|

(in-package :sequel)

(defun sequel
       nil
       (prog (*input*)
             (clear)
             (credit)
             (if *graphics*
                 (connect-widgets))
         loop (prompt)
             (setq *call* 0)
             (setq *pattern* nil)
             (setq *deflog+* nil)
             (setq *skip* 0)
             (setq *input* (lineread))
             (read-char)
             (update-history *input*)
             (catch 'abort
                    (sequel-eval *input*))
             (go loop)))

(defun sequel-eval
       (fp1)
       (cond ((and (consp fp1)
                   (eq '|!| (car fp1)))
              (prog (i)
                    (setq i (car *history*))
                    (sequel-print (first (bracket i)))
                    (terpri)
                    (return (sequel-eval i))))
             ((not (eval '*type-check*))
              (sequel-print (eval (sequel-form fp1))))
             ((and (consp fp1)
                   (consp (car fp1))
                   (reserved (caar fp1))
                   (not
                     (member (caar fp1)
                             (list 'define
                                   'defrew
                                   'deftactic
                                   'mutual))))
              (sequel-print (eval (car fp1))))
             (t
              (prog (sform type oldtype)
                    (setq sform (sequel-form fp1))
                    (not-type-checked fp1)
                    (setq type (typecheck sform
                                          (get-intended-type fp1)))
                    (sequel-print-with-type sform type)
                    (propagate-toplevel type oldtype)))))

(defun sequel-print-with-type
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'delayed fp2)
                   (member (car fp1)
                           (list 'define
                                 'defrew
                                 'deftactic)))
              (format t
                      ";;~A is delayed "
                      (eval fp1)))
             ((eq 'delayed fp2) (raise "code 14: TYPE FAILURE!"))
             (t (sequel-print (eval fp1)) (format t " * ~A" fp2))))

(defun get-intended-type
       (fp1)
       (cond ((equal (last (butlast fp1 1))
                     (list '*))
              (create-internal-variables (car (last fp1))))
             (t (type-var (car fp1)))))

(defun not-type-checked
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 1)
                   (and (consp (car fp1))
                        (> (length (car fp1)) 1)))
              (if (and (member (caar fp1)
                               (list 'define
                                     'defrew
                                     'deftactic))
                       (signature (cadar fp1)))
                  (raise
                    (format nil
                            "code 34: cannot overwrite signature of ~A at top level"
                            (cadar fp1)))))
             (t t)))

(defun propagate-toplevel
       (fp1 fp2)
       (cond ((equal fp2 fp1) t)
             ((and (eq 'delayed fp1)
                   (null fp2))
              t)
             ((occurs '-> fp1)
              (prog (topdefs)
                    (setq topdefs (topdefs (reverse (cdr *history*))
                                           nil
                                           'type-))
                    (return (pt1 topdefs topdefs 'no))))
             (t t)))

(defun topdefs
       (fp1 fp2 fp3)
       (cond ((null fp1) fp2)
             ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 1)
                   (and (consp (caar fp1))
                        (= (length (caar fp1))
                           1))
                   (eq 'type+
                       (caar (car fp1))))
              (topdefs (cdr fp1) fp2 'type+))
             ((and (consp fp1)
                   (eq 'type- fp3))
              (topdefs (cdr fp1) fp2 'type-))
             ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 1)
                   (and (consp (caar fp1))
                        (> (length (caar fp1))
                           1))
                   (member (caar (car fp1))
                           (list 'define
                                 'defrew
                                 'deftactic))
                   (not (signature (cadar (car fp1)))))
              (topdefs (cdr fp1)
                       (adjoin (car (car fp1))
                               fp2
                               ':test
                               'same-def)
                       'type+))
             ((and (consp fp1)
                   (consp (car fp1))
                   (= (length (car fp1)) 1)
                   (and (consp (caar fp1))
                        (= (length (caar fp1))
                           1))
                   (eq 'type-
                       (caar (car fp1))))
              (topdefs (cdr fp1) fp2 'type-))
             ((consp fp1) (topdefs (cdr fp1) fp2 fp3))
             (t (raise "code 13: No Patterns have Fired in topdefs"))))

(defun same-def
       (fp1 fp2)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (and (consp fp2)
                        (> (length fp2) 1))
                   (equal (car fp2) (car fp1))
                   (equal (cadr fp2) (cadr fp1)))
              t)
             (t nil)))

(defun pt1
       (fp1 fp2 fp3)
       (cond ((and (null fp1) (eq 'no fp3)) t)
             ((and (null fp1)
                   (null fp2)
                   (eq 'yes fp3))
              t)
             ((null fp1) (pt1 fp2 fp2 'no))
             ((consp fp1)
              (prog (type)
                    (setq type (typecheck (car fp1)
                                          (get-intended-type (car fp1))))
                    (if (eq type 'delayed)
                        (return (pt1 (cdr fp1) fp2 fp3)))
                    (format t
                            "~%~A * ~A"
                            (cadr (car fp1))
                            type)
                    (return
                      (pt1 (cdr fp1)
                           (remove (car fp1)
                                   fp2
                                   ':test
                                   'equal)
                           'yes))))
             (t (raise "code 13: No Patterns have Fired in pt1"))))

(defun update-history
       (fp1)
       (cond ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq '|!| (car fp1))
                   (listp (cadr fp1)))
              (warn "bad use of !"))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '|!| (car fp1))
                   (eq '|!| (cadr fp1)))
              (update-history (car *history*)))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '|!| (car fp1))
                   (integerp (cadr fp1)))
              (update-history (nth (cadr fp1)
                                   (reverse *history*))))
             ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq '|!| (car fp1)))
              (update-history (retrieve (explode (cadr fp1))
                                        *history*)))
             (t (push fp1 *history*))))

(defun type+
       nil
       (prog2 (format t "Type Checking Enabled~%")
              (set '*type-check* t)))

(defun type-
       nil
       (prog2 (format t "Type Checking Disabled~%")
              (set '*type-check* nil)))

(defun sequel-form
       (fp1)
       (cond ((equal (last (butlast fp1 1))
                     (list '*))
              (sequel-form (butlast fp1 2)))
             ((and (consp fp1)
                   (consp (car fp1))
                   (reserved (caar fp1)))
              (car fp1))
             (t (sequel-form1 (first (bracket fp1))))))

(defun remove-quotes
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (eq 'quote (car fp1)))
              (cadr fp1))
             ((consp fp1) (mapcar 'remove-quotes fp1))
             (t fp1)))

(defun retrieve
       (fp1 fp2)
       (cond ((null fp2) (prog2 (warn "No record of this input")
                                nil))
             ((and (consp fp2)
                   (consp (car fp2))
                   (= (length (car fp2)) 1)
                   (consp (caar fp2))
                   (atom (caar (car fp2)))
                   (prefix-match fp1
                                 (explode (caar (car fp2)))))
              (list (car (car fp2))))
             ((consp fp2) (retrieve fp1 (cdr fp2)))
             (t (raise "code 13: No Patterns have Fired in retrieve"))))

(defun prefix-match
       (fp1 fp2)
       (cond ((null fp1) t)
             ((and (consp fp1)
                   (consp fp2)
                   (equal (car fp2) (car fp1)))
              (prefix-match (cdr fp1) (cdr fp2)))
             (t nil)))

(defun print-untyped-functions
       (fp1 fp2)
       (cond ((null fp1) nil)
             ((null fp2) nil)
             ((consp fp2)
              (print-untyped-function (car fp2))
              (print-untyped-functions t (cdr fp2)))
             (t
              (raise
                "code 13: No Patterns have Fired in print-untyped-functions"))))

(defun print-untyped-function
       (fp1)
       (write-string "|")
       (write fp1)
       (terpri))

(defun prompt
       nil
       (if *type-check*
           (format t
                   "~%~%(~A+) "
                   (length *history*))
           (format t
                   "~%~%(~A-) "
                   (length *history*))))

(defun lineread
       nil
       (prog (*line*)
             (setq *line* (list (read-preserving-whitespace)))
         loop (cond ((or (eq (peek-char) #\Space)
                         (not (eq (peek-char) #\Newline)))
                     (setq *line* (cons (read-preserving-whitespace)
                                        *line*))
                     (go loop))
                    (t (return (reverse *line*))))))

(defun reserve
       (fp1)
       (pushnew fp1 *reserved*))

(defun unreserve
       (fp1)
       (setq *reserved* (remove fp1 *reserved* :test 'equal)))

(defun embed-op
       (fp1)
       (cond ((atom fp1) fp1)
             ((stringp fp1) fp1)
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'and (car fp1)))
              (list 'and
                    (embed-op (cadr fp1))
                    (embed-op (caddr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq 'and (car fp1)))
              (embed-op
                (cons 'and
                      (cons (list 'and
                                  (cadr fp1)
                                  (caddr fp1))
                            (cdddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'or (car fp1)))
              (list 'or
                    (embed-op (cadr fp1))
                    (embed-op (caddr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 2)
                   (eq 'or (car fp1)))
              (embed-op
                (cons 'or
                      (cons (list 'or
                                  (cadr fp1)
                                  (caddr fp1))
                            (cdddr fp1)))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'cond (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2))
                   (equal t (caadr fp1)))
              (embed-op (cadar (cdr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'cond (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2)))
              (embed-op
                (list 'if
                      (caadr fp1)
                      (cadar (cdr fp1))
                      (cons 'cond (cddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'cond (car fp1)))
              nil)
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'list (car fp1)))
              (embed-op (list 'cons
                              (cadr fp1)
                              (cons 'list (cddr fp1)))))
             ((and (consp fp1)
                   (= (length fp1) 1)
                   (eq 'list (car fp1)))
              nil)
             (t (mapcar 'embed-op fp1))))

(defun credit
       nil
       (format t
               "|==============================================|~%")
       (format t
               "| SEQUEL version 5.2, August 1993              |~%")
       (disp-type&version)
       (format t
               "| Running under XTT version 3.0                |~%")
       (format t
               "|                                              |~%")
       (format t "| Written and Designed by Mark Tarver          |~%")
       (disp-jeremy)
       (format t
               "|                                              |~%")
       (format t
               "| Computational Logic Group                    |~%")
       (format t
               "| Division of Artificial Intelligence          |~%")
       (format t
               "| Centre for Theoretical Computer Science      |~%")
       (format t
               "| University of Leeds                          |~%")
       (format t
               "|                                              |~%")
       (format t
               "| Copyright Mark Tarver 1990,1991,1992,1993    |~%")
       (format t
               "|==============================================|~%~%"))

(defun disp-jeremy ()
  (if (equal (lisp-implementation-type) "Lucid Common Lisp")
      (format t "| C Graphical Interface by Jeremy Littler      |~%")))

(defun disp-type&version
       nil
       (disp-type&version1 (lisp-implementation-type)
                           (lisp-implementation-version)))

(defun disp-type&version1
       (fp1 fp2)
       (format t
               "| Written for ~A ~A ~A|~%"
               fp1
               fp2
               (blank-string-n
                 (- 31
                    (+ (length (coerce fp1 'list))
                       (length (coerce fp2 'list)))))))

(defun blank-string-n
       (fp1)
       (cond ((>= 0 fp1) "")
             (t (format nil
                        " ~A"
                        (blank-string-n (1- fp1))))))

(defun sequel-print
       (fp1)
       (cond ((null fp1) (write-string "[]"))
             ((eq '\|- fp1) (write-string "|-"))
             ((bar fp1) (write-string "|"))
             ((and (consp fp1)
                   (atsign (car fp1)))
              (write-string "(")
              (mapcar 'sequel-print-with-spaces
                      (butlast (cdr fp1)))
              (if (dotted-pair (last fp1))
                  (progn (sequel-print (car (last fp1)))
                         (write-string " . ")
                         (sequel-print (cdr (last fp1))))
                  (sequel-print (car (last fp1))))
              (write-string ")"))
             ((consp fp1)
              (write-string "[")
              (mapcar 'sequel-print-with-spaces
                      (butlast fp1))
              (if (dotted-pair (last fp1))
                  (progn (sequel-print (car (last fp1)))
                         (write-string " . ")
                         (sequel-print (cdr (last fp1))))
                  (sequel-print (car (last fp1))))
              (write-string "]"))
             (t (write fp1))))

(defun dotted-pair
       (fp1)
       (cond ((and (consp fp1)
                   (not (listp (cdr fp1))))
              t)
             (t nil)))

(defun sequel-print-with-spaces
       (fp1)
       (sequel-print fp1)
       (write-string " "))

(defun t+
       nil
       (setq *type-check* t)
       (list 'type 'checking 'enabled))

(defun t-
       nil
       (setq *type-check* nil)
       (list 'type-checking 'disabled))

(defun global1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (consp (caddr fp1))
                   (some 'var
                         (flatten (caddr fp1))))
              (raise "code 24: types of globals must be monomorphic"))
             ((and (consp fp1)
                   (= (length fp1) 3)
                   (symbolp (car fp1))
                   (member (cadr fp1)
                           (list 'array
                                 'hash-table
                                 'pointer
                                 'identifier))
                   (pushnew (car fp1) *globals*))
              (setf (get (car fp1) 'global)
                    (cadr fp1))
              (setf (get (car fp1) 'type)
                    (caddr fp1))
              (car fp1))
             (t (raise "code 25: bad global declaration"))))

(defun put-prop
       (fp1 fp2 fp3)
       (setf (get fp1 fp2) fp3))

(defun get-prop
       (fp1 fp2 fp3)
       (get fp1 fp2 fp3))

(defun build-hash-table
       (fp1 fp2 fp3 fp4 fp5 fp6 fp7 fp8 fp9)
       (cond ((and (eq ':size fp2)
                   (eq ':test fp4)
                   (eq ':rehash-size fp6)
                   (eq ':rehash-threshold fp8))
              (prog2
                (set fp1
                     (make-hash-table ':size
                                      fp3
                                      ':test
                                      fp5
                                      ':rehash-size
                                      fp7
                                      ':rehash-threshold
                                      fp9))
                fp1))
             (t (raise "code 31: cannot build hashtable"))))

(defun put-hash
       (fp1 fp2 fp3)
       (setf (gethash fp2 (eval fp1))
             fp3))

(defun get-hash
       (fp1 fp2 fp3)
       (gethash fp2 (eval fp1) fp3))

(defun build-array
       (fp1 fp2)
       (prog2 (set fp1 (make-array fp2))
              fp1))

(defun get-array
       (fp1 fp2 fp3)
       (ga1 (apply 'aref
                   (cons (eval fp1) fp2))
            fp3))

(defun ga1
       (fp1 fp2)
       (cond ((null fp1) fp2)
             (t fp1)))

(defun put-array
       (fp1 fp2 fp3)
       (eval
         (list 'setf
               (cons 'aref
                     (append (list fp1) fp2))
               (list 'quote fp3))))

(defun restart
       nil
       (throw 'abort 'aborted))

(defun primitive1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 2)
                   (symbolp (cadr fp1))
                   (symbolp (car fp1)))
              (setf (get (car fp1) 'recognisor)
                    (cadr fp1))
              (if (not (member (car fp1) *primitives*))
                  (nconc *primitives*
                         (list (car fp1))))
              (car fp1))
             (t (raise "code 26: bad primitive declaration"))))

(defun boolp
       (fp1)
       (cond ((equal t fp1) t)
             ((null fp1) t)
             (t nil)))

(defun mysymbolp
       (fp1)
       (cond ((null fp1) nil)
             (t (symbolp fp1))))

(defun newv
       nil
       (concat (incf *newnumber*) '?))

(defun head
       (fp1)
       (cond ((consp fp1) (car fp1))
             (t (raise "code 13: No Patterns have Fired in head"))))

(defun tail
       (fp1)
       (cond ((consp fp1) (cdr fp1))
             (t (raise "code 13: No Patterns have Fired in tail"))))

(defun output
       (fp1)
       (prog2 (sequel-print fp1) t))

(defun eigen
       (fp1 fp2)
       (not (occurs fp1 fp2)))

(defun input
       nil
       (eval (sequel-form (lineread))))

(defun input+
       (fp1)
       (prog (input-a input-b)
         loop (setq input-a (sequel-form (lineread)))
             (setq input-b (remove-quotes input-a))
             (initstacks)
             (if (eq t
                     (catch 'delay
                            (phi input-b fp1 nil nil)))
                 (return (eval input-b))
                 (prog2
                   (format t
                           "~A is not of type ~A. Please re-enter~%"
                           input-b
                           fp1)
                   (go loop)))))

(defun history
       nil
       (prog (n h)
             (setq n 0)
             (setq h (reverse *history*))
         loop (if (null h)
                  (return t))
             (if (listp (caar h))
                 (format t "~A ~A~%" n (caar h))
                 (format t "~A ~A~%" n (car h)))
             (setq h (cdr h))
             (incf n)
             (go loop)))

(defun lookfor
       (fp1)
       (lookfor1 fp1
                 0
                 (reverse (eval '*history*))))

(defun lookfor1
       (fp1 fp2 fp3)
       (cond ((null fp3) (restart))
             ((and (consp fp3)
                   (consp (car fp3))
                   (= (length (car fp3)) 1)
                   (consp (caar fp3))
                   (equal (caar (car fp3)) fp1))
              (format t "~%~A. " fp2)
              (sequel-print (first (bracket (list (cons fp1
                                                        (cdaar fp3))))))
              (lookfor1 fp1 (1+ fp2) (cdr fp3)))
             ((consp fp3) (lookfor1 fp1 (1+ fp2) (cdr fp3)))
             (t (raise "code 13: No Patterns have Fired in lookfor1"))))

(defun metarewrite
       (fp1)
       (cond ((and (symbolp fp1)
                   (fboundp fp1))
              (metarewrite1 (subst 'head
                                   'car
                                   (subst 'tail
                                          'cdr
                                          (ppx fp1)))))
             (t (raise "code 13: No Patterns have Fired in metarewrite"))))

(defun metarewrite1
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 4)
                   (eq 'named-lambda (car fp1))
                   (and (consp (caddr (cdr fp1)))
                        (= (length (caddr (cdr fp1)))
                           3))
                   (eq 'block
                       (caadr (cddr fp1))))
              (define2
                (cons (concat '- (concat (cadr fp1) '-))
                      (rfps (cadr fp1)
                            (caddr fp1)
                            (caddr (caddr (cdr fp1)))
                            (newv)
                            nil))))
             ((and (consp fp1)
                   (= (length fp1) 5)
                   (eq 'named-lambda (car fp1))
                   (consp (caddr (cdr fp1)))
                   (eq 'declare
                       (caadr (cddr fp1)))
                   (and (consp (caddr (cddr fp1)))
                        (= (length (caddr (cddr fp1)))
                           3))
                   (eq 'block
                       (caadr (cdddr fp1)))
                   (equal (cadar (cdddr (cdr fp1)))
                          (cadr fp1))
                   (and (consp (caddr (caddr (cddr fp1))))
                        (= (length (caddr (caddr (cddr fp1))))
                           3))
                   (eq 'the
                       (caadr (cdadr (cdddr fp1)))))
              (define2
                (cons (concat (concat '- (cadr fp1)) '-)
                      (rfps (cadr fp1)
                            (caddr fp1)
                            (caddr (caddr (caddr (cddr fp1))))
                            (newv)
                            nil))))
             (t (raise "code 13: No Patterns have Fired in metarewrite1"))))

(defun rfps
       (fp1 fp2 fp3 fp4 fp5)
       (cond ((null fp2)
              (list (cons fp1 (reverse fp5))
                    '->
                    (elim-cond fp3)
                    'x
                    '->
                    'x))
             ((consp fp2)
              (rfps fp1
                    (cdr fp2)
                    (subst fp4 (car fp2) fp3)
                    (newv)
                    (cons fp4 fp5)))
             (t (raise "code 13: No Patterns have Fired in rfps"))))

(defun elim-cond
       (fp1)
       (cond ((and (consp fp1)
                   (= (length fp1) 3)
                   (eq 'cond (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2))
                   (and (consp (caddr fp1))
                        (= (length (caddr fp1))
                           2))
                   (equal t (caadr (cdr fp1))))
              (list 'if
                    (caadr fp1)
                    (cadar (cdr fp1))
                    (cadar (cddr fp1))))
             ((and (consp fp1)
                   (> (length fp1) 1)
                   (eq 'cond (car fp1))
                   (and (consp (cadr fp1))
                        (= (length (cadr fp1))
                           2)))
              (list 'if
                    (caadr fp1)
                    (cadar (cdr fp1))
                    (elim-cond (cons 'cond (cddr fp1)))))
             (t (raise "code 13: No Patterns have Fired in elim-cond"))))

(defun unfold
       (fp1)
       (cond ((and (consp fp1)
                   (symbolp (car fp1))
                   (fboundp (concat '- (concat (car fp1) '-))))
              (funcall (concat '- (concat (car fp1) '-))
                       fp1))
             (t fp1)))
