(comment (herald trucks))

; --------------------------------------------------------------------
; ** Objects of type TRUCK **
; ---------------------------
; This file contains definitions for:
;
; Operations:
; Object constructors
; This file makes use of things in - BASIC.T
;                                    THINGOIDS.T
;                                    CONTAINERS.T
;                                    ARMS.T
; --------------------------------------------------------------------
; * Special TRUCK type operations *
; ---------------------------------
; ERRORS: TRUCK-CANT-MOVE, TRUCK-MISHAP, EYE-CANT-FIND
; --------------------------------------------------------------------
; * Generic TRUCK object constructor *
; ------------------------------------

(defclass truck
  (thingoid environment)
  ((arms :accessor arms              :initarg :arms)
   (bays :accessor bays              :initarg :bays)
   (fuel-bay :accessor fuel-bay      :initarg :fuel-bay)
   (tire-bay :accessor tire-bay      :initarg :tire-bay)
   (weapon-bay :accessor weapon-bay  :initarg :weapon-bay)
   (base-speed :accessor base-speed  :initarg :base-speed)
   (base-fuel-consumption 
      :accessor base-fuel-consumption :initarg :base-fuel-consumption)
   (visibles :accessor visibles  :initarg :visibles)
   (environment :accessor environment  :initform *undefined-object*) 
   (base-mpg :accessor base-mpg        :initform 0)
   (base-mph :accessor base-mph        :initform 0)
   (odometer :accessor odometer        :initform 0)
   (truck-speed :accessor truck-speed  :initform 'stop)
   (truck-heading :accessor truck-heading  :initform  'n)
   (truck-status :accessor truck-status    :initform  'happy)
   (miles-left :accessor miles-left        :initform  0)
   (bumpiness :accessor bumpiness          :initform  'low)
   (monitor-used :accessor monitor-used    :initform  nil)
   (monitoring :accessor monitoring        :initform  nil)
   (known-about :accessor known-about      :initform  (fill (make-array 100) nil))
   (known-counter :accessor known-counter  :initform  0)
   (displayer :accessor displayer :initarg :displayer :initform  nil)))

;;; ** Operations handled by all objects

(defmethod update ((self truck) number time)
  (mapc #'(lambda (x) (update x number time))
        (holdings self))
  (cond ((and (monitoring self) 
              (not (monitor-used self)))
         (let ((beeped? '()))
           (do ((x (holdings (environment self)) (cdr x))) 
               ((null x) t)
             (cond ((and (some #'(lambda (y)
                                   (class? (car x) y))
                               (monitoring self))
                         (not (position (car x)
                                        (known-about self))))
                    (post-sensor-datum 'object-seen
                                       'external
                                       (unique-id (car x))
                                       (kind-id (car x))
                                       nil)
                    (vset (known-about self)
                          (known-counter self)
                          (car x))
                    (setf (known-counter self)
                          (+ (known-counter self)
                             1))
                    (cond ((not beeped?)
                           (if (displayer self)
			       (dm-speak (displayer self) '(prompt ding-dong)))
                           (setf beeped? T)))
                    )
                   )))
         ))
  (setf (bumpiness self) 'low)
  (setf (monitor-used self) nil)
  (let ((weapon (car (holdings (weapon-bay self)))))
    (if (and weapon
             (class? weapon 'weapon))
	(setf (displayer weapon) (displayer self))))
  t)

;;; this appears to be right for (Setf (environment self) new-location)

(defmethod (setf environment) :after ((new-location environment) (self truck))
   (if (and (displayer self)
	    (class? (environment self) 'map-node))
       (setf (displayer (environment self)) '()))
   (if (and (class? (environment self) 'environment)
			(class? (environment self) 'map-node))
       (post-sensor-datum 'new-location
			  (unique-id (environment self))
			  nil
			  nil
			  nil))
   (fill (known-about self) '())
   (setf (known-counter self) 0)
   (cond 
     ((and (displayer self) (class? (environment self) 'map-node))
      (setf (displayer (environment self)) (displayer self))
      (dm-set (displayer self) 'current-location (environment self))
      (t-iterate -*- ((arms (arms self)))
		 (cond 
		   ((null arms) (values))
		   (else 
		    (let ((name (unique-id (car arms))))
		      (do ((stuff (holdings (car arms))
				  (cdr stuff)))
			  ((null stuff)
			   T)
			(dm-redraw (displayer self) name (car stuff))))
		    (-*- (cdr arms)))))
      (t-iterate -*- ((bays (bays self)))
		 (cond 
		   ((null bays) (values))
		   (else 
		    (let ((name (unique-id (car bays))))
		      (do ((stuff (holdings (car bays))
				  (cdr stuff)))
			  ((null stuff)
			   T)
			(dm-redraw (displayer self) name (car stuff))))
		    (-*- (cdr bays))))))))

;;; ** Special ENVIRONMENT type operations

(defmethod visible? ((self truck) class) 
  (member class (visibles self) :test #'eq))

(Defmethod bumpiness ((self truck)) 
  (bumpiness self))

;;; ** Special CONTAINER type operations

(defmethod gross-bigness ((self truck))   ; **** cf container
  (cond ((holdings self) 
         (apply #'add
                (mapcar #'(lambda (x)
                            (gross-bigness x))
                        (holdings self))))
        (t 0)))


;;; truck stuff
                                  
(Defmethod truck-arm-move ((self truck) arm-id pos-whatsit)
  (let ((arm (obj-list-get-id (arms self) arm-id)))
    (cond (arm 
           (let ((new-position 
                  (cond ((eq pos-whatsit 'folded) 'folded)
                        ((eq pos-whatsit 'fuel-bay) (fuel-bay self))
                        ((eq pos-whatsit 'fuel-gauge) (fuel-bay self))
                        ((eq pos-whatsit 'tire-bay) (tire-bay self))
                        ((eq pos-whatsit 'weapon-bay) (weapon-bay self))
                        ((eq pos-whatsit 'external) (environment self))
                        ((symbolp pos-whatsit)
                         (or (obj-list-get-id (bays self) pos-whatsit)
                             pos-whatsit))
                        (t pos-whatsit))))     
             (and new-position
                  (arm-move arm new-position))))
          (t nil))))

(Defmethod truck-grasp ((self truck) arm-id thing-id)
 (let ((arm (obj-list-get-id (arms self) arm-id)))
   (and arm 
        (arm-grasp arm thing-id))))
                                  

(Defmethod truck-ungrasp ((self truck) arm-id thing-id)
 (let ((arm (obj-list-get-id (arms self) arm-id)))
   (and arm 
        (arm-ungrasp arm thing-id))))
                  
(Defmethod truck-pour ((self truck) arm-id vessel-id)
  (let* ((arm (obj-list-get-id (arms self) arm-id))
         (result (and arm
                      (arm-pour arm vessel-id))))
    (if result 
	(setf (monitor-used self) t))
    result))


(defmethod truck-ladle ((self truck) arm-id vessel-id)
  (let* ((arm (obj-list-get-id (arms self) arm-id))
         (result (and arm
                      (arm-ladle arm vessel-id))))
    (if result 
      (setf (monitor-used self) t))
    result))

(defmethod truck-examine ((self truck) arm-id thing-id)
  (let ((arm (obj-list-get-id (arms self) arm-id)))
    (and arm 
         (arm-examine arm thing-id))))


(defmethod truck-toggle ((self truck) arm-id thing-id)
  (let ((arm (obj-list-get-id (arms self) arm-id)))
    (and arm 
         (arm-toggle arm thing-id))))

(defmethod truck-move ((self truck))
  (road-traverse (environment self)
                 self
                 (truck-heading self)
                 (monitoring self)))

(defmethod truck-travel ((self truck) link distance)
  (let ((fuel-drag (fuel-drag link))
	(speed-drag (speed-drag link))
	(new-bumpiness (bumpiness link)))
    (cond 
      ((or (eq (truck-speed self) 'stop)
	   (empty? (fuel-bay self))
	   (neq (truck-status self) 'happy)
	   (null (truck-tires self))
	   (some #'(lambda (x)
		     (not (or (arm-folded? x)
			      (held-by-environment? x (id self)))))
		 (arms self))
	   (and (eq (truck-speed self) 'fast)
		(not (every #'arm-folded?
			    (arms self)))))
       (post-hardware-error 'truck-cant-move)
       (values 0 0))
      (t (setf (bumpiness self)
	       (enforce #'bumpiness? new-bumpiness))
	 (let* ((fuel-mpg (percent-adjust fuel-drag (base-mpg self)))
		(fuel-on-hand (space-full (fuel-bay self)))
		(fuel-miles (+ (miles-left self) (random-adjust 5 distance)))
		(fuel-needed (* 1.0 (/ fuel-miles fuel-mpg)))
		(miles-moved (cond ((<= fuel-needed fuel-on-hand)
				    (pour-out (fuel-bay self) fuel-needed)
				    distance)
				   (else (pour-out (fuel-bay self) fuel-on-hand)
					 (min distance 
					      (* fuel-on-hand fuel-mpg)))))
		(travel-time (cond ((> miles-moved 0)
				    (/ (random-adjust 5 (* miles-moved 60))
				       (percent-adjust speed-drag
						       (base-mph self))))
				   (t 0)))) 
	   (setf (miles-left self)
		 (- fuel-miles (* fuel-needed fuel-mpg)))
	   (setf (odometer self)
		 (+ (odometer self) miles-moved))
	   (if (> miles-moved 0)
               (scramble-ids))
	   (values (truncate miles-moved) travel-time))))))

(defmethod truck-tires ((self truck))
  (car (get-if-holds-class (tire-bay self) 'tires)))                                

(defmethod (setf truck-status) :after (new-status (self truck))
  (declare (ignore new-status))
  (if (neq (truck-status self) 'happy)
    (post-hardware-error 'truck-mishap))
  (if (displayer self) 
	  (dm-set (displayer self) 'status-gauge (truck-status self))))
  
(defmethod (setf truck-status) :before (new-status (self truck))
  (enforce #'status? new-status))

(defmethod (setf truck-heading) :after (direction (self truck))
  (declare (ignore direction))
  (if (displayer self) 
	  (dm-set (displayer self) 'heading-gauge (truck-heading self))))

(defmethod (setf truck-heading) :before (direction (self truck))
  (enforce #'map-direction? direction))

(Defmethod truck-fuel ((self truck)) 
  (space-full (fuel-bay self)))

(defmethod (setf truck-speed) :before (new-value (self truck))
  (enforce #'speed? new-value))

(defmethod (setf truck-speed) :after (new-value (self truck))
  (declare (ignore new-value))
  (if (displayer self) 
	  (dm-set (displayer self) 'speed-gauge (truck-speed self)))
  (case (truck-speed self)
	((stop)
	 (setf (base-mpg self) 0)
	 (setf (base-mph self) 0))
	((slow)
	 (setf (base-mpg self)
		   (percent-adjust 150 (base-fuel-consumption self)))
	 (setf (base-mph self)
		   (percent-adjust 50 (base-speed self))))
	((medium)
	 (setf (base-mpg self) (base-fuel-consumption self))
	 (setf (base-mph self) (base-speed self)))
	((fast)
	 (setf (base-mpg self)
		   (percent-adjust 50 (base-fuel-consumption self)))
	 (setf (base-mph self)
		   (percent-adjust 135 (base-speed self)))))
  (if (class? (truck-tires self) 'mud-tires)
	  (setf (base-mph self)
			(percent-adjust 80 (base-mph self)))))


;;; ** Operations needed for sensors

(defmethod eye-examine ((self truck) pos-whatsit thing-id)
  (let* ((place (if (eq pos-whatsit 'external)
                  (environment self)
                  (or (obj-list-get-id (bays self) pos-whatsit)
                      (obj-list-get-id (arms self) pos-whatsit))))             
         (thing (obj-list-get-id (holdings place) thing-id)))                       
    (setf (monitor-used self) t)
    (cond ((null thing)
           (post-hardware-error 'eye-cant-find))
          (t (post-sensor-datum 'object-seen
                                pos-whatsit
                                thing-id
                                (kind-id thing)
                                nil)
             (examine thing)))))

(defmethod eye-scan ((self truck) pos-whatsit)
  (let ((place (if (eq pos-whatsit 'external)
		   (environment self)
		   (or (obj-list-get-id (bays self) pos-whatsit)
		       (obj-list-get-id (arms self) pos-whatsit)))))          
    (setf (monitor-used self) t)
    (cond (place 
           (do ((things-seen (holdings place)
                             (cdr things-seen)))
               ((null things-seen) t)
             (cond ((neq (unique-id (car things-seen)) (id self))
                    (post-sensor-datum 'object-seen
                                       pos-whatsit
                                       (unique-id (car things-seen))
                                       (kind-id (car things-seen))
                                       nil)
                    (cond ((not (position (car things-seen)
                                          (known-about self)))
                           (vset (known-about self) 
                                 (known-counter self) 
                                 (car things-seen))
                           (setf (known-counter self) 
                                 (+ (known-counter self) 1)))))))
           (if (eq place (environment self))
	       (let ((roads-here (map-roads place)))
		 (mapc #'(lambda (x)
			   (if (svref roads-here
				      (map-direction-index x))
			       (post-sensor-datum
				'road-seen
				x
				(unique-id 
				 (car
				  (svref roads-here 
					 (map-direction-index x))))
				nil
				nil)))
		       '(n s e w ne nw se sw))))
	   t)
          (t nil))))

(defmethod eye-monitor ((self truck) class)
  (cond ((not (member class (monitoring self) :test #'eq))
         (setf (monitoring self)
               (cons class (monitoring self)))
         (do ((places 
               (cons (environment self) (holdings self))
               (cdr places)))
             ((null places) t)
           (do ((x (holdings (car places))
                   (cdr x)))
               ((null x) t)
             (cond ((and (class? (car x) class)
                         (not (position (car x) (known-about self))))
                    (vset (known-about self)
                          (known-counter self)
                          (car x))
                    (setf (known-counter self)
                          (+ (known-counter self) 1))))))))
  t)

(defmethod eye-unmonitor ((self truck) class)
  (setf (monitoring self) (delete class (monitoring self)))
  t)

;;; ** Operations needed for neatness

(defmethod (setf displayer) :after (new-displayer (self truck))
  (declare (ignore new-displayer))
   (mapc #'(lambda (thing) (setf (displayer thing) (displayer self)))
         (append (bays self) (arms self)))
   (setf (displayer (tire-bay self))   (displayer self))
   (setf (displayer (fuel-bay self))   (displayer self))
   (setf (displayer (weapon-bay self)) (displayer self))
   (if (class? (environment self) 'map-node)
	   (setf (displayer (environment self)) (displayer self)))
   (cond ((displayer self)
          (dm-set (displayer self) 'current-location (environment self))
          (dm-set (displayer self) 'heading-gauge    (truck-heading self))
          (dm-set (displayer self) 'status-gauge     (truck-status self))
          (dm-set (displayer self) 'speed-gauge      (truck-speed self))
          (dm-set (displayer self) 'fuel-gauge       (truck-fuel self)))))

(defmethod display-info ((self truck))
   (list (mapcar #'unique-id (arms self))
         (mapcar #'unique-id (bays self))
         (capacity (fuel-bay self))))

(defmethod show ((self truck))
   (format t
           "Id:      ~a~&Classes: ~a~&"
           (id self)
           (classes self))
   (format t "arms:")
   (mapc #'(lambda (x)
             (format t "~8T ~a " (unique-id x))
             (show-contents x)
             (format t "~&"))
         (arms self))
   (format t "Bays:")
   (mapc #'(lambda (x)
             (format t "~8T ~a " (unique-id x))
             (show-contents x)
             (format t "~&"))
         (bays self))
   (format t "Status:  ~a~&" (truck-status self))
   (format t
           "Fuel:    ~a (~a) - Tires: ~a - Weapon: ~a~&"
           (space-full (fuel-bay self))
           (capacity (fuel-bay self))
           (kind-id (car (get-if-holds-class (tire-bay self) 'tires)))
           (kind-id (car (get-if-holds-class (weapon-bay self) 'weapon))))
   (format t
           "Speed:   ~a : ~a (~a mph) (~a mpg)~&"
           (truck-heading self)
           (truck-speed self)
           (base-mph self)
           (base-mpg self))
   (format t "Held-By: ~a~&" (environment self))
   (format t "Let-See: ~a~&" (visibles self))
   (format t "Monitor: ~a~&" (monitoring self))
   (format t
           "Know:    ~a~&"
           (do ((x 0 (+ x 1))
                (names '()))
               ((= x (known-counter self)) names)
             (setf names
                   (cons (unique-id (svref (known-about self)
                                          x))
                         names))))
   t)


;*********************************************************************
;  Information needed by the displayer

(defun truck-arm-names (truck)
  (mapcar #'id (arms truck)))

(defun truck-bay-names (truck)
  (mapcar #'id (bays truck)))

(defun truck-fuel-capacity (truck)
  (capacity (fuel-bay truck)))


