;*************************************************************************
;  MAP-NODES.LISP

;  Note that "location" and "node" may be used interchangably

(defun make-location (kind map-sector display-info)
  (let ((the-new-node 
         (make-instance 'map-node
                        :id (make-unique-id 'node)
                        :kind kind
                        :my-map-sector map-sector
                        :display-info display-info)))
    (mapc #'(lambda (x) (put-in the-new-node x))
          (get-global-map-node-handlers))
    (if (symbolp kind)
      (setf (kind the-new-node) (list kind)))
    the-new-node))                                     

; --------------------------------------------------------------------
; * MAP-NODES for simulating locations *
; --------------------------------------
;
; Things defined in this file:
;
; Object builders:
;  (SI:MAKE-MAP-NODE-GENERIC id kind map-sector)
;
; This file makes reference to things in LINKS.T
;                                        ENEMY_UNITS.T
;                                        OPERATIONS.T
;                                        LOWLEVEL.T
;                                        GLOBAL_DEFS.T
;
; NOTE: Map nodes are have an extra property that objects of class
;        SPECIAL-HANDLER can be added to them.  Such objects are
;        updated like contained objects but do not appear in the
;        map nodes environment; they are essentially invisible.
;
;       In addition, there is a pseudo-global list of handler builder
;        functions.  Each builder is called to get an initial handler
;        when a map node is first built.  Handler builders are added
;        to the list using:
;          (SI:ADD-GLOBAL-MAP-NODE-HANDLER tag handler-buiilder)
;        This is supposed to be a convenience to make building worlds
;        easier.  See ENEMY.T for an example of this stuff in use.

(let ((map-node-handler-pairs '()))
  (labels ((match? (x y)
                   (or (and (consp x) (eq (car x) y))
                       (and (consp y) (eq (car y) x)))))

    (defun si.add-global-map-node-handler (tag handler)
      (setf map-node-handler-pairs
            (cons (cons tag handler)
                  (delete tag
                          map-node-handler-pairs
                          :test
                          #'match?))))

    (defun get-global-map-node-handlers ()
      (mapcar #'(lambda (x) (funcall (cdr x)))
              map-node-handler-pairs))))

; --------------------------------------------------------------------
; * Generic MAP-NODE constructor *
; --------------------------------
;
; Map nodes are locations in the world where things can happen.  Map
;  links tie map nodes together like roads.  A map is constructed by
;  building map nodes and map links and connecting them together.  A
;  truck can only exist in a map node.  When a truck is supposed to
;  move, the map node that the truck is in is asked to move it.  The
;  map node figures out which road(link) the truck is trying to move
;  down, and then passes it to the link to actually be moved.  If the
;  truck makes it all the way down the road it ends up in the map
;  node at the other end.  If the truck runs into some trouble along
;  the way, a pseudo-node (map-link-subnode) is constructed and the
;  truck is put in that by the link.  A pseudo-node acts exactly a 
;  real node somewhere along the road.
;
; Operations on MAP-NODE type objects:
;  (MAP-CONNECT self connection-dir link direction)
;  (MAP-ROADS self)
;  (ROAD-TRAVERSE self truck direction pattern) => [distance time]


(defun map-link-list-printer (links)
  (format t
          "N-~a S-~a E-~a W-~a NE-~a NW-~a SE-~a SW-~a"
          (car (svref links (map-direction-index 'n)))
          (car (svref links (map-direction-index 's)))
          (car (svref links (map-direction-index 'e)))
          (car (svref links (map-direction-index 'w)))
          (car (svref links (map-direction-index 'ne)))
          (car (svref links (map-direction-index 'nw)))
          (car (svref links (map-direction-index 'se)))
          (car (svref links (map-direction-index 'sw)))))

; --------------------------------------------------------------------
; * Generic MAP-LINK-SUBNODE constructor *
; ----------------------------------------
;
; Map link subnodes are things that act like map nodes but represent
;  places along a link (ie. places of interest along a road).  A new
;  subnode is created when a truck tries to traverse a link, doesn't
;  make it and ends up where no subnode already exists.  When the last
;  truck in a subnode moves out and no things are left behind, then
;  the subnode is recycled.  If a subnode is left with things inside
;  it, then the subnode stays around for trucks to find later.
;
; Map link subnodes exist in a pool as they will often be created and
;  then tossed aside.

(defvar *si.map-link-subnode-pool*
  (make-pool "Map-Link-Subnode-Pool"
             #'(lambda ()
                 (make-new-map-link-subnode 
                  (make-unique-id 'subnode)
                  '(unconnected)))))

(defun get-map-link-subnode ()
  (obtain-from-pool *si.map-link-subnode-pool*))

(defun make-new-map-link-subnode (id kind)
  (let ((the-new-node 
         (make-instance 'map-link-subnode
                        :id id
                        :kind kind))) 
    (mapc #'(lambda (x) (put-in the-new-node x))
          (get-global-map-node-handlers))
    (if (symbolp kind)
      (setf (kind the-new-node) (list kind)))
    the-new-node))

