;*************************************************************************
;  MAP-LINKS.LISP

; --------------------------------------------------------------------
; * Generic MAP-LINK constructor *
; --------------------------------
;
; Map links are used to model the roads between nodes.  A truck
;  usually travels down a link all the way without stopping.  Some
;  times a truck will have some type of mishap and will end up part
;  way down a road.  When that happens a MAP-LINK-SUBNODE is created
;  and the truck is put in that.  A truck is thus never found in a
;  link object.
;
; (SI:MAKE-MAP-LINK-GENRIC id kind road-length road-forward-dir
;                          fuel-drag speed-drag bumpiness fragility
;                          map-sector)
;  id:   The unique identifier for this link. 
;  kind: The major type and other attributes of this link.
;  road-length: The length of this link in miles.
;  road-forward-dir: The major direction of this road from start to
;        finish.
;  fuel-drag: The percentage of base fuel use that this road forces.
;        For example, setting this to 80 gives 80% of usual fuel use.
;  speed-drag: Like fuel-drag only for base speed.
;  bumpiness: The bumpiness of this road.
;  fragility: The total bigness that this road can support.  A truck
;        with bigger bigness will become DEAD.
;  map-sector: The world map sector that this road is in.  Used for
;        establishing enemy activity and weather.  
;************************************************************************

(defun make-road (kind start sdir finish fdir length 
                       dir  fragility sector display-info)
  (let ((kind-id (if (symbolp kind) kind (car kind)))
        (attributes (if (symbolp kind) nil (cdr kind))))
    (cond 
      ((or (not (road-type? kind-id))
	   (not (every #'road-attribute? attributes)))
       (cerror "To continue" "A road has an undefined attribute: ~a." kind))))
  (let* ((kind-id (if (symbolp kind) kind (car kind)))
	 (speed-drag (road-kind-speed-drag kind-id))
         (fuel-drag (road-kind-fuel-drag kind-id))
	 (bumpiness (road-kind-bumpiness kind-id))
         (new-road (si.make-map-link-generic (make-unique-id 'road)
		      kind
		      length
		      dir
		      fuel-drag
		      speed-drag
		      bumpiness
		      fragility
		      sector
		      display-info)))
    (map-connect-start new-road start sdir)
    (map-connect-finish new-road finish fdir)
    new-road))

;*****************************************************************

(defun road-kind-speed-drag (kind-id)
  (cdr (assq kind-id  '((standard-road . 100)
			(express-road . 120)
			(soft-road . 75)
			(winding-road . 90)
			(muddy-road . 100)
			(bumpy-road . 70)
			(bridge . 100)))))

(defun road-kind-fuel-drag (kind-id)
  (cdr (assq kind-id '((standard-road . 100)
		       (express-road . 100)
		       (soft-road . 80)
		       (winding-road . 100)
		       (muddy-road . 80)
		       (bumpy-road . 70)
		       (bridge . 100)))))

(defun road-kind-bumpiness (kind-id)
  (cdr (assq kind-id '((standard-road . low)
		       (express-road . low)
		       (soft-road . low)
		       (winding-road . low)
		       (muddy-road . high)
		       (bumpy-road . high)
		       (bridge . low)))))

;**************************************************************************
   
(defun si.make-map-link-generic (id kind
                                    road-length
                                    road-direction
                                    fuel-drag
                                    speed-drag
                                    bumpiness
                                    fragility
                                    map-sector
                                    display-stuff)
       (let ((new-thing (make-instance 'map-link
				       :id id
				       :kind kind
				       :road-length road-length
				       :road-direction road-direction
				       :fuel-drag fuel-drag
				       :speed-drag speed-drag
				       :bumpiness bumpiness
				       :fragility fragility
				       :my-map-sector map-sector
				       :display-info display-stuff)))
	 (if (symbolp kind)
	     (setf (kind new-thing) (list kind)))
	 (setf (road-backward new-thing)
	       (map-direction-opposite (road-direction new-thing)))
	 (setf (sub-nodes new-thing) 
	       (fill (make-array (+ 1 (road-length new-thing))) '()))
	 new-thing))

  
; --------------------------------------------------------------------
; * Stuff to help move a truck around *
; -------------------------------------
;
; When a truck is to move down a road, many things happen in an arcane
;  order.  First, the command is passed to the NODE where the truck is
;  and the node figures out which road is to be travelled down.  The
;  node then passes the command to the LINK corresponding to the road
;  that the truck wants to move down.  The link calls the procedure
;  (MOVE-TRUCK ...) which decided the maximum distance the truck can
;  move and asks the TRUCK to move that far if it can.  The truck
;  moves as far as it can without running out of gas and returns the
;  distance it travelled to  (MOVE-TRUCK ...).  (MOVE-TRUCK ...) does
;  everything necessary to make sure that the truck is somewhere and
;  returns how far it went back up through all callers.
;
; This function returns multiple values, the distance moved and the
;  time taken to move that far.

(defun move-truck (link truck direction pattern)
  (let* ((road-vector (sub-nodes link))
	 (road-direction (road-direction link))
	 (start-node (environment truck))
	 (end-node nil)
         (travel-dir (if (eq road-direction direction) 1 -1))
         (road-length (road-length link))
         (start-mile nil)
         (end-mile nil)
         (miles-to-move nil)
         (status-at-end 'happy)
         (miles-moved 0)
         (travel-time 0))

    ;; Figure out the intended start and end locations 
    (setf start-mile (compute-start-mile link start-node))
    (setf end-mile 
	  (compute-end-mile pattern start-mile travel-dir
			    road-vector road-length))
    (setf miles-to-move 
	  (abs (- start-mile end-mile)))

    ;; Get distance to mishap if one will occur
    (multiple-value-setq (miles-to-move status-at-end) 
      (compute-end-status link truck miles-to-move))

    ;; Try it and see where the truck actually ends up
    (multiple-value-setq (miles-moved travel-time)
      (truck-travel truck link miles-to-move))

    ;; Set end point
    (setf end-mile (+ start-mile (* travel-dir miles-moved)))

    ;; Make sure the truck is now in the right place
    (setf end-node (compute-end-node link start-mile end-mile start-node))
    (take-out start-node truck)
    (put-in end-node truck)

    ;; Put the truck in the right state
    (adjust-truck-state miles-to-move miles-moved status-at-end end-node truck)

    ;; Recycle old truck position if empty
    (if (and (class? start-node 'map-link-subnode)
	     (null (holdings start-node)))
	(recycle start-node))
    (values miles-moved travel-time)))

;**********************************************

(defun compute-start-mile (link start-node)
  (let ((start-node-id (unique-id start-node)))
	(cond 
	 ((eq start-node-id (unique-id (start-node link)))
	  0)
	 ((eq start-node-id (unique-id (finish-node link)))
	  (road-length link))
	 ((class? start-node 'map-link-subnode)
	  (position start-node (sub-nodes link)))
	 (t '()))))

(defun compute-end-mile (pattern start-mile travel-dir road-vector road-length)
  (cond 
   (pattern 
	(find-mile-matching-pattern pattern start-mile travel-dir road-vector))
   ((< travel-dir 0) 	0)
   (t                   road-length)))

(defun compute-end-status (link truck miles-to-move)
  (cond 
    ((and (fragility link)  
	  (>= (gross-bigness truck) (fragility link)))
     (values 1 'dead))
    (t (multiple-value-bind (distance mishap)
	   (distance-to-mishap link truck)
         (cond 
	   ((and distance (< distance miles-to-move))
	    (values distance mishap))
	   (t (values miles-to-move 'happy)))))))

(defun compute-end-node (link start-mile end-mile start-node)
  (cond 
    ((= end-mile start-mile) start-node)
    ((= end-mile 0) (start-node link))
    ((>= end-mile (road-length link)) (finish-node link))
    (t (let ((road-vector (sub-nodes link)))
	 (cond 
	   ((class? (svref road-vector end-mile) 'map-link-subnode)
	    (svref road-vector end-mile))
	   (t (let ((new-node (get-map-link-subnode)))
		(setf (environment new-node) link)
		(setf (svref road-vector end-mile) new-node)
		new-node)))))))

(defun adjust-truck-state (miles-to-move miles-moved status-at-end end-node truck)
  (cond 
   ((and (> miles-to-move 0) (= miles-to-move miles-moved))
	(cond 
	 ((eq status-at-end 'enemy-unit)
	  (do ((count (random 1 *enemies-at-mishap*) (- count 1)))
		  ((= count 0) t)
		  (put-in end-node (si.make-enemy-unit))))
	 (t (setf (truck-status truck) status-at-end))))))

; --------------------------------------------------------------------
; * How far can a truck move without crashing?
; --------------------------------------------
;
; This function figures out how far a truck can move down this type of
;  road before it encounters some type of mishap.  Defined mishaps
;  are:
;    BROKEN, ROLLED, STUCK, ENEMY-UNIT, DEAD, CAPTURED
;
; This function returns multiple values => [distance-to-mishap mishap]
;
; This function is ugly!!!

(defun distance-to-mishap (road truck)
  (let ((distance-to-broken most-positive-fixnum)
        (distance-to-dead most-positive-fixnum)
        (distance-to-rolled most-positive-fixnum)
        (distance-to-stuck most-positive-fixnum)
        (distance-to-enemy most-positive-fixnum)
        (daytime (day-time road))
        (speed (truck-speed truck))
        (weather (weather road))
        (enemy-activity (enemy-activity road))
        (not-mud-tires (not (class? (truck-tires truck) 'mud-tires))))
    (cond ((class? road 'bumpy)
           (cond ((eq daytime 'day)
                  (setf distance-to-broken
                        (min distance-to-broken
                             (random-distance 
                              (cond ((eq speed 'slow)
                                     *bumpy-day-broken-slow%*)
                                    ((eq speed 'medium)
                                     *bumpy-day-broken-medium%*)
                                    ((eq speed 'fast)
                                     *bumpy-day-broken-fast%*)))))
                  (setf distance-to-dead
                        (min distance-to-dead
                             (random-distance 
                              (cond ((eq speed 'slow)
                                     *bumpy-day-dead-slow%*)
                                    ((eq speed 'medium)
                                     *bumpy-day-dead-medium%*)
                                    ((eq speed 'fast)
                                     *bumpy-day-dead-fast%*))))))
                 ((eq daytime 'night)
                  (setf distance-to-broken
                        (min distance-to-broken
                             (random-distance
                              (cond ((eq speed 'slow)
                                     *bumpy-night-broken-slow%*)
                                    ((eq speed 'medium)
                                     *bumpy-night-broken-medium%*)
                                    ((eq speed 'fast)
                                     *bumpy-night-broken-fast%*)))))
                  (setf distance-to-dead
                        (min distance-to-dead
                             (random-distance 
                              (cond ((eq speed 'slow)
                                     *bumpy-night-dead-slow%*)
                                    ((eq speed 'medium)
                                     *bumpy-night-dead-medium%*)
                                    ((eq speed 'fast)
                                     *bumpy-night-dead-fast%*)))))))))
    (cond ((class? road 'windy)
           (cond ((eq daytime 'day)
                  (setf distance-to-rolled
                        (min distance-to-rolled
                             (random-distance 
                              (cond ((eq speed 'slow)
                                     *windy-day-rolled-slow%*)
                                    ((eq speed 'medium)
                                     *windy-day-rolled-medium%*)
                                    ((eq speed 'fast)
                                     *windy-day-rolled-slow%*))))))
                 ((eq daytime 'night)
                  (setf distance-to-rolled
                        (min distance-to-rolled
                             (random-distance
                              (cond ((eq speed 'slow)
                                     *windy-night-rolled-slow%*)
                                    ((eq speed 'medium)
                                     *windy-night-rolled-medium%*)
                                    ((eq speed 'fast)
                                     *windy-night-rolled-fast%*)))))))))
    (cond ((class? road 'muddy)
           (cond ((and not-mud-tires (eq weather 'rainy))
                  (setf distance-to-stuck
                        (min distance-to-stuck
                             (random-distance
                              (cond ((eq speed 'slow)
                                     *muddy-stuck-slow%*)
                                    ((eq speed 'medium)
                                     *muddy-stuck-medium%*)
                                    ((eq speed 'fast)
                                     *muddy-stuck-fast%*)))))))))
    (cond ((> enemy-activity 0)
           (cond ((eq daytime 'day)
                  (setf distance-to-enemy
                        (min distance-to-enemy
                             (/ (random-distance (fixnum-min 100
                                                             (cond ((eq speed 'slow)
                                                                    (percent-adjust *enemy-day-slow-adjust* enemy-activity))
                                                                   ((eq speed 'medium)
                                                                    (percent-adjust *enemy-day-medium-adjust* enemy-activity))
                                                                   ((eq speed 'fast)
                                                                    (percent-adjust *enemy-day-fast-adjust* enemy-activity)))))
                                2)))))))
    (let ((distance-to-mishap (min       
                               distance-to-broken
                               distance-to-rolled
                               distance-to-dead
                               distance-to-stuck 
                               distance-to-enemy)))
      (cond ((= distance-to-mishap distance-to-stuck)
             (values distance-to-mishap 'stuck))
            ((= distance-to-mishap distance-to-broken)
             (values distance-to-mishap 'broken))
            ((= distance-to-mishap distance-to-rolled)
             (values distance-to-mishap 'rolled))
            ((= distance-to-mishap distance-to-enemy)
             (values distance-to-mishap 'enemy-unit))
            ((= distance-to-mishap distance-to-dead)
             (values distance-to-mishap 'dead))))))

; --------------------------------------------------------------------
; * Finding a MAP-LINK-SUBNODE holding some things
; ------------------------------------------------
;
; Occasionally, a truck will want to travel to a subnode along a link
;  where it has left something.  Such a node is specified as a list
;  of objects to be found there.  These functions search through a
;  road-vector looking for a subnode that contains the appropriate
;  objects.

(defun find-mile-matching-pattern (pattern start dir road-vector)
  (let ((road-length (length road-vector)))
    (cond ((and start dir (<= 0 start) (<= start road-length))
           (cond ((< dir 0)
                  (find-matching-mile-searcher pattern
                                               road-vector
                                               (+ start dir)
                                               0
                                               dir))
                 (t (find-matching-mile-searcher pattern
                                                 road-vector
                                                 (+ start dir)
                                                 road-length
                                                 dir))))
          (t '()))))

(defun find-matching-mile-searcher (pattern road current finish dir)
  (cond ((= current finish) current)
        ((let* ((this-mile (svref road current))
                (stuff-here (if (class? this-mile 'map-link-subnode)
                                (holdings this-mile)
                                this-mile)))
           (and stuff-here (match-pattern-to-stuff pattern stuff-here)))
         current)
        (t (find-matching-mile-searcher pattern road (+ current dir)
					finish dir))))

(defun match-pattern-to-stuff (pattern stuff)
  (cond ((null pattern) '())
        ((obj-list-holds-class? stuff (car pattern)) t)
        (t (match-pattern-to-stuff (cdr pattern) stuff))))

