(comment (herald enemy))

; --------------------------------------------------------------------
; * ENEMY-UNITS for simulating bad guys *
; ---------------------------------------
;
; Things defined in this file:
;  (UPDATE-ENEMY-UNITS-HERE self number time last-number last-time)
;  The enemy unit object pool
;
; Object builders:
;  
; This file makes reference to things in
;                                        OPERATIONS.T
;                                        LOWLEVEL.T
;                                        GLOBAL_DEFS.T
;
; ------------------------------------------------------------------------------
; * Definitions for ENEMY-TROOPS *
; --------------------------------
;
; Enemy troops are treated in an unusual way by the simulator; they
;  are like weather.  The toplevel world object keeps track of the
;  (ENEMY-ACTIVITY ...) in each sector and whenever a truck is
;  present at a node enemy units might show up.  Similarly, whenever
;  a truck is moving down a link it may run into enemy units and
;  be forced to stop (treated like a mishap).
;
; Each time that a node updates itself, it can change the number of
;  enemy units on hand by calling the (UPDATE-ENEMY-UNITS-HERE ...)
;  procedure below.  Enemy units left over from before the truck
;  arrived are discounted and new units may be added.
;
; When enemy units are present at a node they try and CAPTURE any
;  trucks that are there.  This is accomplished through their
;  (UPDATE ...) function.
;
; Enemy units are stored in a pool so that they can be recycled 
;  easily.  The pool constructors are defined below.


(defclass enemy-unit
  (thingoid)
  ((last-update-time :accessor last-update-time
                     :initform 0)
   (last-update-number :accessor last-update-number
                     :initform 0)
   ))


(defmethod update ((self enemy-unit) number time)
  (cond ((and (= number (+ (last-update-number self) 1))
              (random-chance-per? *enemy-effectiveness*
                                  (- time (last-update-time self))
                                  60))
         (let ((target (some #'(lambda (x)
                                 (cond ((and (class? x 'truck)
                                             (neq (truck-status x) 'captured)
                                             (neq (truck-status x) 'dead))
                                        x)
                                       (t nil)))
                             (holdings (environment self)))))
           (cond (target 
                  (setf (truck-status target) 'captured))))))
  (setf (last-update-time self) time)
  (setf (last-update-number self) number))

;;; this appears to be right for (Setf (environment self) new-holder)

(Defmethod (setf environment) :after (new-holder (self enemy-unit)) 
  (setf (last-update-time self) 0)
  (setf (last-update-number self) 0))

(defclass enemy-unit-handler
  (special-handler)
  ((last-update-number :accessor last-update-number
                       :initform 0)
   (last-update-time :accessor last-update-time
                     :initform 0)
   ))
                                          
(Defmethod update ((self enemy-unit-handler) number time)
  (let* ((location (environment self))
         (activity (enemy-activity location))
         (enemy-on-hand (get-if-holds-class location 'enemy-unit)))           
    ; ** Take out old ones first
    (cond ((> number (+ (last-update-number self) 1))
           (mapc #'(lambda (x)
                     (cond ((random-chance-per? (- 100 activity)
                                                (- time (last-update-time self))
                                                60)
                            (take-out location x)
                            (recycle x))))
                 enemy-on-hand)))
    ; ** Put in new ones
    (cond ((random-chance-per? activity
                               (- time (last-update-time self))
                               60)
           (do ((count (random 1 *maximum-new-enemies*)
                       (- count 1)))
               ((= count 0) t)
             (put-in location (si.make-enemy-unit))))))
  (setf (last-update-number self) number)
  (setf (last-update-time self) time))

