;***************************************************************************
; ARM.LISP

(defun make-arm-disp (dock-x-in dock-y-in corridor-list-in bay-names-in)
  (let ((new-disp (arm.setup-state dock-x-in
                                   dock-y-in
                                   corridor-list-in
                                   bay-names-in)))
    (disp.fill-rectangle (dock-window new-disp))
    new-disp))

;*****************************************************************************

(defun arm.setup-state (dock-x-in dock-y-in corridor-list-in bay-names-in)
  (let ((my-state (make-instance 'arm-disp
                                 :dock-x dock-x-in
                                 :dock-y dock-y-in
                                 :corridor-list corridor-list-in
                                 :bay-names bay-names-in)))
    (setf (dock-window my-state)
          (make-window (- dock-x-in 2) (- dock-y-in 2) 4 4))
    (setf (right-cor-x my-state) (nth 0 corridor-list-in))
    (setf (gauge-cor-y my-state) (nth 1 corridor-list-in))
    (setf (bay1-cor-y my-state) (nth 2 corridor-list-in))
    (setf (bay2-cor-y my-state) (nth 3 corridor-list-in))
    my-state))

;**************************************************************************

(defun arm.add-gauge-cache (sym loc state)
  (let ((final-x (horizontal-midpoint loc))
        (final-y (+ (y-coord loc) (+ (height loc) 4))))
    (push (cons sym
                (arm.build-lines (list (right-cor-x state)
                                       (gauge-cor-y state)
                                       final-x
                                       final-y)
                                 state))
          (cached-paths state))))

(defun arm.add-gripper-cache (sym loc state)
  (let ((x (horizontal-midpoint loc))
        (y (+ (y-coord loc) (+ (height loc) 2)))
        (l (+ (width loc) 4)))
    (push (cons sym (list x y l 'up)) (cached-grippers state))))

(defun arm.build-caches (state)
  (let ((final-loc (truck-display-subwindow (the-truck state) 'fuel-gauge)))
    (arm.add-gauge-cache 'fuel-gauge final-loc state)
    (arm.add-gauge-cache 'fuel-bay final-loc state)
    (arm.add-gripper-cache 'fuel-gauge final-loc state)
    (arm.add-gripper-cache 'fuel-bay final-loc state)
    (setf final-loc (next-empty-space (the-truck state) 'weapon-bay))
    (arm.add-gauge-cache 'weapon-bay final-loc state)
    (arm.add-gripper-cache 'weapon-bay final-loc state)
    (setf final-loc (next-empty-space (the-truck state) 'tire-bay))
    (arm.add-gauge-cache 'tire-bay final-loc state)
    (arm.add-gripper-cache 'tire-bay final-loc state)
    (push (cons 'folded
                (list (make-position (dock-x state)
                                     (dock-y state))
                      (make-position (+ (dock-x state) 10)
                                     (dock-y state))))
          (cached-paths state))
    (push (cons 'folded
                (list (+ (dock-x state) 10)
                      (dock-y state)
                      (+ *small-icon-width* 4)
                      'right))
          (cached-grippers state))))

(defun arm.build-lines (point-list state)
  (arm.build-lines-aux (cons (dock-x state)
                             (cons (dock-y state) point-list))
                       'horiz))

(defun arm.build-lines-aux (point-list direction)
  (cond ((null (cdr point-list)) nil)
        ((eq direction 'horiz)
         (cons (make-position (car point-list) (cadr point-list))
               (arm.build-lines-aux (cdr point-list) 'vert)))
        (t (cons (make-position (cadr point-list) (car point-list))
                 (arm.build-lines-aux (cdr point-list) 'horiz)))))

(defun arm.build-path (loc state)
  (cond ((cdr (assoc loc (cached-paths state) :test #'eq)))
        ((symbolp loc)
         (cerror "To continue" "Can't move to this symbol: ~a" loc))
        (t (let ((loc-sym (car loc))
                 (x (x-coord (cadr loc)))
                 (y (y-coord (cadr loc)))
                 (w (width (cadr loc)))
                 (h (height (cadr loc))))
             (cond ((eq loc-sym 'current-location)
                    (setf x (- x 4))
                    (setf y (+ y (truncate h 2))))
                   (t (setf x (+ x (truncate w 2))) (setf y (+ y (+ h 4)))))
             (cond ((eq loc-sym 'current-location)
                    (arm.build-lines (list (right-cor-x state) y x)
                                     state))
                   ((or (eq loc-sym 'tire-bay)
                        (eq loc-sym 'weapon-bay))
                    (arm.build-lines (list (right-cor-x state)
                                           (gauge-cor-y state)
                                           x
                                           y)
                                     state))
                   ((eq loc-sym (car (bay-names state)))
                    (arm.build-lines (list (right-cor-x state)
                                           (bay1-cor-y state)
                                           x
                                           y)
                                     state))
                   ((eq loc-sym (cadr (bay-names state)))
                    (arm.build-lines (list (right-cor-x state)
                                           (bay2-cor-y state)
                                           x
                                           y)
                                     state))
                   (t (cerror "To continue"
                              "Can't move to object at this location: ~a"
                              loc-sym)))))))

(defun arm.draw-path-lines (loc state)
  (let ((lines (arm.build-path loc state)))
    (setf (path-lines state) lines)
    (disp.draw-connected-lines lines)))

(defun arm.build-gripper-lines (x y length dir disp state)
  (declare (ignore state))
  (cond ((eq disp 'outside)
         (setf length (truncate length 2))
         (cond ((member dir '(up down) :test #'eq)
                (let* ((left-end-x (- x length))
                       (right-end-x (+ x length))
                       (left-pt (make-position left-end-x y))
                       (right-pt (make-position right-end-x y)))
                  (labels ((op (a b) (if (eq dir 'up) 
                                       (- a b)
                                       (+ a b))))
                    (list left-pt
                          right-pt
                          left-pt
                          (make-position left-end-x (op y 4))
                          right-pt
                          (make-position right-end-x (op y 4))))))
               ((member dir '(left right) :test #'eq)
                (let* ((top-y (- y length))
                       (bottom-y (+ y length))
                       (top-pt (make-position x top-y))
                       (bottom-pt (make-position x bottom-y)))
                  (labels ((op (a b) (if (eq dir 'left) 
                                       (- a b)
                                       (+ a b))))
                    (list top-pt
                          bottom-pt
                          top-pt
                          (make-position (op x 4) top-y)
                          bottom-pt
                          (make-position (op x 4) bottom-y)))))))
        ((eq disp 'inside)
         (setf length (truncate length 4))
         (cond ((eq dir 'up)
                (list (make-position x y)
                      (make-position (- x length) (+ y 4))
                      (make-position (- x length) (+ y 4))
                      (make-position (+ x length) (+ y 4))
                      (make-position (+ x length) (+ y 4))
                      (make-position x y)))
               ((eq dir 'right)
                (list (make-position x y)
                      (make-position (- x 4) (- y length))
                      (make-position (- x 4) (- y length))
                      (make-position (- x 4) (+ y length))
                      (make-position (- x 4) (+ y length))
                      (make-position x y)))))))

(defun arm.draw-gripper-lines (loc disp state)
  (t-cond ((or (assoc loc (cached-grippers state) :test #'eq)
               (and (symbolp loc)
                    (assoc (car loc) (cached-grippers state) :test #'eq)))
           =>
           #'(lambda (gr-ass)
               (let ((params (cdr gr-ass)))
                 (setf (gripper-x state) (car params))
                 (setf (gripper-y state) (cadr params))
                 (setf (gripper-length state) (caddr params))
                 (setf (gripper-orient state) (cadddr params))
                 (setf (gripper-disp state) disp))))
          ((eq (car loc) 'current-location)
           (setf (gripper-x state) (- (x-coord (cadr loc)) 4))
           (setf (gripper-y state) (vertical-midpoint (cadr loc)))
           (setf (gripper-length state) (+ (height (cadr loc)) 4))
           (setf (gripper-orient state) 'right)
           (setf (gripper-disp state) disp))
          (t (setf (gripper-x state) (horizontal-midpoint (cadr loc)))
             (setf (gripper-y state)
                   (+ (y-coord (cadr loc)) (+ (height (cadr loc)) 2)))
             (setf (gripper-length state) (+ (width (cadr loc)) 4))
             (setf (gripper-orient state) 'up)
             (setf (gripper-disp state) disp)))
  (setf (gripper-lines state)
        (arm.build-gripper-lines (gripper-x state)
                                 (gripper-y state)
                                 (gripper-length state)
                                 (gripper-orient state)
                                 (gripper-disp state)
                                 state))
  (disp.draw-disconnected-lines (gripper-lines state)))

(defun arm.change-gripper-direction (new-dir state)
  (disp.with-erasure (disp.draw-disconnected-lines (gripper-lines state)))
  (setf (gripper-lines state) nil)
  (if (eq new-dir 'outside) (refresh-arms (the-truck state)))
  (arm.draw-gripper-lines (current-loc state) new-dir state)
  (if (the-truck state) (refresh-arms (the-truck state))))

(defun arm.same-loc? (l1 l2)
  (or (eq l1 l2)
      (and (listp l1)
           (listp l2)
           (eq (car l2) (car l2))
           (= (x-coord (cadr l1)) (x-coord (cadr l2)))
           (= (y-coord (cadr l1)) (y-coord (cadr l2))))))

