/**************************************************************************/
/* xgraphics.c --- after a version by F. J. Jungen              /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/

#define XGRAPHICS_EXTERN

#include "config.h"

#include <stdio.h>		/* Standard IO for errormesages */
#include <X11/Xlib.h>		/* Xlibrary include file */
#include <X11/Xutil.h>		/* Xutilities include file */
#include <signal.h>


#include "xgraphics.h"
#include "matrix.h"
#include "main.h"
#include "datastruct.h"





void init_graphics(int argc,char *argv[])
{
  unsigned long foreground;	/* Foreground color */
  unsigned long background;	/* Background color */
  XSizeHints  hints;		/* Size hints for window manager */
  int bitmap;			/* Bitmap to indicate use of defaults */

  wmhints.flags = (InputHint|StateHint);
  wmhints.input = True;
  wmhints.initial_state = NormalState;
  wmhints.icon_pixmap = 0;
  wmhints.icon_window = 0;
  wmhints.icon_x = 0;
  wmhints.icon_y = 0;
  wmhints.icon_mask = 0;
  wmhints.window_group = 0;

  /*
   * Open a connection to the X server with its associated display
   * by using the value of the environment variable $DISPLAY,
   * that is set by the startX command given to start the current X session.
   */
  if ((display = XOpenDisplay(NULL)) == NULL)
  {
	perror(XDisplayName(NULL));
	exit(1);
  }

  /*
   * Select the defaultscreen of the display,
   * in our case there is only one.
   */
  screen = DefaultScreen (display);

  /*
   * Select colors for foreground and background.
   */
  /*
   * Choose a black-on-white display, optionally.
   * foreground = BlackPixel(display, screen);
   * background = WhitePixel(display, screen);
   */

  /*
   * Default: white on black background.
   */
  background = BlackPixel(display, screen);
  foreground = WhitePixel(display, screen);

  /*
   * Determine the requested position and size of the window
   * - from the argument to the geometry flag,
   *   or from the contents of defaults database in ~/.Xresources
   * - and from the program supplied default position and size.
   *
   *
   * When no geometry argument was specified to the program
   * scan the defaults database.
   */
  if (user_geometry == NULL)
	user_geometry = XGetDefault(display,argv[0],"geometry");

  /*
   * When default values are found, these override program default values.
   * XGeometry parses the geometry specification strings from user and
   * program and returns the window position in the last four parameters:
   * the x, y, width and heigth fields of the size hints structure.
   * The bitmap returned indicates which values
   * are computed from user supplied data.
   */
  bitmap = XGeometry(display, screen, user_geometry, DEFAULT_GEOMETRY,
			BORDER_WIDTH, 1, 1, 0, 0, &(hints.x), &(hints.y),
			&(hints.width), &(hints.height));

  /*
   * We set the window position and window size
   * by giving hints to the window manager.
   * These hints are transferred to the window manager
   * by filling in a so-called `hints structure' with the requested values
   * and by indicating with a mask (the `flag field') which
   * fields of the hints structure contain valid data.
   * We supply  position and size and indicate whether
   * these data are user supplied or program supplied.
   */
  hints.flags = (bitmap & (XValue | YValue)) ? USPosition : PPosition;
  hints.flags |= (bitmap & (WidthValue | HeightValue)) ? USSize : PSize;


  /*
   * Now create the window.
   */
  window = XCreateSimpleWindow(display, DefaultRootWindow(display),
				hints.x, hints.y, hints.width, hints.height,
				BORDER_WIDTH, foreground, background);

  /*
   * Set the standard properties.
   */
  XSetStandardProperties(display, window, TITLE,TITLE,None, argv, argc, &hints);

  /*
   * Set the standard colormap.
   */
  XSetWindowColormap (display, window, DefaultColormap(display, screen));

  /*
   * Communicate the window manager hints to the window managers.
   */
  XSetWMHints(display, window, &wmhints);

  /*
   * Create two graphics contexts, one for drawing in the pixmap and
   * one for clearing the pixmap.
   */
  gc_draw  = XCreateGC(display, window, 0, NULL);
  gc_clear = XCreateGC(display, window, 0, NULL);

  /*
   * Set foreground and background color.   We use an XFillRectangle request
   * to clear the pixmap.   The foreground color is used as the fill
   * color for XFillRectangle requests.  So we must set the foreground color in
   * `gc_clear' to `background'.
   */
  XSetForeground(display, gc_clear, background);
  XSetForeground(display, gc_draw, foreground);
  XSetBackground(display, gc_draw, background);

  /*
   * Reset the GraphicsExposures flag of `gc_clear' and `gc_draw'
   * so that no GraphicsExpose or NoExpose events will be generated
   * when the pixmap is erased or copied to the window.
   */
  XSetGraphicsExposures(display, gc_clear, False);
  XSetGraphicsExposures(display, gc_draw, False);

  /*
   * Linewidth could be set here.
   * We rely on the default value 0
   * that draws faster.
   */

  /* 
   * Notify the X server which events concerning this window 
   * should be reported:  expose, reconfiguration and keyboard events.
   */
  XSelectInput(display, window, EVENT_MASK);

  /*
   * Make the window visible by adding (raising) it to the top of the
   * window stack and by mapping it onto the display.
   */
  XMapRaised(display, window);

  /*
   * Inquire the actual size of the window (as opposed to the
   * size we proposed to the window manager) by XGetGeometry.
   */
  XGetGeometry(display, window, &root, &x, &y, &width, &height,
	     &border, &depth);

  /*
   * Create a pixmap with the same size and depth as the window.
   */
  pixmap = XCreatePixmap(display, root, width, height, depth);


  /*
   * Initialize the transformation of viewpoint to window coordinates
   * by computing the origin of the drawing.
   */
  origx = ((int) (0.5 * width)) + (int) x_off;
  origy = ((int) (0.5 * height)) - (int) y_off;
  i_origx = origx;
  i_origy = origy;

  /*
   * Initialize the variable `new_drawing'.
   */
  new_drawing = True;

  /*
   * The variable `segs' will now contain 
   * the segments that constitute the first wire frame.
   * If there is no input, the number of segments will be zero.
   */

  /*
   * Initialize the pixmap background.
   * This will generate NoExpose (or GraphicsExpose) event
   * that will start the loop of drawing the picture in the pixmap,
   * copying the pixmap to the window and erazing the pixmap again.
   */
  XFillRectangle(display, pixmap, gc_clear, 0, 0, width, height);

  /*
   * It may be correct to flush the output stream here, 
   * to make sure that clearing of the pixmap is performed
   * before drawing.
   * This is be done by calling XSync: all pending requests will
   * be sent to the X server, and XSync will wait till these requests
   * are processed.
   */
  XSync(display, 0); 

  graphics_setup = 1;
}




void exit_graphics(void)		/* Free X data */
{
  if (graphics_setup)
  {
	XAutoRepeatOn(display);
	XFreeGC(display, gc_draw);
	XFreeGC(display, gc_clear);
	XFreePixmap(display, pixmap);
	XDestroyWindow(display, window);
	XCloseDisplay(display);
	graphics_setup = 0;
  }
}





/*
 * The function that draws the segments that constitute the robot.
 * The begin and endpoints of the segments are stored in the variable
 * `segs' by a call to the function `compute_segments'.
 * Using XDrawSegments all the endpoints are send in one request over
 * the network connection to the X server.
 * This avoids network traffic delays caused by many seperate XDrawLine
 * requests.
 */
void redraw_wire_frame(void)
{
  XCopyArea(display, pixmap, window, gc_draw, 0, 0, width, height, 0, 0);
  XFlush(display);
}




void draw_next_wire_frame(void)
{
  /*
   * Compute the next wire frame.
   */
  compute_wire_frame();

  /*
   * Draw the segments in the off-screen pixmap.
   */
  XFillRectangle(display, pixmap, gc_clear, 0, 0, width, height);
  XDrawSegments(display, pixmap, gc_draw, (XSegment *) segs, no_segments);

  /*
   * Draw the wireframe in the window
   * by copying the drawing from the pixmap to the window.
   */
  XCopyArea(display, pixmap, window, gc_draw, 0, 0, width, height, 0, 0);

  /*
   * This one is REALLY necessary, since the events are picked up
   * by another process.
   */
  XFlush(display);
}



#define min(a,b) ((a) < (b) ?  (a) : (b))
#define max(a,b) ((a) > (b) ?  (a) : (b))



void resize_drawing(int new_width, int new_height)	/* Resize the drawing */
/*
 * Resize the drawing after a reconfiguration of the window,
 * because the window size may have changed.
 * Recompute the transformation of viewpoint to window coordinates
 * by recomputing the origin of the drawing  and redraw the picture.
 * New add-on feature: the drawing itself resizes, too!
 */
{
  int k;

  if (width != new_width || height != new_height)
  {
	/* 
	 * Apparently the size of the window has changed.
	 * We recompute the origin of drawing and resize it.
	 */

	if (new_width < new_height)
		scale_fac *= (REAL) new_width / min(width, height);
	else
		scale_fac *= (REAL) new_height / min(width, height);


	/* Scaling is really easy */
	init_transformations(scale_fac, focus, theta, angle, distance);

	width = new_width;
	height = new_height;
	x_off *= scale_fac / i_scale_fac;
	y_off *= scale_fac / i_scale_fac;
	i_origx = origx = ((int) (0.5 * width)) + (int) x_off;
	i_origy = origy = ((int) (0.5 * height)) - (int) y_off;
	i_focus = focus;
	i_angle = angle;
	i_distance = distance;
	i_theta = theta;
	i_scale_fac = scale_fac;



	/* 
	 * We update the wire frame with respect to
	 * the new coordinate system.
	 */

	k = 0;
	re_compute_points();
	update_segments(&k);
	if (z_projection)
	{
		re_compute_z_points();
		update_segments(&k);
	}

	/*
	 * Free the old pixmap.
	 */
	XFreePixmap(display, pixmap);

	/*
	 * Create a new pixmap of the right size.
	 */
	pixmap = XCreatePixmap(display, root, width, height, depth);

	/*
	 * Initialize the background of the pixmap.
	 */
	XFillRectangle(display, pixmap, gc_clear, 0, 0, width, height);

	/*
	 * Redraw the resized drawing in the resized pixmap.
	 */
	XDrawSegments(display, pixmap, gc_draw, (XSegment *) segs, no_segments);

	/*
	 * Draw the wireframe in the window by copying the drawing
	 * from the pixmap to the window.
	 */
	XCopyArea(display, pixmap, window, gc_draw, 0, 0, width, height, 0, 0);
	XFlush(display);
  }
}




void handle_next_window_event(void)
/*
 * Start event processing loop. The first event is generated by the 
 * mapping of the window on the display, subsequentl graphics expose
 * events are generated by the XCopyArea requests that are issued each
 * time the robot is drawn on the window.
 */
#define INIT_VAL	0.4
#define SCALE_VAL	1.2
#define MAX_ROT		20.0
#define MAX_DIST	100.0
{
  char input[10];		/* Keyboard input string */
  int i;			/* To store a return value */
  KeySym key;			/* To store KeyPress information */
  static REAL l_delta =  INIT_VAL, orig_l_delta =  INIT_VAL;
  static REAL r_delta = -INIT_VAL, orig_r_delta = -INIT_VAL;
  static REAL u_delta =  INIT_VAL, orig_u_delta =  INIT_VAL;
  static REAL d_delta = -INIT_VAL, orig_d_delta = -INIT_VAL;
  static REAL n_delta = -INIT_VAL, orig_n_delta = -INIT_VAL;
  static REAL f_delta =  INIT_VAL, orig_f_delta =  INIT_VAL;
  static REAL x_delta =  INIT_VAL, orig_x_delta =  INIT_VAL;
  static REAL X_delta = -INIT_VAL, orig_X_delta = -INIT_VAL;
  static REAL y_delta =  INIT_VAL, orig_y_delta =  INIT_VAL;
  static REAL Y_delta = -INIT_VAL, orig_Y_delta = -INIT_VAL;
  static int x_but1_press, y_but1_press;
  double fabs();
  static int release = 1, press = 0;

  XEvent event;

  XNextEvent(display, &event);

  switch (event.type)
  { 	
	case EnterNotify:
		XAutoRepeatOff(display);
		XFlush(display);
		break;

	case LeaveNotify:
		XAutoRepeatOn(display);
		/*
		 * No XNextEvent() is called, so we have to flush
		 * to make sure that the repeat is switched on again.
		 */
		XFlush(display);
		break;

	case Expose:
	        /* 
		 * When xexpose.count is zero this is the last of a series
	         * of expose events in the event queue.
		 * Remove any later expose events waiting in the queue
		 * to increase the responsiveness to other events.
		 */
		if (event.xexpose.count == 0)
			redraw_wire_frame();
		XFlush(display);
		break;

	case GraphicsExpose:
	        /* When xgraphicsexpose.count is zero this is the last of a
		 * series * of GraphicsExpose events in the event queue.
		 * Remove any later * GraphicsExpose events waiting in
		 * the queue to increase the responsiveness to other events.
		 */
		if (event.xgraphicsexpose.count == 0)
			redraw_wire_frame();
		XFlush(display);
		break;

	case NoExpose:
		break;

	case ConfigureNotify:
		/*
		 * By providing the X server with the StructureNotify mask
		 * we will receive reconfiguration events, but also other
		 * events like for instance CirculateNotify events.
		 * Only configure events are wanted and provide the
		 * necessary information to recompute the origin of the
		 * drawing after resizing of the window.
		 * So we have to test for the type of the event
		 * before inspecting the size of the window to see
		 * if we need to recompute the origin of the drawing.
		 */
		if (event.xconfigure.type == ConfigureNotify)
			resize_drawing(event.xconfigure.width,
						event.xconfigure.height);
		else
		{
			/* 
			 * Possibly a CirculateNotify event,
			 * redraw the picture.
			 */
			printf("stange conf\n");
			redraw_wire_frame();
		}
		XFlush(display);
		break;

	case KeyPress:
		input[0] = '\0';
		i = XLookupString(&(event.xkey), input, 1, &key, 0);
		if (i != 1)
			break;

		/*
		 * We have to simulate repeating keys.  When a key is
		 * pressed, send the same event to the display, unless
		 * the key has already been released.
		 */

		if (!event.xany.send_event)
		/*
		 * This is a genuine key press, i.e., not generated
		 * by XSendEvent.
		 */
		{
			press = input[0];
			release = 0;
		}
		
		/*
		 * Has the key already been released, then ignore old
		 * pending events.
		 */
		if (release)
			break;

		switch(input[0])
		  {
		  case 'p':
		    /*
		     * Make a pic depiction of the robot.
		     */
		    dump_wire_frame();
		    break;
		  case 'q':
		    /*
		     * Exit.
		     */
		    exit_graphics();
		    exit(0);
		  case 'o': case 'O':
		    /*
		     * Restore scales to original values.
		     */
		    init_values();
		    rotate_drawing(i_origx-x_off-(0.5*width),
				   -i_origy-y_off+(0.5*height),
				   0.0, 0.0, 0.0, 0.0);
		    break;
		  case 'l':
		    rotate_drawing(0.0,0.0, 0.0, l_delta, 0.0, 0.0);
		    if (fabs(l_delta) < MAX_ROT)
		      l_delta *= SCALE_VAL;
		    break;
		  case 'r':
		    rotate_drawing(0.0,0.0, 0.0, r_delta, 0.0, 0.0);
		    if (fabs(r_delta) < MAX_ROT)
		      r_delta *= SCALE_VAL;
		    break;
		  case 'u':
		    rotate_drawing(0.0,0.0, 0.0, 0.0, u_delta, 0.0);
		    if (fabs(u_delta) < MAX_ROT)
		      u_delta *= SCALE_VAL;
		    break;
		  case 'd':
		    rotate_drawing(0.0,0.0, 0.0, 0.0, d_delta, 0.0);
		    if (fabs(d_delta) < MAX_ROT)
		      d_delta *= SCALE_VAL;
		    break;
		  case 'n':
		    rotate_drawing(0.0,0.0, 0.0, 0.0, 0.0, n_delta);
		    if (fabs(n_delta) < MAX_DIST)
		      n_delta *= SCALE_VAL;
		    break;
		  case 'f':
		    rotate_drawing(0.0,0.0, 0.0, 0.0, 0.0, f_delta);
		    if (fabs(f_delta) < MAX_DIST)
		      f_delta *= SCALE_VAL;
		    break;
		  case 'x':
		    rotate_drawing(x_delta,0.0, 0.0, 0.0, 0.0, 0.0);
		    if (fabs(x_delta) < MAX_DIST)
		      x_delta *= SCALE_VAL;
		    break;
		  case 'X':
		    rotate_drawing(X_delta,0.0, 0.0, 0.0, 0.0, 0.0);
		    if (fabs(X_delta) < MAX_DIST)
		      X_delta *= SCALE_VAL;
		    break;
		  case 'y':
		    rotate_drawing(0.0,y_delta, 0.0, 0.0, 0.0, 0.0);
		    if (fabs(y_delta) < MAX_DIST)
		      y_delta *= SCALE_VAL;
		    break;
		  case 'Y':
		    rotate_drawing(0.0,Y_delta, 0.0, 0.0, 0.0, 0.0);
		    if (fabs(Y_delta) < MAX_DIST)
		      Y_delta *= SCALE_VAL;
		    break;
		  }

		/*
		 * Now send the event again.  Only if the event we
		 * just got matches the key that has been pressed last.
		 */
		if (input[0] == press)
		{
			XSendEvent(display, InputFocus, True, 0, &event);
			XFlush(display);
		}
	        break;

	case KeyRelease:
		i = XLookupString(&(event.xkey), input, 1, &key, 0);
		if (i != 1)
			break;
		switch(input[0])
		{
			case 'l': case 'r': case 'u': case 'd': case 'n':
			case 'f': case 'x': case 'X': case 'y': case 'Y':
				l_delta = orig_l_delta; r_delta = orig_r_delta;
				u_delta = orig_u_delta; d_delta = orig_d_delta;
				n_delta = orig_n_delta; f_delta = orig_f_delta;
				x_delta = orig_x_delta; X_delta = orig_X_delta;
				y_delta = orig_y_delta; Y_delta = orig_Y_delta;
				break;
		}
		release = input[0];
		XFlush(display);
		break;

        case ButtonPress:
		if (event.xbutton.button == 1)
		{
		  accept_homos = 0;
		  x_but1_press = event.xbutton.x;
		  y_but1_press = event.xbutton.y;
		}
		XFlush(display);
		break;

        case ButtonRelease:
		if (event.xbutton.button == 1)
		{
			accept_homos = 1;
		}
		XFlush(display);
		break;
	
	case MotionNotify:
		/*
		 * Follow the mouse pointer with a shift
		 * of the robot.  Only when the left mouse
		 * button is depressed.
		 */
		if (accept_homos)
			break;
		rotate_drawing(
			(REAL) (event.xbutton.x - x_but1_press),
			(REAL) (y_but1_press - event.xbutton.y),
			0.0, 0.0, 0.0, 0.0);
		x_but1_press = event.xbutton.x;
		y_but1_press = event.xbutton.y;
		XFlush(display);
		break;
		

	case MappingNotify:
		/* Keyboard mapping has changed. This entry is standard
		 * practice. Since some other client of the X server may
		 * have changed the keyboard mapping, we have to reset this
		 * mapping to keep the standard key bindings, whenever we
		 * receive a MappingNotify event.
		 */
		XRefreshKeyboardMapping((XMappingEvent *) (&event));
		XFlush(display);
		break;
	}
}




void rotate_drawing(REAL dx, REAL dy, REAL df, REAL dt, REAL da, REAL dd)
{
  int k;

  focus += df;
  angle += da;
  while (angle > 360.0) angle -= 360.0;
  while (angle < 0.0) angle += 360.0;
  distance += dd;
  theta += dt;
  while (theta > 360.0) theta -= 360.0;
  while (theta < 0.0) theta += 360.0;

  init_transformations(scale_fac, focus, theta, angle, distance);

  x_off += dx;
  y_off += dy;
  origx = ((int) (0.5 * width)) + (int) x_off;
  origy = ((int) (0.5 * height)) - (int) y_off;

  /* 
   * We update the wire frame with respect to
   * the new coordinate system.
   */
  k=0;
  re_compute_points();
  update_segments(&k);
  if (z_projection)
  {
	re_compute_z_points();
	update_segments(&k);
  }

  /* Initialize the background of the pixmap */
  XFillRectangle(display, pixmap, gc_clear, 0, 0, width, height);

  /*
   * It may be correct to flush the output stream now, so that
   * all pending requests will be sent to the X server, and to
   * wait till they are processed to make sure that the Fillrectangle
   * request to erase the pixmap will be handled before starting to
   * draw the wire frame.
   * This is done with:
   */
  XFlush(display);
  
  /*
   * Redraw the resized drawing in the resized pixmap.
   */
  XDrawSegments(display, pixmap, gc_draw, (XSegment *) segs, no_segments);

  /*
   * Draw the wireframe in the window
   * by copying the drawing from the pixmap to the window.
   */
  XCopyArea(display, pixmap, window, gc_draw, 0, 0, width, height, 0, 0);

  /*
   * Make sure by another synchronization request that the
   * drawing in the pixmap is actually finished, before
   * copying the pixmap to the window.  This is quite necessary,
   * since XNextEvent(), which implicitly calls XFlush(), is never
   * called in this program, nor are XPending() or XWindowEvent().
   */
  XFlush(display);
}
