/**************************************************************************/
/* main.c                                                       /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/
#define MAIN_EXTERN

#include "config.h"


#include <stdio.h>
#include <signal.h>
#include "main.h"
#include "input.h"
#include "compute.h"
#include "communication.h"
#include "read_hmatrix.h"
#include "xgraphics.h"


void panic_exit(void)
{
  exit_graphics();
}


void init_values(void)
{
  focus = i_focus;
  angle = i_angle;
  distance = i_distance;
  theta = i_theta;
  scale_fac = i_scale_fac;
}




void main(int argc, char *argv[])
{
  FILE *objfilep, *matfilep;
  /*
   * The sequence number (i.e., numeral in name string) of this executable.
   */
  int sequence;

  focus = 1000.0,
  angle = 30.0,
  distance = 50.0,
  theta = 30.0,
  scale_fac = 1.0;
  z_projection = 0;
  pic_scale = fig_scale = 1.0;
  lineno = 1;

  check_options(argc, argv, &objfilep, &matfilep);
  i_focus = focus;
  i_angle = angle;
  i_distance = distance;
  i_theta = theta;
  i_scale_fac = scale_fac;

  read_robotfile(objfilep);
  fclose(objfilep);
  init_transformations(scale_fac, focus, theta, angle, distance);
  init_homogen();
  read_hmatrix(matfilep);

  (void) signal(SIGINT, exit_graphics);
  (void) signal(SIGQUIT, exit_graphics);

  init_graphics(argc, argv);

  read_next_message();
}



/*
 * check_options checks the options and if all is ok  then opens the inputfile
 * leaving the filepointer in filep. if anything goes wrong then an error
 * message is printed and the program stops.
 */

void check_options(int argc, char **argv, FILE **objfilep, FILE **matfilep)
{
  int i;
  REAL *address, offset_x = 0, offset_y = 0, delay = 0;
  char *df;
  char *geometry;
  double tmp_f;

  if (argc <= 2)
  {
	fprintf(stderr, "Usage:  %s <objects-file> <matrices-file>\n", argv[0]);
	exit(1);
  }

  for(i = 1 ; i < argc - 2 ; i ++)
  {
	address = NULL;
	df = NULL;
	geometry = NULL;


	if (! strcmp(argv[i], "-z")) 
	{
		z_projection = 1;
		continue;
	}
	if (! strcmp(argv[i], "-f")) address = &focus;
	if (! strcmp(argv[i], "-a")) address = &angle;
	if (! strcmp(argv[i], "-d")) address = &distance;
	if (! strcmp(argv[i], "-t")) address = &theta;
	if (! strcmp(argv[i], "-s")) address = &scale_fac;
	if (! strcmp(argv[i], "-p")) df = argv[i + 1];
	if (! strcmp(argv[i], "-w")) address = &delay;
	if (! strcmp(argv[i], "-x")) address = &offset_x;
	if (! strcmp(argv[i], "-y")) address = &offset_y;
	if (! strcmp(argv[i], "-P")) address = &pic_scale;
	if (! strcmp(argv[i], "-F")) address = &fig_scale;
	if (! strcmp(argv[i], "-geometry")) geometry = argv[i + 1];
	if (df != NULL)
	{
		if (i + 1 >= argc - 2)
		{
			   fprintf(stderr, "%s: argument expected after %s",
				    argv[0], argv[i]);
			   exit(1);
		}
		dumpfile = df;
		i++;
		continue;
	}


	if (geometry != NULL)
	{
		if (i + 1 >= argc - 2)
		{
			   fprintf(stderr, "%s: argument expected after %s",
				    argv[0], argv[i]);
			   exit(1);
		}
		user_geometry = geometry;
		i++;
		continue;
	}


	if (address == NULL) {
		fprintf(stderr, "%s: illegal option %s.\n", argv[0],
				argv[i]);
		exit(1);
	}


	if (i + 1 >= argc - 2)
	{
		fprintf(stderr, "%s: argument expected after %s.\n",
				argv[0], argv[ i ]);
		exit(1);
	}


	i++;

	if (sscanf(argv[i], "%lf", &tmp_f) == 1)
	{
		*address = (REAL) tmp_f;
		continue;
	}

	fprintf(stderr, "%s: illegal argument %s.\n", argv[0], argv[i]);
	exit(1);
  }


  if (scale_fac <= 0.0)
  {
	fprintf(stderr, "%s: scale factor <= 0.\n", argv[0]);
	exit(1);
  }

  if ((*objfilep = fopen(argv[argc - 2], "r")) == NULL)
  {
	perror(argv[argc - 2]);
	exit(1);
  }
  if ((*matfilep = fopen(argv[argc - 1], "r")) == NULL)
  {
	perror(argv[argc - 1]);
	exit(1);
  }

  if (dumpfile == NULL) dumpfile = DEFAULT_DUMP;
  x_off = (REAL) offset_x;
  y_off = (REAL) offset_y;
  wait_time = (unsigned) delay;
}
