/**************************************************************************/
/* compute.c --- after a version by F. J. Jungen                /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/




#define COMPUTE_EXTERN

#include "config.h"

#include <stdio.h>
#include "compute.h"
#include "matrix.h"
#include "datastruct.h"
#include "main.h"
#include "xgraphics.h"

/*
 * init_transformations: computes PV i.e. the perspective transform times
 * the viewpoint transform.
 */

void init_transformations(REAL scale_fac, REAL focus,
			  REAL theta, REAL alfa, REAL distance)
{
  homo P, V;

  perspective_transform(P, focus);
  viewpoint_transform(V, theta, alfa, distance);
  mul_matrix(P, HOMO_DEG, HOMO_DEG,
		V, HOMO_DEG, HOMO_DEG,
		PV, HOMO_DEG, HOMO_DEG);
  PV[HOMO_DEG*HOMO_DEG-1] /= scale_fac;
}



/*
 * init_homogen initialises the homogeneous matrices which
 * will be specified by the user.
 */
void init_homogen(void)
{
  if ((homogen = (homo *) calloc(nr_links+1, sizeof(homo))) == NULL)
  {
	perror("init_homogen(): homogen");
	exit(1);
  }

  /*
   * homogen[0] is the homogenous matrix for the floor.  It needs
   * not be read in, only initialised once.
   */
  init_matrix(homogen[0], HOMO_DEG, HOMO_DEG);
}




/*
 * re_compute_points: computes the new points from PV and the homogeneous
 * matrices.
 */
void re_compute_z_points(void)
{
  int i;
  REAL p[4], pres[4];
  register struct link *current_link;

  current_link = links;
  for(i = 0 ; i < nr_links ; i++, current_link++)
  {
	int k, j;
	int nr_points = current_link->nr_points;
	matrix p_p1, p_p2;
	matrix p_r1, p_r2;

	p_p1 = current_link->points;
	p_r1 = current_link->result;

	for (j=0; j<links[i].nr_points; j++)
	{
		for (k=0, p_p2 = p_p1+j; k<4; k++, p_p2 += nr_points)
			p[k] = *p_p2;
		mul_matrix(homogen[i], HOMO_DEG, HOMO_DEG,
			   p, HOMO_DEG, 1, pres, HOMO_DEG, 1);

		/*
		 * We want to see the projection of the 
		 * 3D image on the Z=0 plane.  So, set the Z
		 * coordinate to 0.
		 */
		pres[2] = 0.0;
		mul_matrix(PV, HOMO_DEG, HOMO_DEG,
			   pres, HOMO_DEG, 1,
			   p, HOMO_DEG, 1);
		for (k=0, p_r2 = p_r1+j; k<4; k++, p_r2 += nr_points)
			*p_r2 = p[k];
	}
  }
}



void re_compute_points(void)
{
  homo res;
  int i;
  register struct link *current_link;

  current_link = links;
  for (i = 0; i < nr_links; i++, current_link++)
  {
	mul_matrix(PV, HOMO_DEG, HOMO_DEG,
		   homogen[i], HOMO_DEG, HOMO_DEG,
		   res, HOMO_DEG, HOMO_DEG);
	mul_matrix(res, HOMO_DEG, HOMO_DEG,
	    current_link->points, HOMO_DEG, current_link->nr_points,
	    current_link->result, HOMO_DEG, current_link->nr_points);
  }
}



/*
 * update_segments: update the segments of the wire frame of the robot.
 */
void update_segments(int *k)
{
  register struct link *current_link;
  register int i, j;
  REAL x1, x2, y1, y2;
  register struct segment *seg;

  /* Fill the segments array with newly computed data from
   * the links array.  Transform the ViewPoint coordinates
   * of these segments to Window coordinates by adding or
   * substracting origx or origy.  The Window origin is at
   * the left upper corner.  The ViewPoint origin will be
   * mapped to the point (origx, origy).
   * This ought to be done by a function in the xgraphics.c
   * file.  Here we chose for quick and dirty and combined
   * update and transformation in one loop.
   */

  seg = &segs[*k];
  current_link = &links[0];
  for (i = 0 ; i < nr_links ; i ++)
  {
	REAL *p_w, *p_c;
	int points, lines;

	p_c = current_link->result;
	points = current_link->nr_points;
	p_w = &p_c[points*3];
	lines = current_link->nr_lines;

	for (j = 0 ; j < lines ; j++)
	{
		REAL w;

		if ((w = p_w[current_link->lines[j].start]) == 0)
			x1 = y1 = 0.0;
		else
		{
			x1 = p_c[current_link->lines[j].start]/w;
			y1 = p_c[points + current_link->lines[j].start]/w;
		}
		seg->x1 = (int) x1 + origx;
		seg->y1 =  origy - (int) y1;

		if ((w = p_w[current_link->lines[j].finish]) == 0)
			x2 = y2 = 0.0;
		else
		{
			x2 = p_c[current_link->lines[j].finish]/w;
			y2 = p_c[points + current_link->lines[j].finish]/w;
		}
		seg->x2 = (int) x2 + origx;
		seg->y2 = origy - (int) y2;

		(*k)++; seg++;
	}

	current_link++;
  }
}



/*
 * Recompute the wire_frame.
 */
void compute_wire_frame(void)
{
  int k;

  k = 0;
  re_compute_points();
  update_segments(&k);

  if (z_projection)
  {
	re_compute_z_points();
	update_segments(&k);
  }
}




/*
 * dump_wire_frame dumps the segments in pic format on a file whose name is 
 * specified in dumpfile.
 */
void dump_wire_frame(void)
{
  int i;
  FILE *fp;
  char s[100];
  REAL min_x, min_y;
  int fig_offset = 10;
  
  sprintf(s, "%s.pic", dumpfile);

  if ((fp = fopen(s, "w")) == NULL)
  {
	perror(s);
	return;
  }
  fprintf(fp, ".PS\n.ps 8\n");
  
  min_x = min_y = 10.0;
  for (i = 0 ; i < no_segments ; i ++)
  {
	fprintf(fp, "line from %f,%f to %f,%f\n",
		segs[i].x1 / pic_scale, (origy - segs[i].y1) / pic_scale,
		segs[i].x2 / pic_scale, (origy - segs[i].y2) / pic_scale);
	if (segs[i].x1 * fig_scale < min_x)
		min_x = segs[i].x1 * fig_scale;
	if (segs[i].x2 * fig_scale < min_x)
		min_x = segs[i].x1 * fig_scale;
	if (-(origy - segs[i].y1) * fig_scale < min_y)
		min_y = -(origy - segs[i].y1) * fig_scale;
	if (-(origy - segs[i].y2) * fig_scale < min_y)
		min_y = -(origy - segs[i].y2) * fig_scale;
  }
  fprintf(fp, ".PE\n");
  fclose(fp);


  sprintf(s, "%s.fig", dumpfile);

  if ((fp = fopen(s, "w")) == NULL)
  {
	perror(s);
	return;
  }
  fprintf(fp, "#FIG 2.1\n80 2\n");
  for (i = 0 ; i < no_segments ; i ++)
  {
	fprintf(fp, "2 1 0 1 -1 0 0 0 0.000 -1 0 0\n");
	fprintf(fp, "	%d %d %d %d 9999 9999\n",
		(int) (segs[i].x1 * fig_scale - min_x + fig_offset),
		(int) (-(origy - segs[i].y1) * fig_scale - min_y + fig_offset),
		(int) (segs[i].x2 * fig_scale - min_x + fig_offset),
		(int) (-(origy - segs[i].y2) * fig_scale - min_y + fig_offset)
	);
  }
  fclose(fp);
}
