/*  EXEC.PL  */


/*
SPECIFICATION
-------------

This module provides facilities for writing ``brains'' for Eden that, as
far as they are concerned, run continuously, rather than starting afresh
on each call to 'think'. In effect, it provides a simple form of
coroutine. Experienced Poplog programmers will see that it is an simple
interface to LIB PROCESS (see HELP PROCESS): I have written it so that
novices can write co-routine style brains without having to know
anything more about LIB PROCESS, or how to call it from Prolog.


You can make a goal into a process by calling
    goal_to_process( Goal, Co ).

You can then start running it by calling
    restart_process( Co, Action ).

Eventually, Goal should call
    exec( A )
where A is an action. This will get passed back as the second argument
of 'restart_process'. Goal will stop running, and return control to
just after this call of 'restart_process'. To start Goal from where it
left off, call 'restart_process' again.


PUBLIC goal_to_process( Goal+, Co- ):

Converts Goal into a process, returned in Co.


PUBLIC restart_process( Co+ ):

Runs Co until it calls exec(Action). Action is then returned as the
second argument of 'restart_process', and Co suspends itself, as
described above. Co can either be a process which has not been run
before, or one which has, and is suspended.


PUBLIC exec( Action+ ):

Called from inside Goal to suspend and return an action.


Example:

    start_thinking :-
        goal_to_process( brain, Co ),
        asserta( co(Co) ).

    think( Action ) :-
        co( Co ),
        restart_process( Co, Action ).

    brain :-
        smell( here ),
        !,
        exec( grab ),
        exec( use ),
        brain.

    brain :-
        smell( S ),
        exec( S ),
        brain.

    bugdead( _, rerun ) :-
        start_thinking.

This bug can be found in EXEC_BUG.PL.
*/


/*
IMPLEMENTATION
--------------

This is Poplog-specific, and relies on the process feature. See REF
PROCESSES.
*/


exec( Result ) :-
    prolog_eval( suspend(quote(Result),1) ).


restart_process( Co, Result ) :-
    prolog_eval( runproc(0,Co), Result ).


goal_to_process( Goal, Co ) :-
    prolog_eval(
                apply(
                      quote(Goal),
                      valof(word_identifier(goal_to_process,
                                            section_subsect(exec,valof(pop_section),false),
                                            true
                                           )
                           )
                     ),
                Co
               ).


:- prolog_language(pop11).

section $-exec;

define goal_to_process( Goal );
    lvars Goal, Coroutine;
    consproc( 0, prolog_invoke(%Goal%) )
enddefine;

endsection;

:- prolog_language(prolog).
