/*  SEARCH_BUG.P  */


/*
This bug demonstrates depth- and breadth-first search. It needs to
be run in treeworld:
    eden( 'search_bug', 'treeworld' );

It is implemented using LIB EXEC. When the brain starts, by
start_thinking being called, it calls
    search( [ [go forward [] ] ], [] );

'search' is a routine of two arguments. The first is a queue of points
to be explored. The second is the current position. Points to be
explored are represented as
    [ go <direction> <position> ]
where <direction> is one of forward, left, or right. <position>
represents a previous bug position, as the path taken from the root of
the tree in treeworld. Such paths are represented as a list of
<direction>s.

'search' first "forces evaluation" of the first point to be explored, by
making the bug walk to the node it indicates, from wherever it is now.
The bug has now ended up at an X, *, or +. If it's at the +, it eats it.
If at a * or X, it makes a list of the possible directions it can go in,
and adds these to the queue. (For a *, there are none). It then repeats,
and this continues until it eats the food, or the queue is empty. The
search_type variable controls whether the bug engages in depth-first or
breadth-first search, by controlling how the queue is updated.


I derived this program by starting with a standard search algorithm. The
key idea was that points to be explored can be positions which the bug
may not yet have reached. If you represent them in the form
    "You have to go in <direction> from <here>
then you can insert an "evaluation-forcing" stage, as long as it happens
before the bug needs to find the point's descendants. The evaluation-
forcer corresponds to quite a lot of activity (see 'force' below), but
can be hived off from the search routine itself.
*/


needs exec;
needs fault;
needs smatch;


define start_thinking();
    proc_to_process( search(% [ [go forward [] ] ], [] %) ) -> co;
enddefine;


define think();
    restart_process( co );
enddefine;


define bugdead();
    start_thinking();
    "rerun";
enddefine;


vars search_type;


define search( Queue, Here );
    lvars Queue, Here;
    vars Dir, Parent, RestQ, Dirs;

    /*  Ask user what kind of search to demonstrate: sets search_type.
    */
    get_search_type();

    while Queue /= [] do

        /*  1. Take next point off queue.  */
        Queue --> [ [go ? ^ !Dir ? ^ !Parent] ?? ^ !RestQ ];

        /*  2. "Force evaluation", by making bug walk to the
            node (an X-object) at the new point from its current
            position which is Here.
        */
        force( [^Dir ^^Parent], Here );

        /*  3. Get possible directions from this new node.  */
        directions( [^Dir ^^Parent] ) -> Dirs;

        if search_type = "dfs" then
            Dirs<>RestQ -> Queue;
        elseif search_type = "bfs" then
            RestQ<>Dirs -> Queue;
        else
            FAULT( 'search: illegal search type', [%search_type%] )
        endif;

        /*  4. Add these choices to the queue.  */ 
        [ ^Dir ^^Parent ] -> Here;
    endwhile;
enddefine;


/*  force( Next, Here ):
        Make bug walk to Next from Here. In general, bug will have to
        find the common ancestor of the nodes represented by Here and
        Next; walk back to it from Here; and then walk forward to Next.
*/
define force( Next, Here );
    lvars Next, Here;
    lvars A, AToNext, HereToA;

    common_ancestor( Next, Here ) -> AToNext -> HereToA;
    backtrack_along( HereToA );
    forward_along( AToNext );
enddefine;


/*  common_ancestor( Next, Here ) -> AToNext -> HereToA:
        HereToA becomes the path from Here to the common ancestor
        of Next and Here; AToNext becomes the path from it to Next.
*/
define common_ancestor( Next, Here ) -> AToNext -> HereToA;
    lvars Next, Here, AToNext, HereToA;

    common_ancestor_1( rev(Next), rev(Here) ) -> AToNext -> HereToAR;
    rev( HereToAR ) -> HereToA;
enddefine;


/*  common_ancestor_1( L1, L2 ) -> T1_ -> T2_:
        L1 and L2 are paths from the root of the tree to two nodes,
        represented as reversed lists of directions. Let A be the nodes'
        common ancestor. Then T1_ and T2_ are the (reversed) paths from
        A to each of the nodes. This just involves taking off the lists'
        common prefix.
*/
define common_ancestor_1( L1, L2 ) -> T1_ -> T2_;
    lvars L1, L2, T1_, T2_;

    L1 -> T1_;
    L2 -> T2_;
    while T1_ /= [] and T2_ /= [] and hd(T1_) = hd(T2_) then
        tl(T1_) -> T1_;
        tl(T2_) -> T2_;
    endwhile;
enddefine;


/*  backtrack_along( Path ):
        Walk backwards along Path.
*/
define backtrack_along( Path );
    lvars Path;
    vars Dir, P;

    while Path /= [] do
        Path --> [ ? ^ !Dir ?? ^ !P ];

        go_back_to_X();
        if Dir = "left" then
            exec( "right" )
        elseif Dir = "right" then
            exec( "left" )
        elseif Dir = "forward" then
        else
           FAULT( 'backtrack_along: bad Dir', [% Dir, Path %] )
        endif;
        P -> Path;
    endwhile;
enddefine;


/*  go_back_to_X():
        Bug is standing on a line. Walk back to the previous X.
*/
define go_back_to_X();
    /*  Walk backwards along the line.  */
    while see_line( 0, -1 ) do
        exec( "back" );
    endwhile;

    if see( 0, -1 ) = `X` then
        /*  Step back onto the X.  */
        exec( "back" );
    endif;
enddefine;


/*  forward_along( Path ):
        Bug is standing on a X. If Path is non-empty, walk along
        it to the end. This leaves bug standing on an X, or in
        front of a *.
*/
define forward_along( Path );
    lvars Path;
    vars Dir, Next, End, Obj;

    while Path /= [] do
        Path --> [ ? ^ !Dir ?? ^ !Next ];

        if Dir = "forward" then
            exec( "forward" )
        elseif Dir = "left" then
            exec( "left" );
            exec( "forward" )
        elseif Dir = "right" then
            exec( "right" );
            exec( "forward" )
        else
            FAULT( 'forward_along: bad direction', [% Path %] )
        endif;

        /*  Bug is now standing on a - or |. Walk along the line to
            the thing at the end: either a *, X, or +.
        */
        walk_to_end() -> End;

        Next -> Path;
    endwhile;
enddefine;


/*  walk_to_end():
        Bug is standing on a - or |. Walk along it until it hits a
        X, * or +. If the latter, eat the food; if an X, stay standing
        on it; if a *, stay in front of it.
        The routine returns the object found, as either an X or a *. It
        will never return +, since eating the food kills the bug.
*/
define walk_to_end();
    lvars Obj;

    see( 0, 1 ) -> Obj;
    while Obj = `-` or Obj = `|` do
        exec( "forward" );
        see( 0, 1 ) -> Obj;
    endwhile;
    /*  Walk along line until object in front is not a - or |  */

    if Obj = `+` then
        /*  Food is one square in front. Eat it.  */
        exec( "forward" );
        exec( "grab" );
        exec( "use" );
        /*  This terminates the current life.  */
    elseif Obj = `X` then
        exec( "forward" );
        `X`
    elseif Obj = `*` then
        `*`
    else
        FAULT( 'walk_to_end: bad object', [%Obj%] );
    endif;
enddefine;


/*  directions( P ):
        Bug is standing on an X or in front of a *, and has to find out
        what directions it can go in. This routine returns them as a
        list. The elements of this list are in the order
            right, forward, left

        P is the path to here (representing ``here'').
*/
define directions( P ) -> Dirs;
    lvars P, Dirs;
    lvars D1, D2;

    if see_line( -1, 0 ) then
        [ [go left ^P] ]
    else
        []
    endif -> D1;
    /*  If we saw a line to the left. */

    if see_line( 0, 1 ) then
        [ [go forward ^P] ^^D1 ]
    else
        D1
    endif -> D2;
    /*  If we saw a line ahead. */

    if see_line( 1, 0 ) then
        [ [go right ^P] ^^D2 ]
    else
        D2
    endif -> Dirs;
    /*  If we saw a line to the right. */

enddefine;


/*  see( xB, yB ):
        Return the object at (xB,yB) in the bug's field of vision,
        where xB and yB are bug-centered co-ordinates.
*/
define see( xB, yB );
    lvars xB, yB;
    retina()( xB+3, yB+2 );
enddefine;


/*  see_line( xB, yB ):
        True if the object seen at (xB,yB) is a vertical or horizontal
        line.
*/
define see_line( xB, yB );
    lvars xB, yB;
    lvars Obj;
    see( xB, yB ) -> Obj;
    Obj = `-` or Obj = `|`
enddefine;


/*  get_search_type():
        Ask the user for a type of search to do, and set search_type.   
*/
define get_search_type();
    lvars keypress;

    bug_message( 'Search type: D(epth-first)/B(readth-first)' );

    if using_ved() then
        rawcharin().tlc -> keypress
    else
        charin().tlc -> keypress;
    endif;

    switchon keypress
    case = `d` then "dfs"
    case = `b` then "bfs"
    else
        get_search_type()
    endswitchon -> search_type;
enddefine;
