(in-package "RRL")

;; Eqn solving package

(defun solve (lhs rhs)
;; The top level function called by the user, which performs
;; all necessary work before calling the search function for
;; the solutions.

; replaces get-all-vars - siva.
   (setq *list-of-vars* (union (var-list lhs) (var-list rhs))
         *rule-list*    $rule-set)
   (compile-information)
   (set-time-debug-info)
   (setq *run-time* (get-internal-run-time))
   (let ((total-time 0))
     (add-time total-time 
               (traverse-tree  (form-init-eqn-lst lhs rhs)))
     (reset-all-globals)
     (format t "~% Total Time used for solving =   ~6,2f seconds ~%" 
               (nrm-time total-time))))
   


(defun traverse-tree (node)
        (catch '*solve* (traverse node)) )


(defun traverse (node)
;; This  is the driver function, which tries to solve the goal,
;; and every time the IDFS procedure returns without completing
;; the entire tree, it increments the depth bound (*max-depth*)
;; and calls the function again.
    (prog (temp time-const)
         (setq time-const 0)
       again
         (add-time time-const (setq temp (solve-idfs node)))
         (if (or (equal temp 'fail) (null temp)) 
             (format t "~%No more solutions to this equation.~%")
             (progn 
                (setq *prv-depth* *max-depth*)
                (setq *max-depth* (add *max-depth* *increment*))
                (stop-if-depth-too-large *max-depth*)
                (if (member  'traverse  *time-debug*)
                 (progn
                  (format t "~%   Time taken for this depth is :  ~6,2f secs"
                                   (nrm-time time-const))
                  (format t "~%   Incrementing depth bound to ~a ~%" *max-depth*)
                  (setq  time-const  0)))
                (go again)) )))



(defun solve-idfs (node)
;; This is the  function which performs a DFS on the search space.
;; The function checks whether a GOAL state has already been seen,
;; and if so, a solution is printed. Else, if the depth bound has
;; been  exceeded then it returns CONTINUE,  else it gets all the
;; successors of the current node and resolves them.
    (cond ((null node) nil)
          ((s-all-null (s-get-elist node))
              (if (s-show-ans node)
                  (let* ((temp)
                         (tme  (get-internal-run-time))
                         (tot  (- tme *run-time*)))
                     (if (member 'solution *time-debug*)
                      (progn
                        (setq *run-time*  tme)
                        (format t "Time taken for this solution : ~6,2f secs~%"
                                  (nrm-time  tot))))
                     (if (eq 'n (ask-choice temp '(y n) "Try Again?"))
                         (throw '*solve* nil)))))
          ((>  (s-get-depth node) *max-depth*) 'cont)
          ( t  (let  ((succs   (get-all-succs  node)))
                 (if (null succs) 'fail
                     (do*  ((lst      succs      (cdr lst))
                            (one      (car lst)  (car lst))
                            (ans-lst  (list (solve-idfs one))
                                      (cons (solve-idfs one) ans-lst)))
                           ((null (cdr lst)) (interpret ans-lst)) ))) )) )


(defun get-all-succs (node)
;; This function  is used by  SOLVE-IDFS to get the list of all
;; possible  successors  of NODE.  Successors are  generated by
;; testing for possible DECOMPOSE and all possible RESTRUCTURES.
;; In  the process  FEASIBILITY is  checked, and DECOMPOSED and
;; UNIFY are forced wherever possible.
    (let*   ((depth    (s-get-depth   node))
             (elst     (s-get-elist   node))
             (ans      (s-get-ans     node))
             (curr     (s-get-curr    elst))
             (rest     (s-get-rest    elst))
             (lhs      (s-get-eqn-lhs   curr))
             (rhs      (s-get-eqn-rhs   curr)))
       (cond ((equal-normal-forms lhs rhs ans)
                (list (make-node depth rest ans)))
             ( t  (let  ((succs   (s-successors   curr   ans)))
                     (do*  ((lst      succs      (cdr lst))
                            (feqns    (get-first-eqs   lst)
                                      (get-first-eqs   lst))
                            (fans     (get-first-ans   lst)
                                      (get-first-ans   lst))
                            (all-ans  nil))
                            ((null lst) all-ans) 
                            (setq all-ans 
                                      (cons 
                                       (make-node (add1 depth)
                                                  (s-a-lists1 feqns rest)
                                                  fans) all-ans))) )) ) ) )


(defun set-time-debug-info()
;; This function is called when the solver is started up, and it
;; makes entries in the list *TIME-DEBUG* which is later used to
;; print time informations.
    (cond  ((equal  *time*  1) (push  'traverse  *time-debug*))
           ((equal  *time*  2) (push  'solution  *time-debug*))
           ((equal  *time*  3) (push  'traverse  *time-debug*)
                               (push  'solution *time-debug*)) ) )


(defun reset-all-globals ()
;; When the call to SOLVE finally returns, it must reset all the
;; global variables used.
    (setq 
          *rule-list*      nil
          *max-depth*       15
          *prv-depth*        0
          *cur-depth*        0
 
          *reachable*      nil
          *list-of-vars*   nil
          *all-lhs-ops*    nil
          *all-ops*        nil

          *unwanted*       nil

          *time-debug*     nil
          *MAX-LIM*         50
    )
)
