;;; isa-ruletable.el 
;;;       - Buffer holding table of rules and tactics for Isabelle mode.
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-ruletable.el,v 1.11 1994/03/11 19:54:57 da Exp $
;;;

;;; DESIRED CHANGES:
;;;   - improve pretty margin setting bits.
;;;   - tidy

(require 'isa-mode)



;;; ============ Ruletable Mode ============

(cond (window-system
       (make-face 'ruletableGroupname)
       (defvar ruletableGroupname-default 'bold)))

(defvar ruletable-mode-map nil)

(or ruletable-mode-map
    (let ((map (make-keymap)))
      (suppress-keymap map)
      (isa-clear-mouse-bindings map)
      (set-keymap-name map 'ruletable-mode-map)
      ;; This defines the action on a mouse button down event, if
      ;; such a distinction is made.  (Presently, yes in FSF, no in Lucid).
      (isa-define-popup-key map 'button1 'ruletable-select-indicated-rule)
      (isa-define-popup-key map 'button3 'ruletable-show-indicated-rule)
      (define-key map " " 'ruletable-select-pointed-rule)
      (define-key map "h" 'describe-mode)
      (define-key map "i" 'isa-select-isa-buffer)
      (define-key map "q" 'isa-remove-temp-buffer)
      (define-key map "\C-f" 'ruletable-forward-word)
      (define-key map "\C-b" 'ruletable-backward-word)
      (define-key map "\C-p" 'ruletable-up-word)
      (define-key map "\C-n" 'ruletable-down-word)
      (define-key map "\C-i" 'ruletable-toggle-long)
      (define-key map "\C-m" 'ruletable-show-pointed-rule)
      (setq ruletable-mode-map map)))

(defvar ruletable-rule-regexp "[ \t]+\\([a-zA-Z0-9_']+\\)")

(defun ruletable-forward-word ()
  (interactive)
  (if (re-search-forward ruletable-rule-regexp nil t)
      (goto-char (match-beginning 1))))

(defun ruletable-backward-word ()
  (interactive)
  (if (re-search-backward ruletable-rule-regexp nil t)
      (goto-char (match-beginning 1))))

(defun ruletable-up-word ()
  (interactive)
  (forward-line -1)
  (end-of-line)
  ;; without space below, regexp fails in Lucid.
  (if (re-search-backward (concat "^ " ruletable-rule-regexp) nil t)
      (goto-char (match-beginning 1))
    (ruletable-forward-word)))

(defun ruletable-down-word ()
  (interactive)
  ;; without space below, regexp fails in Lucid.
  (if (re-search-forward (concat "\n " ruletable-rule-regexp) nil t)
      (goto-char (match-beginning 1))))

(defun ruletable-mode ()
  "Major mode for Isabelle rule-table buffers.
The cursor keys or mouse are used to move over the rule
names.

\\<ruletable-mode-map>
\\[ruletable-select-pointed-rule] or \\[ruletable-select-indicated-rule] copy the rule name under the 
cursor or mouse respectively into the related Isabelle buffer.
\\[ruletable-show-pointed-rule] or \\[ruletable-show-indicated-rule] display the rule using prth(rulename) 
in the related Isabelle buffer.

\\[ruletable-toggle-long] toggles the display format between short and
long forms: the long form includes section headings and subgroupings,
but takes up more space.
"
  (setq buffer-read-only t)
  (setq major-mode 'ruletable-mode)
  (set-syntax-table isa-thy-mode-syntax-table)
  (use-local-map ruletable-mode-map)
  (setq mode-line-buffer-identification '("Rules: %17b"))
  (make-local-variable 'theory-name)
  (put 'ruletable-mode 'mode-class 'special)
  (isa-remove-menubar-if-multiple-screen-mode))



(defun ruletable (&optional thy)
  "Display a table of rules for THEORY.
The table is constructed from the table isa-theory-rules."
  (interactive
   (list 
    (if (eq major-mode 'isa-mode)
	(completing-read "Name of theory: " 
			 isa-theory-rules nil t 
			 (if (assoc isa-logic-name isa-theory-rules)
			     isa-logic-name))
      (error "Must be in Isabelle buffer"))))
  (if (not (eq major-mode 'isa-mode))
      (error "Must be in Isabelle buffer"))
  (setq thy (or thy isa-logic-name))
  (if (not (assoc thy isa-theory-rules))
      (error "No rules for theory %s in isa-theory-rules" thy))
  (isa-display-buffer 
   (save-excursion
     (set-buffer 
      (isa-create-new-associated 'ruletable)) ; did have last arg t
     (ruletable-mode)
     (setq theory-name thy)
     (setq mode-name (concat thy " rules"))
     (ruletable-make-table)
     (current-buffer))
   t))



;;; ========== Making rule tables ==========

(defun ruletable-make-table ()
  (let ((buffer-read-only nil)
	(rules (cdr (assoc theory-name isa-theory-rules)))
	(sc    (and isa-is-19
		    (isa-get-screen-for-buffer-noselect (current-buffer)))))
    (turn-on-auto-fill)
    (setq fill-column (- (screen-width sc) 3))
    (erase-buffer)
    (setq fill-prefix "  ")
    (if isa-use-long-ruletables
	(mapcar 'ruletable-insert-group rules)
      (mapcar 'ruletable-insert-group-short rules))
    (auto-fill-mode nil)					; Fill off
    (if window-system
	(progn 
	  (or (face-differs-from-default-p 'ruletableGroupname sc)
	      (copy-face ruletableGroupname-default 'ruletableGroupname sc))
	  (ruletable-make-extents)))
    (goto-char (point-min))
    (ruletable-forward-word)))

(defun ruletable-toggle-long ()
  "Toggle isa-use-long-ruletables and update table accordingly"
  (interactive)
  (setq isa-use-long-ruletables (not isa-use-long-ruletables))
  (ruletable-make-table))

(defun ruletable-insert-group-short (rulegroup &optional long)
  (mapcar '(lambda (name)
	     (cond (name 
		    (insert (concat (if long "  " "  ")
				    name
				    (if long ""))))  ; was "\t".
		   (long
		    (newline))))
	  (cdr rulegroup))
  (newline))
;  (let ((fill-prefix ""))   - doesn't quite fix small bug resulting
;    (newline)))               in blank line above.  Why not?

(defun ruletable-insert-group (rulegroup)
  (insert (car rulegroup))				; heading
  (newline)
  (ruletable-insert-group-short rulegroup t))


(defun ruletable-make-extents ()
  (let (extent)
    (goto-char 0)
    (while (looking-at "[ \t\n]*\\([a-zA-Z][^ \n\t]*\\)[ \t\n]*")
      (setq extent (make-extent (match-beginning 1) (match-end 1)))
      (goto-char (match-beginning 1))
      (if (> (current-column) 0)
	  (set-extent-attribute extent 'highlight)
	(set-extent-face extent 'ruletableGroupname))
      (goto-char (match-end 0)))))


; not such a great idea
;
;(defvar ruletable-font-lock-keywords
;  '(("[ \t\n]*\\([a-zA-Z][^ \n\t]*\\)[ \t\n]*" 1 ruletableGroupname)))

;(defun ruletable-fontify-buffer ()
;  (setq font-lock-keywords ruletable-font-lock-keywords)
;  (font-lock-fontify-buffer))

  



;;; ===== Finding the rule that mouse or point is on =====

(defun ruletable-indicated-rule (event &optional nomove)
  (let* ((win    (event-window event))
	 (buffer (window-buffer win))
	 (p      (event-point event))
	 (extent (and p (extent-at p buffer 'highlight))))
    (if extent
	 (let* ((curbuf (current-buffer))
		(exs    (extent-start-position extent))
		(exe    (extent-end-position extent))
		result)
	   (if (not nomove) 
	       (set-window-point win exs))
	   (set-buffer buffer)
	   (setq result
		 (buffer-substring
		  (extent-start-position extent)
		  (extent-end-position extent)))
	   (set-buffer curbuf)
	   result))))

(defun ruletable-on-word-char ()
  (char-equal ?w (char-syntax (char-after (point)))))

(defun ruletable-pointed-rule ()
  "Return a rulename for point, or nil."
    (if (ruletable-on-word-char)
	(save-excursion
	  (let (start)
	    (forward-char)
	    (forward-word -1)				; Start of word
	    (if (bolp) nil				; mustn't be at start of line
	      (setq start (point))
	      (forward-word 1)
	      (buffer-substring start (point)))))))


;;; ========== Selecting (inserting) rules ==========

(defun ruletable-select-indicated-rule (event)
  "Select (insert into other buffer) the rule at the click-location.
If the point in the other buffer is on an identifier character,
then a comma is inserted before the rule name."
  (interactive "e")
  (let ((rule (ruletable-indicated-rule event)))
    (if rule
	(save-excursion 
	  (set-buffer (ruletable-pick-insert-buffer))
	  (ruletable-comma-insert rule)
	  (isa-display-buffer (current-buffer))))))

(defun ruletable-select-pointed-rule ()
  "Select (insert into other buffer) the rule that point is on.
If the point in the other buffer is on an identifier character,
then a comma is inserted before the rule name."
  (interactive)
  (let* ((rule (ruletable-pointed-rule))
	 (buf  (ruletable-pick-insert-buffer)))
    (if rule
	(save-excursion
	  (isa-display-buffer buf)
	  (set-buffer buf)
	  (ruletable-comma-insert rule)))))

(defun ruletable-pick-insert-buffer ()
  "Pick the buffer to do insertion in"
  ;; other-buffer always prefers non-visible buffers - this
  ;; is a mess:  
  ;;   (other-buffer (current-buffer))
  ;; Instead, simply use Isabelle buffer.
  (if (isa-buffer-active isa-buffer)
      isa-buffer
    (error "No associated Isabelle buffer")))

(defun ruletable-comma-insert (string)
  (isa-insert-as-if-selected
   (ruletable-add-comma string)))

(defun ruletable-add-comma (string)
  "Return STRING, possibly with a preceding comma."
  (if (> (point) (point-min))
	(progn
	  (backward-char)
	  (cond ((ruletable-on-word-char)
		 (forward-char)
		 (concat "," string))
		 (t (forward-char) string)))
    string))



;;; ========== Displaying rules ==========

(defun ruletable-show-indicated-rule (event)
  "Display the rule at the click-location, using prth."
  (interactive "e")
  (ruletable-show-rule (ruletable-indicated-rule event)))

(defun ruletable-show-pointed-rule ()
  "Display the rule that the point is on."
  (interactive)
  (ruletable-show-rule (ruletable-pointed-rule)))

(defun ruletable-show-rule (rule)
  "Show RULE in a temporary buffer using prth."
  (interactive "sRule name: ")
  (let* ((proc  (get-buffer-process isa-buffer))
	 (logic (save-excursion (set-buffer isa-buffer) isa-logic-name))
	 (cmd   (concat "(prth " rule "; ());"))
	 (text  (and rule proc (isa-send-string-catch-result proc cmd)))
	 (text2 (and text
		     (substring text 0 (string-match "^val" text)))))
    (cond (text2
	   (isa-set-pretty-margin proc (- (screen-width) 1))
	   (isa-show-in-temp-buffer "*Rule*" 
	     (concat "Rule " rule " in " logic " :\n\n" text2))
;	   (with-output-to-temp-buffer "*Rule*" 
;	     (concat "Rule " rule " in " logic " :\n\n" text2)
;	     (print-help-return-message))
	   (isa-set-default-pretty-margin proc))
	  ((null rule) nil) ; (error "No rule selected!"))
	  (t           (error "Can't find Isabelle process")))))


(defun isa-set-pretty-margin (proc marg)
  "Set Isabelle pretty printer margin"
  (isa-send-string-catch-result proc
   (concat "Pretty.setmargin " 
	   (int-to-string marg)
	   ";")))

(defun isa-default-pretty-margin ()
  "Default value for pretty printer margin"
  (-
   (let* ((pb (car-safe (isa-find-buffers-in-mode 'proofstate-mode)))
	  (wp (and pb (get-buffer-window pb)))
	  (ib (or wp (car-safe (isa-find-buffers-in-mode 'isa-mode))))
	  (wi (or wp (and ib (get-buffer-window ib)))))
     (if wi
	 (window-width wi)
       (window-width)))
   1))

(defun isa-set-default-pretty-margin (proc)
  "Set default value for Isabelle pretty printer margin"
  (interactive)
  (isa-set-pretty-margin proc (isa-default-pretty-margin)))




;;; ===== Startup =====

(add-hook 'isa-mode-startup-hook 
	   (function
	    (lambda () (isa-startup-function-for 'ruletable))))



(provide 'isa-ruletable)

;;; End of isa-ruletable.el
