;;; isa-rules.el - Rules, tactics and tacticals in Isabelle.
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-rules.el,v 1.2 1994/02/17 17:03:12 da Exp $
;;;


;;; ======== ML value completion ========


;; Naive completion: don't distinguish between values used
;; as commands and others.  Smarter would be to split into
;; tables for commands, tactics and tacicals.

(defconst isa-completion-list
  ;; It'd be good to have Isabelle make this during build
  '("quit" 
    "cd" "use" "use_thy" "time_use" "time_use_thy"
    "Pretty.setdepth" "Pretty.setmargin" "print_depth"
    "show_hyps" "show_types" "show_sorts"
    "print_exn"
    "goal" "goalw" "goalw_cterm" "premises"
    "by" "byev" 
    "result" "uresult"
    "chop" "choplev" "back" "undo"
    "pr" "prlev" "goals_limit"
    "proof_timing"
    "prove_goal" "prove_goalw" "prove_goalw_cterm"
    "push_proof" "pop_proof" "rotate_proof"
    "save_proof" "restore_proof"
    "read" "prin" "printyp"
    "topthm" "getgoal" "gethyps"
    "filter_goal" "compat_goal"

    ;; short cuts - should these be included?
    "ba" "br" "be" "bd" "brs" "bes" "bds"
    "fs" "fr" "fe" "fd" "frs" "fes" "fds"
    "bw" "bws" "ren"

    "resolve_tac" "eresolve_tac" "dresolve_tac" "forward_tac"
    "assume_tac" "eq_assume_tac"
    "match_tac" "ematch_tac" "dmatch_tac"
    "res_inst_tac" "eres_inst_tac" "dres_inst_tac" "forw_inst_tac"
    "rewrite_goals_tac" "rewrite_tac" "fold_goals_tac"
    "fold_goals_tac" "fold_tac"
    "cut_facts_tac" "subgoal_tac"

    ;; short cuts - should these be included?
    "rtac" "etac" "dtac" "atac" "ares_tac" "rewtac"

    ;; In general, I think rules should appear in rule tables, not here.
    "asm_rl" "cut_rl"  

    "flexflex_tac" "rename_tac" "rename_last_tac"
    "Logic.set_rename_prefix" "Logic.auto_rename"

    "compose_tac"

    "biresolve_tac" "bimatch_tac" "subgoals_of_brl" "lessb"
    "head_string" "insert_thm" "delete_thm" "compat_resolve_tac"

    "could_unify" "filter_thms" "filt_resolve_tac"

    ;; probably shouldn't be included:
    "tapply" "Tactic" "PRIMITIVE" "STATE" "SUBGOAL"

    "pause_tac" "print_tac"

    "THEN" "ORELSE" "APPEND" "INTLEAVE"
    "EVERY" "FIRST" "TRY" "REPEAT_DETERM" "REPEAT" ("REPEAT1")
    "trace_REPEAT"
    "all_tac" "no_tac"
    "FILTER" "CHANGED" "DEPTH_FIRST" "DEPTH_SOLVE"
    ("DEPTH_SOLVE_1") "trace_DEPTH_FIRST"
    "BREADTH_FIRST" "BEST_FIRST" "THEN_BEST_FIRST"
    "trace_BEST_FIRST"
    "COND" "IF_UNSOLVED" "DETERM" 
    
    "SELECT_GOAL" "METAHYPS"

    "ALLGOALS" "TRYALL" "SOMEGOAL" "FIRSTGOAL"
    "REPEAT_SOME" "REPEAT_FIRST" "trace_goalno_tac"

    ;; include primed versions of tacticals?

    ("EVERY1") ("FIRST1")

    "prth" "prths" "prthq"
    "RSN" "RLN" "RL"
    
    ;; that's it for now!
    ))





;;; ========== Logics and Rules ==========

(defvar isa-builtin-object-logic-names 
  '(("FOL") ("LK") ("ZF") ("CTT") ("HOL") 
    ("LCF") ("FOLP") ("Modal") ("Cube") ("CCL") ("Pure"))
  "alist of built-in object logic names (cdrs ignored)")

(defvar isa-user-theory-rules
  nil
  "*Appended to isa-theory-rules, see doc. for isa-theory-rules.")



;; These should be split up into a special directory
;; and auto-loaded as needed.  User logics could then
;; be added more easily.


;;; =============== Default rule lists ===============

(defconst isa-theory-rules
  '(("Pure")

    ("FOL"  
     ("Equality"     . ["refl" "sym" "trans" "subst" "ssubst"])
     ("Propositions" . ["conjI" "conjunct1" "conjunct2" "conjE" ()
			"disjI1" "disjI2" "disjE" ()
			"impI" "mp" "impE" ()
			"TrueI" "FalseE" ()
			"notI" "notE" ()
			"iffI" "iffE" "iffD1" "iffD2"])
     ("Quantifiers"  . ["allI" "spec" "allE" "all_dupE" ()
			"exI" "exE"  ()
			"ex1I" "ex1E"])
     ("Classical"    . ["excluded_middle"
			"disjCI" "exCI" "impCE" "iffCE"
			"notnotD" "swap"])
     ("Definitions"  . ["True_def" "not_def" "iff_def" "ex1_def"]))

    ("HOL"  
     ("Equality"     . ["refl" "sym" "trans" "subst" "ssubst"
			"box_equals" "arg_cong" "fun_cong"
			"eqTrueI" "eqTrueE"])
     ("Propositions" . ["conjI" "conjunct1" "conjunct2" "conjE" ()
			"disjI1" "disjI2" "disjE" ()
			"notI" "notE" ()
			"impI" "mp" "impE" ()
			"iffI" "iffE" "iffD1" "iffD2"])
     ("Quantifiers" . ["allI" "spec" "allE" "all_dupE" ()
		       "exI" "exE" ()
		       "ex1I" "ex1E" ()
		       "select_equality"])
     ("Classical"   . ["ccontr" "classical" "excluded_middle" ()
		       "disjCI" "exCI" "impCE" "iffCE" ()
		       "notnotD" "swap"])
     ("Definitions" . ["True_def" "All_def" "Ex_def" "False_def"
		       "not_def" "and_def" "or_def" "Ex1_def"
		       "Inf_def" "o_def" "if_def"]))

    ("ZF" 
     ("Quantifiers"   . ["ballI" "bspec" "ballE" "ball_cong" ()
			 "bexI" "bexCI" "bexE" "bex_cong"])
     ("Subsets"       . ["subsetI" "subsetD" "subsetCE"
			 "subset_refl" "subset_trans" ()
			 "PowI" "PowD" ()
			 "equalityI" "equalityD1" "equalityD2" "equalityE"])
     ("Comprehension" . ["ReplaceI" "ReplaceE" "RepFunI" "RepFunE" ()
			 "separation" ()
			 "CollectI" "CollectE" "CollectD1" "CollectD2"])
     ("Un,Int,-"      . ["UnI1" "UnI2" "UnCI" "UnE" ()
			 "IntI" "IntD1" "IntD2" "IntE" ()
			 "DiffI" "DiffD1" "DiffD2" "DiffE"])
     ("UN,INT"        . ["UnionI" "UnionE" "UN_I" "UN_E" ()
			 "InterI" "InterD" "InterE" "INT_I" "INT_E"])
     ("Finite-Sets"   . ["emptyE" "empty_subsetI" "equals0I" "equals0D" ()
			 "singletonI" "singletonE" ()
			 "consI1" "consI2" "consCI" "consE"])
     ("Succ,if,THE"   . ["succI1" "succI2" "succCI" "succE"
			 "succ_neq_0" "succ_inject" ()
			 "if_P" "if_not_P" ()
			 "the_equality" "theI"])
     ("Properties"    . ["mem_anti_sym" "mem_anti_refl" ()
			 "Union_upper" "Union_least" 
			 "Un_upper1" "Un_upper2" "Un_least" ()
			 "Inter_lower" "Inter_greatest"
			 "Int_lower1" "Int_lower2" "Int_greatest" ()
			 "Diff_subset" "Diff_contains" ()
			 "Collect_subset"])
     ("Pairs"         . ["pairing" "UpairI1" "UpairI2" "UpairE" ()
			 "fst" "snd" "split"
			 "Pair_inject1" "Pair_inject2" "Pair_inject" 
			 "Pair_neq_0" ()
			 "SigmaI" "SigmaE" "SigmaE2"])
     ("Relations"     . ["domainI" "domainE" "domain_subset" ()
			 "rangeI" "rangeE" "range_subset" ()
			 "fieldI1" "fieldI2" "fieldCI" 
			 "fieldE" "field_subset" ()
			 "imageI" "imageE" "vimageI" "vimageE"])
     ("Functions"     . ["lamI" "lamE" "lam_type" "beta" "eta" "lam_theI" ()
			 "fun_empty" "fun_single" 
			 "fun_disjoint_Un" "fun_disjoint_apply1" 
			 "fun_disjoint_apply2"])
     ("Axioms"        . ["Ball_def"  "Bex_def"
			 "subset_def" "extension"
			 "union_iff" "power_set"
			 "foundation" "replacement"]))

    ("LK"
     ("Structural"    . ["basic" "thinR" "thinL" "cut"])
     ("Equality"      . ["refl" "sym" "trans" "symL"])
     ("Propositions"  . ["TrueR" "iffR" "iffL" "True_def" "iff_def" ()
			 "conjR" "conjL" ()
			 "disjR" "disjL" ()
			 "impR" "impL" ()
			 "notR" "notL" ()
			 "FalseL"])
     ("Quantifiers"   . ["allR" "allL" "allL_thin" ()
			 "exR" "exL" "exR_thin" ()
			 "The"]))

    ("CTT"
     ("Equality"      . ["refl_type" "refl_elem" ()
			 "sym_type" "sym_elem" ()
			 "trans_type" "trans_elem" ()
			 "equal_types" "equl_typesL" ()
			 "subst_type" "subst_typeL" ()
			 "subst_elem" "subst_elemL" ()
			 "replace_type" "subst_eqtyparg" ()
			 "refl_ref" "red_if_equal" "trans_red"])
     ("N-type"        . ["NF" () 
			 "NI0" "NI_succ" "NI_succL" ()
			 "NE" "NEL" ()
			 "NC0" "NC_succ" "zero_ne_succ"])
     ("Prod-type"     . ["ProdF" "ProdFL" ()
			 "ProdI" "ProdIL" ()
			 "ProdE" "ProdEL" "subst_prodE" ()
			 "ProdC" "ProdC2"])
     ("Sum-type"      . ["SumF" "SumFL" ()
			 "SumI" "SumIL" "SumIL2" ()
			 "SumE" "SumEL" "SumE_fst" "SumE_snd" ()
			 "SumC" ()
			 "fst_def" "snd_def"])
     ("Plus-type"     . ["PlusF" "PlusFL" ()
			 "PlusI_inl" "PlusI_inlL" ()
			 "PlusI_inr" "PlusI_inrL" ()
			 "PlusE" "PlusEL" ()
			 "PlusC_inl" "PlusC_inr"])
     ("Eq-type"       . ["Eqf" "EqFL" "EqI" "EqE" "EqC"])
     ("F-type"        . ["FF" "FE" "FEL"])
     ("T-type"        . ["TF" "TI" "TE" "TEL" "TC"])
     ("Arith"         . ["add_def" "diff_def" "absdiff_def" 
			 "mult_def" "mod_def" "quo_def" ()
			 "add_typing" "addC0" "addC_succ" 
			 "add_assoc" "add_commute" ()
			 "mult_typing" "multC0" "multC_succ" 
			 "mult_assoc" "mult_commute" ()
			 "add_mult_dist" ()
			 "diff_typing" "diffC0" 
			 "diff_0_eq_0" "diff_succ_succ"
			 "diff_self_eq_0" "add_inverse_diff"
			 "add_assoc" "add_commute"])
     ("Rule-Lists"    . ["form_rls" "formL_rls" () 
			 "intr_rls" "intrL_rls" ()
			 "elim_rls" "elimL_rls" ()
			 "comp_rls" "basic_defs"]))

    ("LCF"
     ("Domains"       . ["eq_def" "less_trans" "less_ext" 
			 "mono" "minimal" "FIX_eq" ()
			 "ap_term" "ap_thm" "cong" "ext" ()
			 "Contrapos" ()
			 "eq_imp_less1" "eq_imp_less2" ()
			 "less_anti_sym" "less_ap_term" 
			 "less_ap_thm" "less_refl" "less_UU" ()
			 "UU_abs" "UU_app"])
     ("TR"            . ["tr_cases" "tr_induct" ()
			 "not_TT_less_FF" "not_FF_less_TT"
			 "not_TT_less_UU" "not_FF_less_UU" ()
			 "COND_UU" "COND_TT" "COND_FF" ()
			 "COND_cases" "COND_cases_iff" ()
			 "not_UU_eq_TT" "not_UU_eq_FF" "not_TT_eq_UU" ()
			 "not_TT_eq_FF" "not_FF_eq_UU" "not_FF_eq_TT"])
     ("Pairs"         . ["FST" "SND" "surj_pairing"])
     ("Strict-Sum"    . ["INL_DEF" "INR_DEF" ()
			 "INR_STRICT" "INL_STRICT" ()
			 "WHEN_UU" "WHEN_INL" "WHEN_INR" ()
			 "SUM_EXHAUSTION"])
     ("Void"          . ["void_cases"])
     ("Admiss."       . ["induct" ()
			 "adm_less" "adm_not_less" "adm_not_free" ()
			 "adm_subst" "adm_conj" "adm_disj" 
			 "adm_imp" "adm_all" ()
			 "adm_eq" "adm_not_eq_tr" "adm_not_not" ()
			 "not_eq_TT" "not_eq_FF" "not_eq_UU" ()
			 "induct2"]))

    ("Modal"
     ("s4"           . ["lstar0" "lstar1" "lstar2" ()
			"rstar0" "rstar1" "rstar2" ()
			"boxR" "boxL" ()
			"diaR" "diaL"])
     ("s43"          . ["lstar0" "lstar1" "lstar2" ()
			"rstar0" "rstar1" "rstar2" ()
			"boxL" "diaR" ()
			"pi1" "pi2"])
     ("Defs"         . ["strimp_def" "streqv_def"]))

    ("Cube"
     ("basic"        . ["s_b"  "strip_s" "strip_b" ()
			"app" "beta"])
     ("simple"       . ["pi_ss" "lam_ss"])
     ("L2"           . ["pi_ss" "pi_bs" "lam_ss" "lam_bs"])
     ("Lomega"       . ["pi_ss" "pi_bb" "lam_ss" "lam_bb"])
     ("LOmega"       . ["pi_ss" "pi_bb" "pi_bs" "lam_ss" "lam_bb" "lam_bs"])
     ("LP"           . ["pi_ss" "pi_sb" "lam_ss" "lam_sb"])
     ("LP2"          . ["pi_ss" "pi_sb" "pi_bs" 
			"lam_ss" "lam_sb" "lam_bs"])
     ("LPomega"      . ["pi_ss" "pi_sb" "pi_bb" 
			"lam_ss" "lam_sb" "lam_bb"])
     ("CC"           . ["pi_ss" "pi_bs" "pi_bb" "pi_sb"
			"lam_ss" "lam_bs" "lam_bb" "lam_sb"]))

    ("CCL"
     ("Ev-Sems"      . ["trueV" "falseV" "pairV" "lamV" ()
			"caseVtrue" "caseVfalse" "caseVpair" "caseVlam" ()
			"canonical" "substitute"])
     ("Rules"        . ["po_refl" "po_trans" "po_cong" "po_abstractn" ()
			"eq_iff" "PO_iff" "EQ_iff" ()
			"caseBtrue" "caseBfalse" "caseBpair" 
			"caseBlam" "caseBbot" ()
			"distinctness"])
     ("Defs"         . ["apply_def" "bot_def" "fix_def" ()
			"SIM_def" "POgen_def" "EQgen_def" 
			"PO_def" "EQ_def" ()
			"Dvg_def" "Trm_def"]))
    )
  "Table of logics and rule names for rule-table buffers.

The format is:
  A list of lists, one per theory (or logic).
  Each theory list consists of:  (THYNAME  RULEGROUP1 RULEGROUP2 ...)
  Each rule group constists of:  (RULEGROUPNAME . [rulenames...])
  Each rule-name is either a string or nil - nil represents a further
    (un-named) subgrouping of the rules.")

; add FOLP rules by extending 
(let ((folp '(("Proofs" . ["ieqI" "ieqE" ()
			   "prefl" "psym" "ptrans" ()
			   "idpeelB" ()
			   "fstB" "sndB" "pairEC" ()
			   "whenBinl" "whenBinr" "plusEC" ()
			   "applyB" "funEC" "specB" ()
			   "norm_eq" "NORM_iff"]))))
  (setcdr folp (cdr (assoc "FOL" isa-theory-rules)))
  (setq isa-theory-rules
	(cons (cons "FOLP" folp) isa-theory-rules)))

; add rules for user's theories
(nconc isa-theory-rules isa-user-theory-rules)




;;; ============ User-level functions ===============

(defun isa-add-theory-rules (rulelist)
  "Add or overwrite a rule table for a theory in isa-theory-rules.
See doc. of isa-theory-rules for format of argument."
  (let ((oldthy (assoc (car rulelist) isa-theory-rules)))
    (if oldthy
	(setcdr oldthy (cdr rulelist))
      (setq isa-theory-rules
	    (nconc isa-theory-rules (list rulelist))))))

(defun isa-add-theory-rulegroup (thy rulegroup)
  "Add or overwrite a rulegroup for a theory in isa-theory-rules."
  (let* ((oldthy  (assoc thy isa-theory-rules))
	 (oldgrps (and oldthy (assoc (car rulegroup) (cdr oldthy)))))
    (if oldthy
	(if oldgrps
	    (setcdr oldgrps (cdr rulegroup))
	  (nconc oldthy (list rulegroup)))
      (setq isa-theory-rules
	    (nconc isa-theory-rules 
		   (list (cons thy (list rulegroup))))))))







(provide 'isa-rules)

;;; End of isa-rules.el
