;;; isa-proofstate.el - Buffer holding proof state in Isabelle mode
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-proofstate.el,v 1.12 1994/03/11 19:58:12 da Exp $
;;;


;;; DESIRABLE CHANGES:
;;;  - tidy up!
;;;  - use a ring of previous proofstates for paging backwards,
;;;    to speed up by avoiding querying process.
;;;  - attempt to increase goals_limit on movement off the 
;;;    bottom of subgoal list?


(require 'isa-mode)



;;; ============ Proofstate Mode ============

(defvar proofstate-mode-map nil)

(cond (window-system
       (make-face 'proofstateGoal)
       (defvar proofstateGoal-default 'bold-italic
	 "*Face name for goal in proofstate buffer.")
       (make-face 'proofstateSubgoalNumber)
       (defvar proofstateSubgoalNumber-default 'bold
	 "*Face name for subgoal numbers in proofstate buffer.")))


(or proofstate-mode-map
  (let ((map (make-keymap)))
      (suppress-keymap map)
      (isa-clear-mouse-bindings map)
      (define-key map "i" 'isa-select-isa-buffer)
      (define-key map "h" 'describe-mode)
      (define-key map "\C-f" 'proofstate-next-level)
      (define-key map "\C-b" 'proofstate-previous-level)
      (define-key map "\C-n" 'proofstate-next-subgoal)
      (define-key map "\C-p" 'proofstate-previous-subgoal)
      (define-key map 'button1 'proofstate-motion)
      (define-key map " " 'proofstate-refresh)
      (define-key map "\C-m" 'proofstate-resize-and-refresh)
      (setq proofstate-mode-map map)))


; (defvar proofstate-level-string "Level 0")

(defun proofstate-mode ()
  "Major mode for Isabelle Proof-State buffers.

The cursor keys move up and down the subgoal list and left and right
through previous levels.  The subgoal that the cursor appears on will
be used in the menu tactic commands (i.e. isa-assume_tac, etc).

Commands:
\\{proofstate-mode-map}"
  (setq buffer-read-only t)
  (setq major-mode 'proofstate-mode)
  (set-syntax-table isa-thy-mode-syntax-table)
; spoils C-xC-b and gives interleaving o/p problem
;  (setq mode-name 'proofstate-level-string)  
  (setq mode-name "Proof-State")
  (setq mode-line-buffer-identification '("Proof: %17b"))
  (use-local-map proofstate-mode-map)
  (put 'proofstate-mode 'mode-class 'special)
  (isa-remove-menubar-if-multiple-screen-mode)
; (setq mode-motion-hook 'proofstate-motion)
  )


(defun proofstate ()
  "Activate a proofstate buffer for the Isabelle process in the current buffer."
  (interactive)
  (let*
      ((proofstate-buffer
	;; Make proofstate buffer
	(save-excursion
	  (set-buffer (isa-create-new-associated
		       'proofstate))
	  (proofstate-mode)
	  (proofstate-refresh)		        
	  (current-buffer)))
       (subgoal-tracker (proofstate-track-denoted-subgoal proofstate-buffer)))

    ;; Set weeder fn and subgoal change hook
    (setq isa-weed-output-function 'proofstate-weeder)
    (add-hook 'isa-set-denoted-subgoal-hook subgoal-tracker)

    ;; Highlight buffer if running on a window system
    (if window-system
	(let ((sc (isa-get-screen-for-buffer-noselect proofstate-buffer)))
	  (or (face-differs-from-default-p 'proofstateSubgoalNumber sc)
	      (copy-face proofstateSubgoalNumber-default
			 'proofstateSubgoalNumber sc))
	  (or (face-differs-from-default-p 'proofstateGoal sc)
	      (copy-face proofstateGoal-default
			 'proofstateGoal sc))
	  (proofstate-set-sizes sc)))
    
    ;; 
    (isa-display-buffer proofstate-buffer t)))		; show it.

(defun proofstate-set-sizes (sc)
  "Set Isabelle's printer margin and goals limit according to screen size"
  (let* ((proc   (get-buffer-process (current-buffer)))
	 (marg   (concat "Pretty.setmargin "
			 (isa-int-to-ml-string (- (screen-width sc) 2))
			 ";"))
	 (levl   (concat "goals_limit := "
			 (isa-int-to-ml-string (- (screen-height sc) 4))
			 ";")))
    (isa-send-string-catch-result proc marg)
    (isa-send-string-catch-result proc levl)))
	 
(defun proofstate-refresh (&optional level)
  "Refresh current proofstate buffer."
  (interactive "P")
  ;; allow interactive use w/o arg.
  (setq level (and level (prefix-numeric-value level)))	
  (let* ((proc     (get-buffer-process isa-buffer))
	 (cmd      (if level 
		       (concat "prlev (" (isa-int-to-ml-string level) ");")
		     "pr();"))
	 (prf-text (isa-send-string-catch-result proc cmd))
	 (sg       (save-excursion (set-buffer isa-buffer) isa-denoted-subgoal)))
    (proofstate-strip-text prf-text)
    ;; Try to set marked subgoal to isa-denoted-subgoal, or else 1.
    (if (proofstate-goto-subgoal sg t)
	nil
      (if (proofstate-goto-subgoal 1 t)
	  (save-excursion
	    (set-buffer isa-buffer)
	    (setq isa-denoted-subgoal 1))))))

(defun proofstate-resize-and-refresh ()
  (interactive)
  (save-excursion
    (set-buffer isa-buffer)
    (proofstate-set-sizes (selected-screen)))
  (proofstate-refresh))



;;; ========== Updating the proofstate buffer ==========

(defun proofstate-weeder (outbuf)
  "Weeder for proofstates."				; Assume in isabelle buffer.
  (let ((proofstate-buffer
	 (car-safe (isa-find-associated 'proofstate))))
    (if (isa-buffer-active proofstate-buffer)	        ; if active, 
	(proofstate-weed-text outbuf proofstate-buffer)	; strip proof text.
							; If buffer killed, reset
      (setq isa-weed-output-function                    ; weeder and call it. 
	    'isa-weed-output-default)
      (isa-weed-output-default outbuf))
    t))							; Flag for showing prompt.

(defconst proofstate-pattern 
  "\\(.\\|\n\\)*\\(^Level [0-9]+$\\)\\(.\\|\n\\)*\\(^val\\)")

(defun proofstate-weed-text (outbuf proofstate-buffer)
  "Used during output filtering to remove proof state from output."
  (save-excursion
    (let ((data (match-data))
	  (ibuf (current-buffer)))
      (set-buffer outbuf)
      (goto-char (point-min))
      (unwind-protect					; strip proof text
	  (if (re-search-forward proofstate-pattern nil t)
	      (let* ((proof-start (match-beginning 2))
		     (proof-end   (match-beginning 4))
		     (proof       (buffer-substring proof-start proof-end)))
		(set-buffer ibuf)
		(proofstate-update-buffer proof proofstate-buffer)
		(set-buffer outbuf)
		(delete-region proof-start proof-end)))
	(store-match-data data)))))

(defun proofstate-strip-text (text)
  "Strip proof state from TEXT and put it in current proofstate buffer."
  (save-excursion
    (let ((data (match-data)))
      (unwind-protect					; strip proof text
	  (if (string-match proofstate-pattern text)
	      (let* ((proof-start (match-beginning 2))
		     (proof-end   (match-beginning 4))
		     (proof       (substring text proof-start proof-end)))
		(proofstate-update-buffer proof (current-buffer))))
	(store-match-data data)))))

(defun proofstate-update-buffer (newstate proofstate-buffer)
  "Reset buffer PROOFSTATE-BUFFER with NEWSTATE."
  (let (csb (bf (current-buffer)))
    (set-buffer proofstate-buffer)
    (let ((buffer-read-only nil))
      (setq csb (proofstate-current-subgoal))
      (erase-buffer)
      (insert newstate))
    (if window-system
	(proofstate-make-extents))
;; removed because of possibility of output interleaving.
;; really need to trigger this at end of process o/p somehow.
;    ;; Set mode-line level indicator
;    (let ((lev (proofstate-current-level)))
;      (setq proofstate-level-string
;	    (concat "Level " 
;		   (if lev (int-to-string lev)
;		     "?"))))
    ;; Try and preserve point position in subgoal list
    ;; (It may fail if you apply a tactic to a subgoal above the cursor).
    (proofstate-goto-subgoal-near csb)
    (isa-update-window-point)
    (isa-display-buffer (current-buffer))
    (set-buffer bf)))

(defun proofstate-make-extents ()
  "Do highlighting in proofstate buffer.  Only for Emacs 19's in window sys."
  (goto-line 2)
  (let ((goalstart (point)) extent)
    (if (re-search-forward "^ 1\\.\\|^No subgoals" nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (setq extent (make-extent goalstart (1- (point))))
	  (set-extent-face extent 'proofstateGoal)
	  (while (re-search-forward "^ [0-9]+\\. " nil t)
	    (setq extent (make-extent (match-beginning 0) (match-end 0)))
	    (set-extent-face extent 'proofstateSubgoalNumber))))))



;;; ========== Browsing levels ==========

(defun proofstate-displayed-level ()
  (save-excursion
    (goto-char 0)
    (cond ((looking-at "Level [0-9]")
	   (goto-char 7)	
	   (skip-chars-forward "0-9")
	   (string-to-int (buffer-substring 6 (point))))
	  (t (proofstate-current-level)))))


(defun proofstate-current-level ()
  (let* ((proc (get-buffer-process isa-buffer))
	 (cmd  
"let val a = !goals_limit in (goals_limit:=0;pr();goals_limit:=a) end;")
	 ; my other way of doing this was "choplev ~1", but raising an
	 ; exception might not be so nice.

	 ;; expects "Level x"
	 (txt  (isa-send-string-catch-result proc cmd)))
    (string-to-int (substring txt 6))))

(defun proofstate-next-level (&optional arg)
  (interactive "p")
  (proofstate-refresh 
   (min (+ arg (proofstate-displayed-level))
	(proofstate-current-level))))

(defun proofstate-previous-level (&optional arg)
  (interactive "p")
  (proofstate-refresh
   (max (- (proofstate-displayed-level) arg) 0)))




;;; ====== Movement through subgoals ======


(defconst proofstate-subgoal-regexp "^ [0-9]+\\.")

(defun proofstate-previous-subgoal ()
  (interactive)
  (re-search-backward proofstate-subgoal-regexp nil t)
  (proofstate-set-denoted-subgoal))

(defun proofstate-next-subgoal ()
  (interactive)
  (forward-char)
  (if (re-search-forward proofstate-subgoal-regexp nil t)
      (goto-char (match-beginning 0))
    (backward-char))
  (proofstate-set-denoted-subgoal))

(defun proofstate-goto-subgoal (subgoal &optional noerror)
  "Goto subgoal number SUBGOAL, error if impossible unless NOERROR non-nil.
Return t on success."
  (interactive "n")
  (cond
   ((save-excursion
      (goto-char 0)
      (re-search-forward (concat "^ " (int-to-string subgoal) "\\.") nil t))
    (goto-char (match-beginning 0))
    (proofstate-set-denoted-subgoal)
    t)
   (noerror nil)
   (t	    (error "Can't see subgoal %d" subgoal))))

(defun proofstate-goto-subgoal-near (subgoal)
  "Try to goto subgoal SUBGOAL, if can't choose nearest one."
  (cond ((and subgoal (proofstate-goto-subgoal subgoal t)) nil)
	(subgoal 
	 ;; can't find same number, so go to last
	 (goto-char (point-max))
	 (or (proofstate-previous-subgoal)
	     ;; or top if no subgoals
	     (goto-char 0)))
	(t
	 ;; If no previous position, go to first (if any).
	 (goto-char 0)
	 (proofstate-next-subgoal))))

(defun proofstate-current-subgoal ()
  (if (looking-at " [0-9]+")
      (save-excursion
	(forward-char)
	(string-to-int (buffer-substring 
			(point) 
			(save-excursion
			  (skip-chars-forward "0-9") 
			  (point)))))
    1))

(defun proofstate-set-denoted-subgoal ()
  (let ((sg (proofstate-current-subgoal))
	(bf (current-buffer)))
    (set-buffer isa-buffer)
    (setq isa-denoted-subgoal sg)
;    (message "Working subgoal: %d" isa-denoted-subgoal)
    (set-buffer bf)))

;; Really there should be just one variable controlling the
;; denoted subgoal: we should trash the functions that determine
;; it by looking at the proofstate buffer, and use 
;; isa-denoted-subgoal in the Isabelle buffer instead.
;; Current duplicity is a bit of a mess.

(defun proofstate-track-denoted-subgoal (pf-buf)
  "For making values for isa-set-denoted-subgoal-hook."
  (` (lambda ()
	(if (isa-buffer-active (, pf-buf))
	      (let ((sg isa-denoted-subgoal))
		(save-excursion
		  (set-buffer (, pf-buf))
		  (proofstate-goto-subgoal-near sg)
		  (isa-update-window-point)))))))




;;; this doesn't work quite yet (perhaps in later version of Lucid?)
(defun proofstate-motion (event)
  "For use as the value of `mode-motion-hook' move point with mouse."
  (interactive "e")
  (let* ((window (event-window event))
	 (screen (if window (window-screen window) (selected-screen)))
	 (buffer (and window (window-buffer window)))
	 (point  (and buffer (event-point event)))
	 (curwin (selected-window))
	 (sbgl	 (save-excursion
		   (set-buffer buffer)
		   (if point
		       (progn
			 (goto-char point)
			 (beginning-of-line)
			 (and (looking-at " [0-9]+.") 
			      (setq point (point))))))))
    (cond (sbgl
	   (select-window window)
	   (goto-char sbgl)
	   ;; (sit-for 1) successfully updates screen but leads to
	   ;; an overflow of recursive calls here!  How else can
	   ;; I do this?  Nothing else seems to allow a proper update
	   ;; - it seems *amazingly* difficult to alter (point) in this way.
	   (sit-for 1)
	   (select-window curwin)))))



;;; Startup

(add-hook 'isa-mode-startup-hook 
	   (function
	    (lambda () (isa-startup-function-for 'proofstate))))


(provide 'isa-proofstate)

;;; End of isa-proofstate.el
