;;; isa-listener.el - Listener buffer for Isabelle mode
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-listener.el,v 1.5 1994/02/16 20:50:31 da Exp $
;;;

(require 'isa-mode)


;;;;

(defvar listener-copy-pred 'listener-copy-all)        ; Copy predicate

(defun listener-copy-all (str) t)

(defun listener ()
  "Activate a listener buffer for the Isabelle process in the current buffer."
  (interactive)
  (isa-display-buffer					; create and display.
   (save-excursion
     (set-buffer (isa-create-new-associated 
		  'listener))
     (listener-mode)
     (let ((buffer-read-only nil))
       (erase-buffer))
     (current-buffer))
   t)
  (setq isa-input-sentinel 'listener-copy))		; Set sentinel


(defun listener-mode ()
  "Mode for Isabelle listener buffers."
  (fundamental-mode)
  (set-syntax-table isa-thy-mode-syntax-table)
  (setq major-mode 'listener-mode)
  (setq mode-name "Listener")
  (put 'listener-mode 'mode-class 'special)
  (isa-remove-menubar-if-multiple-screen-mode))


(defun listener-copy (str)
  (funcall (default-value 'isa-input-sentinel) str)	; Call usual sentinel
  (let ((listener-buffer
	 (car-safe (isa-find-associated 'listener))))
    (if (isa-buffer-active listener-buffer)
	(save-excursion
	  (set-buffer listener-buffer)			; copy lines satisfying 
	  (if (funcall listener-copy-pred str)		; predicate.
	      (progn
		(goto-char (point-max))			; should be at end
		(isa-insert-as-if-selected 
		 (concat str "\n")))))
      (setq isa-input-sentinel 
	  (default-value 'isa-input-sentinel)))))	; Change back to usual sentinel
							; if listener killed.
   



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Following is dead.
;;;



;; NB:- This doesn't work with listener as input-sentinel
;;      it needs listener as filter again.

(defun listener-copy-by-undo (proc str)
  "Only accept lines that look like tactics, and that don't cause ERROR.
undo() and choplev(n) are interpreted simplistically."
  (let ((accept nil))
    (accept-process-output proc)                           ; wait for output
    (save-excursion
      (goto-char comint-last-input-start)
      ;; NB:- should save match data here.
      (cond ((looking-at "[ \t]*undo();")                 ; in case of "undo"
	     (listener-delete-last-lines 1))
	    ((looking-at "[ \t]*choplev.*\\([0-9]+\\)")   ; in case of "choplev"
	     (listener-delete-last-lines
	      (string-to-int (buffer-substring
			      (match-beginning 1)
			      (match-end 1)))))
	    ((looking-at "[ \t]*b;")
	     (goto-char comint-last-input-end)
	     (if (not (looking-at "\\(.\\|\n\\)*ERROR"))
		 (setq accept t)))))
    accept))

(defun listener-delete-last-lines (num)
  (set-buffer listener-buffer)
  (goto-char (point-max))
  (forward-line (- num))
  (delete-region (point) (point-max)))


(defconst listener-proof-start-regexp
  "^val.*goal\\|^goal"  "Start of proof in listener pattern")

(defconst listener-proof-end-regexp
  "^val.*result\\|^result\\|^uresult" "End of proof in listener pattern")

(defun listener-last-proof (&optional num)
  "Return the last (or NUMth last) proof from a listener buffer as a string."
  (let ((listener-buffer (car-safe (isa-find-buffers-in-mode 'listener-mode))))
    (if listener-buffer
	(save-excursion
	  (set-buffer listener-buffer)
	  (goto-char (point-max))
	  (let* ((start (and (re-search-backward 
			      listener-proof-start-regexp nil t num)
			     (point)))
		 ;; NB:- separating start and end means 
		 ;; confusion with incomplete proofs.
		 (end   (if start
			    (progn
			      (re-search-forward 
			       listener-proof-end-regexp nil t)
			      (end-of-line) 
			      (point)))))
	    (if end 
		(buffer-substring start end)
	      (error "Can't find listener proof"))))
      (error "Can't find listener buffer"))))

;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Startup

(add-hook 'isa-mode-startup-hook 
	   (function
	    (lambda () (isa-startup-function-for 'listener))))


(provide 'isa-listener)

;;; End of isa-listener.el
