;;; isa-display.el - Display manipulation for Isabelle mode.
;;; 
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-display.el,v 1.12 1994/03/11 19:50:06 da Exp $
;;;



(require 'isa-load)
(require 'isa-screen)



;;; ========== User display options ==========

(defvar isa-multiple-screen-mode
  (and isa-is-19 window-system)
  "*If non-nil, use multiple Emacs 19 frames (aka "screens"). 
Setting this to nil in Emacs 19 varieties restricts Isabelle-interaction 
mode to use a single screen.")

(defvar isa-default-menubar 
  nil
  "*Base menubar for Isabelle mode.")

(defvar isa-use-long-ruletables 
  isa-multiple-screen-mode
  "*If non-nil, use long-form of ruletables, which include section headings.
Probably best set to nil for single-screen working.")


;;; ============ Display properties ============

(defvar isa-multi-screen-display-props
  '((proofstate-mode     (screen-name . proofstate))
    (listener-mode       (screen-name . listener))
    (ruletable-mode      (screen-name . ruletable))
    (isa-mode            (screen-name . isabelle))
    (ruletable           (instance-limit . 3))
    (ruletable 		 (screen-defaults .
			   ((top . 590)   (left . 5)
			    (height . 18) (width . 60)
			    (menu-bar-lines . 0)
			    (minibuffer . nil)				
			    )))
    (isabelle            (screen-defaults .             
			  ((top .  0)    (left .  0)        
			   (height . 35) (width . 80)   
			   (pointer . "xterm")          
			   )))
    (listener            (screen-defaults .
			  ((top . 620)   (left . 530)
			   (height . 7)  (width . 65)
			   (menu-bar-lines . 0)
			   (minibuffer . nil)
			   )))
    (proofstate          (screen-defaults .
			  ((top .  30)    (left . 700)
			   (height . 35)  (width . 50)
			   (menu-bar-lines . 0)
			   (minibuffer . nil)
			   (vertical-scroll-bars . nil)
			   ))))
  "Display properties for Isabelle buffers in multiple screen mode.")


(defvar isa-associated-screen-names
   '(isa-mode listener-mode proofstate-mode ruletable-mode))

(defvar isa-single-screen-display-props
  '((proofstate-mode     (window-height . 8))    ; height for single mode
    (listener-mode       (window-height . 5))
    (ruletable-mode      (window-height . 15))
    (ruletable-mode      (shrink-to-fit . t)))
  "Display properties for Isabelle buffers in single screen mode.")



;;; ========== Buffer names ==========

(defvar isa-buffer-names
  '((listener   . "listener")
    (proofstate . "proofstate")
    (ruletable  . "rules")))

(defun isa-buffer-name-for (sym)
  (if (and (boundp 'isa-logic-name) isa-logic-name)
      (concat "*" isa-logic-name "-" (cdr (assq sym isa-buffer-names))  "*")
    (error "Not in Isabelle buffer")))

(defun isa-name (logic)
  (concat "*" logic "*"))



;;; ========== Setting display properties ==========

(defun isa-toggle-use-19 (&optional set)
  "Toggle use of Emacs 19 features, or switch on if arg non-nil."
  (interactive)
  (setq isa-multiple-screen-mode 
	(and isa-is-19 window-system
	     (or set
		 (not isa-multiple-screen-mode))))
  (cond (isa-multiple-screen-mode
	 (isa-remove-display-props isa-single-screen-display-props)
	 (isa-set-display-props isa-multi-screen-display-props))
	(t
	 (isa-remove-display-props isa-multi-screen-display-props)
	 (isa-set-display-props isa-single-screen-display-props))))

(defun isa-set-display-props (props)
  (let ((setprops
	 '(lambda (symprops)
	    (mapcar '(lambda (p) (put (car symprops) 
				      (car p) 
				      (cdr p))) 
		    (cdr symprops)))))
    (mapcar setprops props)))

(defun isa-remove-display-props (props)
  (let ((remprops
	 '(lambda (symprops)
	    (mapcar '(lambda (p) (remprop (car symprops) (car p))) 
		    (cdr symprops)))))
    (mapcar remprops props)))
	 


;;; ===== Locating screens and buffers ===== 

(defun isa-find-screens (screen-name-sym)
  (if isa-multiple-screen-mode
    (let ((scs-left (screen-list)) scs-got)
      (while scs-left
	(if (eq screen-name-sym (intern (screen-name (car scs-left))))
	    (setq scs-got (nconc scs-got (list (car scs-left)))))
	(setq scs-left (cdr scs-left)))
      scs-got)))

(defun isa-find-buffers-in-mode (mode &optional buflist)
  "Return a list of the buffers in the buffer list in major-mode MODE."
  (save-excursion
    (let ((bufs-left (or buflist (buffer-list))) bufs-got)
      (while bufs-left
	(if (isa-buffer-active (car bufs-left))
	    (progn
	      (set-buffer (car bufs-left))
	      (if (eq mode major-mode)
		  (setq bufs-got (nconc bufs-got (list (car bufs-left)))))))
	(setq bufs-left (cdr bufs-left)))
      bufs-got)))

(defun isa-buffer-active (buf)
  "Test to see if a buffer is active"
  (and (bufferp buf) (buffer-name buf)))



;;; ========== Temporary buffer management ==========

;;; <should really change this to use regular method, because
;;;  that can be customised via hook>
;;; <another possibility: use comint's "space to flush" routines>

(make-variable-buffer-local 'isa-temp-window)
(defvar isa-temp-window nil
  "Flag indicating status of temporary window associated with current buffer")

(defun isa-show-in-temp-buffer (bufname text)
  (setq isa-temp-window (or isa-temp-window
			    (if (one-window-p t) 'one-window t)))
  (with-output-to-temp-buffer bufname 
    (princ text)
    (princ "\n(Hit q to remove)")))
;  (message (concat "Hit q to remove buffer " bufname)))

(defun isa-remove-temp-buffer ()
  (interactive)
  (if isa-temp-window
      (let ((itw isa-temp-window)
	    (sw  (selected-window)))
	(setq isa-temp-window nil)
	(if (eq itw 'one-window)
	    (if pop-up-windows
		(delete-other-windows)
	      (switch-to-buffer (other-buffer)))
	  (progn
	    (switch-to-buffer-other-window (other-buffer))
	    (select-window sw))))))


;;; ========== Single screen mode  ==========

(defun isa-one-screen-set-format ()   
  "Set single screen format for the current Isabelle buffer."
  (interactive)
  (delete-other-windows)
  (let ((bufs isa-associated-buffers))
    ;; Squeeze as many buffers as will fit according to their
    ;; window-height property, which is treated as a minimal height.
    (condition-case ()
	(while bufs
	  (if (and (isa-buffer-active (car bufs))
		   (isa-one-screen-height (car bufs)))
	      (save-excursion
		(split-window-vertically (isa-one-screen-height (car bufs)))
		(set-window-buffer (selected-window) (car bufs))
		(other-window 1)))
	  (setq bufs (cdr bufs)))
      (args-out-of-range nil)))
  (set-window-buffer (selected-window) (current-buffer))
  ;; After squeezing on windows, we shrink to fit those that allow it,
  ;; but make no attempt to squeeze more on.
  (walk-windows 'isa-one-screen-shrinker))

(defun isa-one-screen-height (buf)
  "Return the height BUF should have in single-screen format."
  (get (save-excursion (set-buffer buf) major-mode)
       'window-height))

(defun isa-one-screen-shrinker (win)
  "Shrink window if buffer it displays has 'shrink-to-fit of t."
  (if (get (save-excursion (set-buffer (window-buffer win)) major-mode)
	   'shrink-to-fit)
      (shrink-window-if-larger-than-buffer win)))


;;; ========== Multiple screen mode ============

;;; See isa-screen.el for get-screen-for-buffer.

(defun isa-get-screen-for-buffer-noselect (buffer)
  (get-screen-for-buffer-noselect buffer))

(defun isa-remove-menubar-if-multiple-screen-mode ()
  (if isa-multiple-screen-mode
      (set-buffer-menubar nil)))

(defun isa-select-isa-buffer ()
  "Select the isabelle buffer associated with the current buffer."
  (interactive)
  (if (isa-buffer-active isa-buffer)
      (if isa-multiple-screen-mode
	  (isa-select-buffer isa-buffer) ; or perhaps nothing?
	(switch-to-buffer-other-window isa-buffer))))


;;; ===== Displaying and selecting buffers =====

(defun isa-display-if-active (buffer)
  "Show BUFFER if it is an active buffer, in the right place."
  (if (isa-buffer-active buffer)
      (isa-display-buffer buffer t)))

(defun isa-display-buffer (buffer &optional raise)
  "Ensure that BUFFER is displayed.
If raise is non-nil, make sure it's in the right place too."
  (if (and (not raise) (get-buffer-window buffer t))
      nil
    (if isa-multiple-screen-mode
	;; Emacs 19 version...
	(let ((sc (selected-screen)))
	  (save-excursion
	    (isa-select-buffer buffer raise)
	    (select-screen sc)))
	  
      ;; Emacs 18 version
      (let ((curbuf  (current-buffer))
	    ;; find an Isabelle buffer
	    (isabuf  (or (if (eq major-mode 'isa-mode) (current-buffer))
			 (if (boundp 'isa-buffer) isa-buffer)
			 (isa-find-buffers-in-mode 'isa-mode))))
	;; format the screen using it
	(set-buffer isabuf)
	(isa-one-screen-set-format)
	;; if not displayed now, use display-buffer.
	(if (not (get-buffer-window buffer))
	    (display-buffer buffer))
	;; if possible, select window on original buffer
	(if (get-buffer-window curbuf)
	    (select-window (get-buffer-window curbuf))
	  (set-buffer curbuf))))))

(defun isa-select-buffer (buffer &optional raise)
  "Display and select buffer on a suitable screen, without splitting windows."
  (interactive "B")
  (if isa-multiple-screen-mode
      (let ((sc (isa-get-screen-for-buffer-noselect buffer)))
	(select-screen sc)
	(if raise (raise-screen sc))
	(delete-other-windows)))
  (let ((pop-up-windows nil))
    (switch-to-buffer buffer))
  (sit-for 0))


;;; ========== Updating window point ==========

(defun isa-insert-as-if-selected (str)
  "Insert STR into the current buffer, as if the buffer's window was selected."
  (insert str)
  (isa-update-window-point))

(defun isa-update-window-point ()
  "Update a window point, if any, for current buffer, to match point."
  (let ((win (get-buffer-window				; Look everywhere for a window
	      (current-buffer) t)))
    (if win
	(set-window-point win (point)))			; update window point
    (sit-for 0)))					; update display
							; (doesn't work)


;;; Initialisation

(isa-toggle-use-19 isa-multiple-screen-mode)

(provide 'isa-display)

;;; End of isa-display.el
