;;; isa-cmd.el - User-level commands for Isabelle mode.
;;;
;;; Author:  David Aspinall <da@dcs.ed.ac.uk>
;;;
;;; $Id: isa-cmd.el,v 1.4 1994/03/11 19:46:27 da Exp $
;;;

;;; DESIRED CHANGES:
;;;   - Change key bindings to longer but more logical formats,
;;;     using different maps for each grouping.  For example,
;;;     tactics begin with C-c C-t <key>, etc.


(require 'isa-load)
(require 'isa-proc)
(require 'isa-ml-comp)

;;; ========== User Options ==========

(defvar isa-dont-query-quit nil
  "*If non-nil, don't query the user before issuing isa-quit-command")


;;; =============== Command Table ===============

(defconst isa-commands-table 
'(("Tactic" .
   (("\C-c\C-r" isa-resolve_tac
     "resolve"  
     (isa-insert "by (resolve_tac [%p] %s);"))
    ("\C-c\C-p" isa-resolve_tac-prems 
     "resolve prems" 
     (isa-insert  "by (resolve_tac prems %s);%n"))
    ("\C-c\C-a" isa-assume_tac
     "assume"   
     (isa-insert  "by (assume_tac %s);%n"))
    ("\C-c\C-e" isa-eresolve_tac      
     "eresolve" 
     (isa-insert "by (eresolve_tac [%p] %s);"))
    ("\C-c\C-d" isa-dresolve_tac
     "dresolve" 
     (isa-insert "by (dresolve_tac [%p] %s);"))
    ("\C-c\C-f" isa-forward_tac
     "forward"  
     (isa-insert "by (forward_tac [%p] %s);"))
    ()
    ("\C-cm"    isa-match_tac
     "match"    
     (isa-insert "by (match_tac [%p] %s);"))
    ("\C-ca"    isa-eq_assume_tac
     "eq_assume" 
     (isa-insert "by (eq_assume_tac %s);%n"))
    ("\C-cM"    isa-ematch_tac
     "ematch"   
     (isa-insert "by (ematch_tac [%p] %s);"))
    ("\C-cd"    isa-dmatch_tac
     "dmatch"
     (isa-insert "by (dmatch_tac [%p] %s);"))
    ()
    ("\C-c\C-w" isa-rewrite_goals_tac
     "rewrite_goals" 
     (isa-insert "by (rewrite_goals_tac [%p] %s);"))
    ("\C-cw"    isa-rewrite_tac
     "rewrite"
     (isa-insert "by (rewrite_tac [%p] %s);"))
    ("\C-cf"    isa-fold_goals_tac
     "fold_goals" 
     (isa-insert "by (fold_goals_tac [%p] %s);"))
    ("\C-cc"    isa-cut_facts_tac
     "cut_facts" 
     (isa-insert "by (cut_facts_tac [%p]  %s);"))
    ("\C-cC"    isa-cut_facts_tac-prems
     "cut_facts prems" 
     (isa-insert "by (cut_facts_tac prems %s);%n"))))
  ("Prover" .
   (("\M-s"     isa-simp_tac
     "simp" 
     (isa-insert "by (simp_tac %l_ss %s);%n"))
    ("\M-S"     isa-asm_simp_tac 
     "asm_simp" 
     (isa-insert "by (asm_simp_tac %l_ss %s);%n"))
    ("\M-\C-s"  isa-asm_full_simp_tac 
     "asm_full_simp" 
     (isa-insert "by (asm_full_simp_tac %l_ss %s);%n"))
    ()
    ("\M-f"     isa-fast_tac 
     "fast"
     (isa-insert "by (fast_tac %l_cs %s);%n"))
    ("\M-b"     isa-best_tac 
     "best" 
     (isa-insert "by (best_tac %l_cs %s);%n"))
    ()
    (nil         isa-contr_tac 
     "contr"
     (isa-insert "by (contr_tac %s);%n"))
    (nil         isa-mp_tac 
     "mp"
     (isa-insert "by (mp_tac %s);%n"))
    (nil        isa-eq_mp_tac 
     "eq_mp"
     (isa-insert "by (eq_mp_tac %s);%n"))))
  ("Goal" .
   (("\C-c\C-u" isa-undo "undo" (isa-insert "undo();%n"))
    ("\C-ch"    isa-chop "chop" (isa-insert "chop();%n"))
    ("\C-c\C-b" isa-back "back" (isa-insert "back();%n"))
    ("\C-cl"    isa-choplev "choplev" (isa-insert "choplev %p;"))
    ()
    ("\C-cg"    isa-goal-thy
     "goal thy" (isa-insert "val prems = goal thy \"%p\";"))
    ("\C-cG"    isa-goalw-thy
     "goalw thy" (isa-insert "val prems = goalw thy %p \"\";"))
    ()
    ("\C-cr"    isa-result "result" (isa-insert "val %p= result();"))
    ()
    ("\C-cp"    isa-push-proof
     "push-proof" (isa-insert "push_proof();%n"))
    ("\C-cP"    isa-pop-proof
     "pop proof"  (isa-insert "pop_proof();%n"))
    ("\C-cR"    isa-rotate-proofs
     "rotate proofs" (isa-insert "rotate_proof();%n"))))
  ("Option" .
   (("\C-c\C-l" nil "Listener"	     listener)
    ("\C-c\C-s" nil "Proof State"    proofstate)
    ;; could activate here with logic-name.
    ("\C-c\C-t" isa-ruletable-for-logic "Rules" (ruletable))
    (nil        nil "Rules..."       ruletable)        
    ()
    ("\C-cL"    isa-goals-limit
     "Set goals limit" (isa-insert "goals_limit:=%p;"))
    ("\C-cD"    isa-print-depth
     "Set print depth" (isa-insert "print_depth %p;"))
    ("\C-cT"    isa-show-types
     "Toggle show types" (isa-insert "show_types:= not(!show_types);%n"))
    ("\C-cS"    isa-show-sorts
     "Toggle show sorts" (isa-insert "show_sorts:=not(!show_sorts);%n"))
    (""         isa-repeat-tracing
     "Toggle REPEAT tracing" (isa-insert "trace_REPEAT:=not(!trace_REPEAT);%n"))
    ()
    ("\C-cq"    nil  "Exit Isabelle"     isa-quit-isabelle))))
"Table for building Isabelle commands and menus.
An alist: key is group name (menu heading), each entry is:

   (KEY SYMBOL TEXT FORM)  or  ()

where:
 
   KEY    - default key binding
   SYMBOL - function name (prefixed by \"isa-\")
   TEXT   - text form of function name (for menu entry)
   FORM   - body of command

KEY may be nil for no default binding.

SYMBOL may be nil, in which case FORM must be a symbol.

FORM may be a symbol, in which case it is invoked with call-interactively,
or else a list representing the body of a function to which SYMBOL 
will be bound.

() represents a grouping in the commands - separation bar in the menu.")



;;; =============== Command Functions ===============


(defun isa-quit-isabelle ()
  "Query and exit Isabelle using isa-quit-command"
  (interactive)
  (if (or isa-dont-query-quit
	  (save-excursion
	    (y-or-n-p "Quit Isabelle? ")))
      (funcall isa-quit-command)))



;;; ========== Command and Menu Generation ==========

(defun isa-generate-commands ()
  "Generate commands from isa-commands-table"
  (mapcar '(lambda (pair)
	     (let* ((groupname (car pair))
		    (comtab    (cdr pair))
		    (docprefix (concat "Isabelle menu command.   Execute " groupname ": ")))
	       (mapcar 'isa-make-command comtab))) isa-commands-table))

(defun isa-make-command (table-entry)
  "Simulate a defun based on a command table entry.
Requires docprefix to be set to a prefix for the documentation string."
  (if (and table-entry (nth 1 table-entry))
      (let* ((sym  (nth 1 table-entry))
	     (doc  (nth 2 table-entry))
	     (form (nth 3 table-entry))
	     (cmd  (if (symbolp form)
		       (list 'call-interactively (list 'quote form))
		     form))
	     (f (list 'lambda () 
		      (concat docprefix doc)
		      '(interactive)
		      cmd)))
	(fmakunbound sym)
	(fset sym f))))

(isa-generate-commands)  ; Generate the commands now.




(provide 'isa-cmd)

;;; end of isa-cmd.el