;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: stacks.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Macros for handling stacks and trails.
;;; For efficiency, we prefer to use stack-based iteration instead
;;; of recursion for several routines.  The routines below help
;;; make the use of stacks, in conjunction with the direct array 
;;; access routines, more convenient.

;; If we were programming in C and we wanted to use a stack for
;; an iterative routine, we would ordinarily allocate it locally 
;; on the call stack.  In Lisp, this can't be done, so rather than
;; invoke make-array each time we call the routine, we'll instead
;; assign an array to a global variable for each iterative routine,
;; and then get a local handle on the array on entry to the routine.
(defmacro defstack (name type)
  `(progn
     (defvar ,name nil)
     (eval-when (load eval compile)
	 (cond ((eq ,type :trail)
	        #+kcl (setf ,name (make-array (list *trail-size*)
					      :element-type 'fixnum
					      :static t))
	        #-kcl (setf ,name (make-array (list *trail-size*)
					      :element-type 'fixnum))
		)
	       ((eq ,type :fixnum-stack)
		#+kcl (setf ,name (make-array (list *stack-size*)
					      :element-type 'fixnum
					      :static t))
		#-kcl (setf ,name (make-array (list *stack-size*)
					      :element-type 'fixnum))
		)
	       (t
		#+kcl (setf ,name (make-array (list *stack-size*)
					      :static t))
		#-kcl (setf ,name (make-array (list *stack-size*)))
		)))))
  

;; CAREFUL: The sublis might accidentally bash symbols not intended
;; to be replaced.  Make sure all names in the "stacks" parameter
;; are distinguished in "code".
(defmacro with-stacks (stacks &body code &aux subst defs)
  (unless (listp stacks)
	  (error "Parameter STACKS of WITH-STACKS must be a list."))
  (dolist (s stacks)
	  (push (list (first s) (gentemp "ARRAY") (second s)) subst))
  (dolist (s subst)
	  (push `(defstack ,(second s) ,(third s)) defs)
	  (setf (cdr s) (second s)))
  `(progn ,@(append defs (sublis subst code))))




;; Trails are used to record bindings of variables which will
;; be undone later.  For example, after a term is rewritten
;; by a rule, the matching substitution is undone so that we
;; can re-use the rule.  

   
;; Unwind a trail for which we have a direct handle on its body.
(defmacro unwind-local-trail (array i vars)
  `(until (= ,i 1)
	  (decf ,i)
	  (c-aset ,vars (f-aref ,array ,i) nil)))


;; Allocate a trail locally (actually, by setting aside a global 
;; stack specifically for the current function) and
;; return the value of executing "code". Unwind and throw away the
;; trail before returning to caller.  Some local macros are 
;; available within code:
;;    (trail var) Push var onto the trail stack (where var is the ft-symbol
;;                of some variable flatterm)
;;    (var-binding var) Retrieve the binding of var.
;;    (set-binding var) Set the binding of var.
;; The local var-binding and set-binding macros shadow the global
;; var-binding and set-binding macros, and they provide faster
;; access to the *vars* binding array.
(defmacro with-local-trail (array &body code)
  (let ((trail (gentemp "TRAIL"))
	(top (gentemp "TOP"))
	(vars (gentemp "VARS")))
    `(let* ((,trail (get-fixnum-array ,array))
	    (,top 1)
	    (,vars (get-c-array *vars*)))
       (declare-fixnum-array ,trail)
       (declare-c-array ,vars)
       (declare (type fixnum ,top))
       (macrolet
	(
	 (trail (var) `(f-stack-push ,var ,',trail ,',top))
	 (var-binding (sym) `(c-aref ,',vars ,sym))
	 (set-binding (sym b) `(c-aset ,',vars ,sym ,b))
	 )
	(prog1
	    (progn ,@code)
	  (unwind-local-trail ,trail ,top ,vars)
	  )))))


;; Allocate a trail which is passed back to the caller on success of 
;; "code".  Otherwise, undo any bindings and return nil.
(defmacro with-trail (&body code)
  (let* ((trail (gentemp "TRAIL"))
	 (tr (gentemp "TR"))
	 (top (gentemp "TOP"))
	 (result (gentemp "RESULT"))
	 (vars (gentemp "VARS")))
    `(let* ((,tr (new-trail))
	    (,trail (get-fixnum-array ,tr))
	    (,top 1)
	    (,vars (get-c-array *vars*)))
       (declare (type (array fixnum) ,tr))
       (declare-fixnum-array ,trail)
       (declare-c-array ,vars)
       (declare (type fixnum ,top))
       (macrolet
	(
	 (trail (var) `(f-stack-push ,var ,',trail ,',top))
	 (var-binding (sym) `(c-aref ,',vars ,sym))
	 (set-binding (sym b) `(c-aset ,',vars ,sym ,b))
	 (save-trail-top (trail top)
			 `(f-aset ,trail 0 ,top))
	 )
	(let ((,result (progn ,@code)))
	  (if ,result
	      (progn
		(save-trail-top ,trail ,top)
		,tr)
	    (progn (unwind-local-trail ,trail ,top ,vars)
		   (free-trail ,tr)
		   nil)))))))

	  
;; General stack and trail routines
(defmacro stack-push (x stk i)
  `(progn
     (setf (aref (the (array t) ,stk) (the fixnum ,i)) ,x)
     (incf ,i)))

(defmacro stack-pop (stk i)
     `(aref (the (array t) ,stk) (the fixnum (decf ,i))))
	      
(defmacro trail-var (var trail top)
  `(progn
     (f-stack-push ,var ,trail ,top)))
  
(defmacro trail-pop (trail top)
  `(the fixnum (f-aref (the (array fixnum) ,trail) (decf ,top))))

;; Record the top of the trail stack in element 0 of the trail.
;; This is usually used when a trail is returned to a calling 
;; routine.
(defmacro save-trail-top (trail top)
  `(setf (aref (the (array fixnum) ,trail) 0) (the fixnum ,top)))

;; Find the top of a trail that was returned by another routine.
(defmacro get-trail-top (trail)
  `(aref (the (array fixnum) ,trail) 0))

;; Undo a single frame of a trail stack.  Used primarily for the
;; E-unification algorithm, which must incrementally bind and 
;; unbind variables on backtracking.
(defmacro restore-trail-frame (trail frame top)
  (let ((tr (gentemp "TRAIL")))
    `(let ((,tr ,trail))
       (declare (type (array fixnum) ,tr))
       (until (= ,top ,frame)
	      (make-unbound-var (trail-pop ,tr ,top))))))

;; Undo all the bindings of a trail that was returned by another routine.
(defmacro restore-vars (trail)
  (let ((tr (gentemp "TRAIL"))(i (gentemp "I"))(vars (gentemp "VARS")))
    `(let* ((,tr (get-fixnum-array ,trail))
	    (,i (f-aref ,tr 0))
	    (,vars (get-c-array *vars*)))
       (declare-fixnum-array ,tr)
       (declare-c-array ,vars)
       (declare (type fixnum ,i))
       (until (= ,i 1)
	      (decf ,i)
	      (c-aset ,vars (f-aref ,tr ,i) nil))
       (free-trail ,trail))))

;; Undo all the bindings of a local trail
(defmacro unwind-trail (trail top)
  (let ((tr (gentemp "TRAIL"))(vars (gentemp "VARS")))
    `(let* ((,tr (get-fixnum-array ,trail))
	    (,vars (get-c-array *vars*)))
       (declare-fixnum-array ,tr)
       (declare-c-array ,vars)
       (until (= ,top 1)
	      (decf ,top)
	      (c-aset ,vars (f-aref ,tr ,top) nil))
       (free-trail ,trail))))
  
;; Start a new trail frame
(defmacro push-trail-frame (trail frame top)
  `(progn
     (trail-var ,frame ,trail ,top)
     (setf ,frame ,top)))

;; Return to a previous trail frame
(defmacro pop-trail-frame (trail frame top)
  `(progn
     (setf ,frame (trail-pop ,trail ,top))))





