;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: rpo.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Lexicographic recursive path ordering


;; Are op1 and op2 equal under the precedence relation?
(defmacro op-equal (op1 op2 prec)
  (declare (ignore prec))
  `(same-symbol? ,op1 ,op2))

;; Is op1 greater than op2 under the precedence relation.
(proclaim '(function op-greater (t t t) t))
(defun op-greater (op1 op2 prec &aux)
  (or
   (sloop for x in prec thereis (and (eql (first x) '>)
				      (same-symbol? (second x) op1)
				      (same-symbol? (third x) op2)))
   (sloop for x in prec thereis (and (eql (first x) '>)
				     (same-symbol? (second x) op1)
				     (op-greater (third x) op2 prec)))))

;; Does op have status?
(proclaim '(function op-status (t t) t))
(defun op-status (op prec &aux status)
  (sloop for x in prec thereis (and (eql (first x) 'status)
				    (same-symbol? (second x) op)
				    (setf status (third x))))
  status)

;; Are t1 and t2 equivalent?
(proclaim '(function equiv-terms (t t t) t))
(defun equiv-terms (t1 t2 precedence)
  (let ((p precedence))
    (declare (special p))
    (equiv-terms-aux t1 t2)))

(proclaim '(function equiv-terms-aux (t t) t))
(defun equiv-terms-aux (t1 t2)
  (declare (special p))
  (when (and (null t1) (null t2))
	(return-from equiv-terms-aux nil))
  (when (not (same-symbol? (ft-symbol t1) (ft-symbol t2)))
	(return-from equiv-terms-aux nil))
  (when (var? (ft-symbol t1))
	(return-from equiv-terms-aux t))
  (cond
   ((and (not (fsym-mutations (ft-symbol t1)))
	 (op-status (ft-symbol t1) p))
    (do-args2 t1 t2
	      (unless (equiv-terms-aux (ft1) (ft2)) (abort))))
   (t
    (let ((count 0))
      (declare (type fixnum count))
      (do-args t1
	       (setf count 0)
	       (let ((x1 (ft)))
		 (do-args t1
		    (when (equiv-terms-aux x1 (ft)) (incf count)))
		 (do-args t2
		    (when (equiv-terms-aux x1 (ft)) (decf count))))
	       (unless (zerop count) (abort)))))))
    
    

;; Multiset extension of rpo ordering
(proclaim '(function multiargs-greater (t t t) t))
(defun multiargs-greater (t1 t2 prec &aux ml1 ml2)
  (do-args t1 (push1 (ft) ml1))
  (do-args t2 (push1 (ft) ml2))
  (let* (
	 tmp
	 (m2-m1
	  (progn
	    (sloop for x in ml2
		   when (> (count x ml2 :test #'e-equal-func)
			   (count x ml1 :test #'e-equal-func))
		   do (push1 x tmp)) 
	    tmp))
	 (m1-m2
	  (progn
	    (setf tmp nil)
	    (sloop for x in ml1
		   when (> (count x ml1 :test #'e-equal-func)
			   (count x ml2 :test #'e-equal-func))
		   do (push1 x tmp))
	    tmp))
	 (result
	  (and m1-m2
	       (sloop for x in m2-m1
		      always
		      (sloop for y
			     in m1-m2 thereis (rpo-greater y x prec)))))
	  )
    (free-list m2-m1)
    (free-list m1-m2)
    (free-list ml1)
    (free-list ml2)
    result))


;; Top level of rpo ordering.
(proclaim '(function rpo-greater (t t t) t))
(defun rpo-greater (t1 t2 precedence
		       &aux (sym1 (ft-symbol t1))
		            (sym2 (ft-symbol t2)))
  (declare (type fixnum sym1 sym2))
  
  (cond
   ((var? sym1) nil)
   ((var? sym2) (occurs? sym2 t1))
   (t
    (cond
     ((op-equal sym1 sym2 precedence)
      (if (and (not (fsym-mutations sym1))
	       (op-status sym1 precedence))
	  (lex-rpo t1 t2 precedence)
	(multiargs-greater t1 t2 precedence)))
     ((op-greater sym1 sym2 precedence)
      (all-args t2 (rpo-greater t1 (ft) precedence)))
     (t
      (some-arg t1
	 (or (rpo-greater (ft) t2 precedence)
	     (equiv-terms (ft) t2 precedence))))
     ))))


;; Lex extension of rpo ordering.
(proclaim '(function lex-rpo (t t t) t))
(defun lex-rpo (t1 t2 precedence &aux list1 list2 result)

  (do-args t1 (push1 (ft) list1))
  (do-args t2 (push1 (ft) list2))
  (unless (eq (op-status (ft-symbol t1) precedence) 'rl)
	  (setf list1 (rev1 list1))
	  (setf list2 (rev1 list2)))
  (while (equiv-terms (car list1) (car list2) precedence)
    (pop1 list1)
    (pop1 list2))
  (setf result
	(cond
	 ((null list1) nil)
	 ((null list2) t)
	 ((rpo-greater (pop1 list1) (pop1 list2) precedence)
	  (sloop for x in list2 always (rpo-greater t1 x precedence)))
	 (t
	  (sloop for x in list1
		 thereis (or (equiv-terms x t2 precedence)
			     (rpo-greater x t2 precedence))))))
  (free-list list1)
  (free-list list2)
  result)
	

	   









