;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: queues.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Prioritized equation queues.
;; Queues are simply doubly linked lists of equations.  For efficiency,
;; they may be indexed with an array.  Each queue has
;; a primary and a secondary priority function associated with it.
;; The secondary function is called in the event of a tie using the
;; primary function.  Default priority functions are provided, but
;; they can be overridden with arbitrary functions (see below).
;; The head and tail of a queue are dummy equations; this simplifies
;; boundary checks.

;; Create a new queue.
(defun get-queue (&key pfunc1 pfunc2 indexed &aux q)
  (setf q (new-queue))
  (setf (queue-head q) (new-eqn))
  (setf (queue-tail q) (new-eqn))
  (setf (queue-pfunc1 q) pfunc1)
  (setf (queue-pfunc2 q) pfunc2)
  (if indexed
      (setf (queue-index q)
	    #+kcl (make-array *queue-index-size* :static t)
	    #-kcl (make-array *queue-index-size*)
	    )
    (setf (queue-index q) nil))
  (init-queue q)
  q)

;; Initialize a queue.
(proclaim '(function init-queue (t) nil))
(defun init-queue (q)
  (setf (eqn-next (queue-head q)) (queue-tail q))
  (setf (eqn-prev (queue-tail q)) (queue-head q))
  (setf (eqn-priority1 (queue-head q)) 0)
  (setf (eqn-priority2 (queue-head q)) 0)
  (setf (eqn-priority1 (queue-tail q)) 0)
  (setf (eqn-priority2 (queue-tail q)) 0)
  (when (queue-index q)
	(dotimes (i *queue-index-size*)
		 (declare (type fixnum i))
		 (setf (aref (the (array t) (queue-index q)) i)
		       (queue-tail q))))
  )


;; Macro for traversing a queue from left to right (highest priority to
;; lowest).
(defmacro do-queue (e q &body code)
  (let ((e-next (gentemp "EQN"))
	(qt (gentemp "EQN")))
  `(let ((,qt (queue-tail ,q))
	 ,e-next
	 (,e (eqn-next (queue-head ,q))))
     (until (eq ,e ,qt)
	    :return nil
	    (setf ,e-next (eqn-next ,e)) ;In case e is munched by code
	    ,@code
	    (setf ,e ,e-next)))))

;; Backwards traversal of a queue
(defmacro do-queue-backwards (e q &body code)
  (let ((e-prev (gentemp "EQN"))
	(qh (gentemp "EQN")))
  `(let ((,qh (queue-head ,q))
	 ,e-prev
	 (,e (eqn-prev (queue-tail ,q))))
     (until (eq ,e ,qh)
	    :return nil
	    (setf ,e-prev (eqn-prev ,e)) ;In case e is munched by code
	    ,@code
	    (setf ,e ,e-prev)))))

;; Insert an equation into a queue
(proclaim '(function enqueue-eqn (t t) nil))
(defun enqueue-eqn (e q &aux e1 (p1 0))
  (declare (type fixnum p1))
  ;; Determine priority of e
  (if (queue-pfunc1 q)
      (setf (eqn-priority1 e) (the fixnum (funcall (queue-pfunc1 q) e)))
    (setf (eqn-priority1 e) (the fixnum (pfunc-lrsmall e))))
  (if (queue-pfunc2 q)
      (setf (eqn-priority2 e) (the fixnum (funcall (queue-pfunc2 q) e)))
    (setf (eqn-priority2 e) (the fixnum (pfunc-old e))))

  (setf p1 (the fixnum (eqn-priority1 e)))
  ;; Find first e1 with same or greater priority1 
  (if (queue-index q)
      ;; If indexed queue, look in index for correct position
      (progn
	(when (>= (the fixnum (eqn-priority1 e))
		  *queue-index-size*)
	      (setf p1 (the fixnum (1- *queue-index-size*))))
	(setf e1 (aref-with-decl (queue-index q) p1)))
    (progn
      ;; Else search from beginning of queue
      (setf e1 (eqn-next (queue-head q)))
      (until (or (>= (the fixnum (eqn-priority1 e1)) p1)
		 (eq e1 (queue-tail q)))
	     (setf e1 (eqn-next e1)))))

  ;; Now compare according to priority2
  (until (or (eq e1 (queue-tail q))
	     (> (the fixnum (eqn-priority1 e1))
		(the fixnum (eqn-priority1 e)))
	     (>= (the fixnum (eqn-priority2 e1))
		 (the fixnum (eqn-priority2 e))))
	 (setf e1 (eqn-next e1)))

  ;; Enqueue e in front of e1
  (disabling-interrupts
    (setf (eqn-queue e) q)
    (setf (eqn-prev e) (eqn-prev e1))
    (setf (eqn-next e) e1)
    (setf (eqn-next (eqn-prev e)) e)
    (setf (eqn-prev e1) e)
    
    ;; Update index as needed
    (when (queue-index q)
	  (do ((i p1 (the fixnum (1- i))))
	      ((<= i 0))
	      (declare (type fixnum i))
	      (if (eq e1 (aref-with-decl (queue-index q) i))
		  (setf (aref-with-decl (queue-index q) i) e)
		(break-from-loop))))
    )
  )
		  

;; Place an equation at the front of a queue.  Doesn't work
;; with indexed queues.
(proclaim '(function queue-in-front (t t) nil))
(defun queue-in-front (e q)
  (when (queue-index q)
	(error "Front insertion attempted with indexed queue"))
  (disabling-interrupts
    (setf (eqn-next e) (eqn-next (queue-head q)))
    (setf (eqn-next (queue-head q)) e)
    (setf (eqn-prev (eqn-next e)) e)
    (setf (eqn-prev e) (queue-head q))
    (setf (eqn-queue e) q)
  ))

;; Place an equation in the back of a queue.  Doesn't work
;; with indexed queues.
(proclaim '(function queue-in-back (t t) nil))
(defun queue-in-back (e q)
  (when (queue-index q)
	(error "Back insertion attempted with indexed queue"))
  (disabling-interrupts
    (setf (eqn-prev e) (eqn-prev (queue-tail q)))
    (setf (eqn-prev (queue-tail q)) e)
    (setf (eqn-next (eqn-prev e)) e)
    (setf (eqn-next e) (queue-tail q))
    (setf (eqn-queue e) q)
  ))
    

;; Remove the highest-priority equation from a queue.
(proclaim '(function dequeue-eqn (t) t))
(defun dequeue-eqn (q &aux e e1 (i 0))
  (declare (type fixnum i))
  (disabling-interrupts
    (unless
     (eq (eqn-next (queue-head q)) (queue-tail q))
     (setf e (eqn-next (queue-head q)))
     (setf (eqn-next (queue-head q)) (eqn-next e))
     (setf (eqn-prev (eqn-next e)) (queue-head q))
     (when (queue-index q)
	   (if (>= (the fixnum (eqn-priority1 e))
		   *queue-index-size*)
	       (setf i (the fixnum (1- *queue-index-size*)))
	     (setf i (eqn-priority1 e)))
	   (setf e1 (eqn-next e))
	   (while (> i 0)
	     (if (eq (aref-with-decl (queue-index q) i) e)
		 (progn
		   (setf (aref-with-decl (queue-index q) i) e1)
		   (decf i))
	       (break-from-loop))))
     e)
    ))


;; Delete an equation from a queue.
(proclaim '(function delete-eqn (t) nil))
(defun delete-eqn (e &aux (q (eqn-queue e)) (i 0) e1)

 (declare (type fixnum i))

 (disabling-interrupts
  (setf (eqn-prev (eqn-next e)) (eqn-prev e))
  (setf (eqn-next (eqn-prev e)) (eqn-next e))
  (when (queue-index q)
	(if (>= (the fixnum (eqn-priority1 e))
		*queue-index-size*)
	    (setf i (the fixnum (1- *queue-index-size*)))
	  (setf i (eqn-priority1 e)))
	(setf e1 (eqn-next e))
	(while (> i 0)
	  (if (eq (aref-with-decl (queue-index q) i) e)
	      (progn
		(setf (aref-with-decl (queue-index q) i) e1)
		(decf i))
	    (break-from-loop))))
  ))

;; Emptiness of queues
(defmacro queue-empty-p (q)
  `(eq (eqn-next (queue-head ,q)) (queue-tail ,q)))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Priority functions.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Prefer small combined lhs size and rhs size
(proclaim '(function pfunc-lrsmall (t) fixnum))
(defun pfunc-lrsmall (e)
  (the fixnum (+ (sym-count (eqn-lhs e))
		 (sym-count (eqn-rhs e)))))

;; Prefer small lhs size
(proclaim '(function pfunc-lsmall (t) fixnum))
(defun pfunc-lsmall (e)
  (the fixnum (sym-count (eqn-lhs e))))

;; Prefer small rhs size
(proclaim '(function pfunc-rsmall (t) fixnum))
(defun pfunc-rsmall (e)
  (the fixnum (sym-count (eqn-rhs e))))

;; Prefer older equations
(proclaim '(function pfunc-old (t) fixnum))
(defun pfunc-old (e)
  (unless (eqn-id e) (setf (eqn-id e) (incf-eqn-counter)))
  (the fixnum (eqn-id e)))

;; Prefer younger equations
;; ### Can't use with indexed queues
(proclaim '(function pfunc-young (t) fixnum))
(defun pfunc-young (e)
  (unless (eqn-id e) (setf (eqn-id e) (incf-eqn-counter)))
  (the fixnum (- (the fixnum (eqn-id e)))))

;; Queue in front -- i.e. behave like stack
(proclaim '(function pfunc-front (t) fixnum))
(defun pfunc-front (e)
  (declare (ignore e))
  0)

