;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: print.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Printing of various data types

(defvar *name-counter* 0)
(proclaim '(type fixnum *name-counter*))

;; An array of print names for variables
(defvar *var-names*
  #+kcl (make-array *max-all-vars* :static t)
  #-kcl (make-array *max-all-vars*)
  )
(eval-when (load eval)
  (dotimes (i *max-vars*)
	   (setf (aref *var-names* i)
		 (intern (format nil "X~S" i))))
  (dotimes (i *max-vars*)
	   (setf (aref *var-names* (+ i *max-x-vars*))
		 (intern (format nil "Y~S" i))))
  )

;; Retrieve the print name of a variable.
(defmacro get-var-name (i)
  `(aref (the (array t) *var-names*) (the fixnum ,i)))

;; No var deref.
(proclaim '(function print-term (t) nil))
(defun print-term (X)
  (setf *name-counter* -1)
  (let ((*print-level* nil)
	(term (build-term-no-deref X)))
    (declare (special *print-level*))
    (princ term)
    (free-conses term)))

;; Deref only one level of vars.  Mainly for debugging.
(proclaim '(function print-term-after-match (t) nil))
(defun print-term-after-match (X)
  (setf *name-counter* -1)
  (let ((*print-level* nil)
	(term (build-term-after-match X)))
    (declare (special *print-level*))
    (princ term)
    (free-conses term)))

;; Fully deref vars.  Mainly for debugging.
(proclaim '(function print-term-after-match (t) nil))
(defun print-term-after-unify (X)
  (setf *name-counter* -1)
  (let ((*print-level* nil)
	(term (build-term X)))
    (declare (special *print-level*))
    (princ term)
    (free-conses term)))

;; Ignore var bindings
(proclaim '(function build-term-no-deref (t) t))
(defun build-term-no-deref (ft &aux (sym 0))
  (declare (type fixnum sym))
  (setf sym (ft-symbol ft))
  (cond ((var? sym) (get-var-name sym))
	(t
	 (let ((functor (fsym-name sym))
	       (args nil)
	       (end (ft-next (ft-end ft))))
	   (setf ft (ft-next ft))
	   (while (not (eq ft end))
	     (setf args (nconc args (cons1 (build-term-no-deref ft) nil)))
	     (setf ft (ft-next (ft-end ft))))
	   (cons1 functor args)))))

;; Deref first level of vars
(proclaim '(function build-term-after-match (t) t))
(defun build-term-after-match (ft &aux (sym 0))
  (declare (type fixnum sym))
  (setf sym (ft-symbol ft))
  (cond ((and (var? sym) (unbound-var? sym))
	 (get-var-name sym))
	((and (var? sym) (bound-var? sym))
	 (build-term-no-deref (var-binding sym)))
	(t
	 (let ((functor (fsym-name sym))
	       (args nil)
	       (end (ft-next (ft-end ft))))
	   (setf ft (ft-next ft))
	   (while (not (eq ft end))
	     (setf args
		   (nconc args (cons1 (build-term-after-match ft) nil)))
	     (setf ft (ft-next (ft-end ft))))
	   (cons1 functor args)))))

;; Fully deref vars
(proclaim '(function build-term (t) t))
(defun build-term (ft &aux (sym 0))
  (declare (type fixnum sym))
  (setf sym (ft-symbol ft))
  (cond ((and (var? sym) (unbound-var? sym))
	 (get-var-name sym))
	((and (var? sym) (bound-var? sym))
	 (build-term (var-binding sym)))
	(t
	 (let ((functor (fsym-name sym))
	       (args nil)
	       (end (ft-next (ft-end ft))))
	   (setf ft (ft-next ft))
	   (while (not (eq ft end))
	     (setf args (nconc args (cons1 (build-term ft) nil)))
	     (setf ft (ft-next (ft-end ft))))
	   (cons1 functor args)))))

;; Print a discrimination tree
(proclaim '(function print-net (t) nil))
(defun print-net (n)
  (terpri)
  (princ "<<< Discrimination Net >>>")
  (terpri)
  (print-net-aux (net-nodes n) 0))

(proclaim '(function print-net-aux (t t) nil))
(defun print-net-aux (n d)
  (cond ((null n))
	((not (node-p n)) (print-leaves n))
	(t
	 (dotimes (i d) (princ "  "))
	 ;(princ "<<NODE>>")
	 (dotimes (i *slotarray-size*)
	    (when (slot-val (assoc-slot i n))
	      (terpri)
	      (dotimes (j d) (princ "  "))
	      (if (zerop i)
		  (princ '*var*)
		(princ (fsym-name (to-fsym (the fixnum (1- i))))))
	      (princ " ");
	      (print-net-aux (slot-val (assoc-slot i n)) (+ d 1))
	      )))))

(proclaim '(function print-leaves (t) nil))
(defun print-leaves (lf)
  (while lf
    (princ "<Leaf: ");
    (when (leaf-subterm lf) (print-term (leaf-subterm lf)))
    (princ "> ")
    (setf lf (leaf-next lf))))

(proclaim '(function print-eqn (t) nil))
(defun print-eqn (e &aux (*print-level* nil))
  (declare (special *print-level*)) 
  (setf *name-counter* -1)
  (when (eqn-id e)
	(princ (eqn-id e))
	(princ ": "))
  (let ((term (build-term-no-deref (eqn-lhs e))))
    (princ term)
    (free-conses term))
  (if (member (eqn-type e) '(*rewrite-rule* *pseudo-rule*))
      (princ " --> ")
    (princ " = "))
  (let ((term (build-term-no-deref (eqn-rhs e))))
    (princ term)
    (free-conses term))
  )


(proclaim '(function print-deleted (t) nil))
(defun print-deleted (e)
  (unless *benchmark*
    (when (or (eqn-id e) (not (eqn-parents e)))
	  (format t "~%Deleted ")
	  (case (eqn-type e)
		(*rewrite-rule* (format t "rule ~S" (eqn-id e)))
		(*failure* (format t "failure ~S" (eqn-id e)))
		(*equation* (format t "equation ~S" (eqn-id e)))
		(*pseudo-rule* (format t "pseudo-rule ~S" (eqn-id e)))
		(t (format t "~S" (eqn-id e)))))))
		

(proclaim '(function print-subsumed (t t) nil))
(defun print-subsumed (e1 e2)
  (unless *benchmark*
	  (if (member (eqn-type e1) '(*rewrite-rule* *pseudo-rule*))
	      (format t "~%Rule ")
	    (format t "~%Failure "))
	  (format t "~S Subsumed by ~S" (eqn-id e1) (eqn-id e2))))
	
(proclaim '(function print-oriented (t) nil))
(defun print-oriented (e)
  (unless *benchmark*
    (if (not (member (eqn-type e) '(*new-pair*)))
	(progn
	  (format t "~%~%Re-oriented ")
	  (case (eqn-type e)
		(*failure* (format t "failure "))
		(*rewrite-rule* (format t "rule "))
		(*pseudo-rule* (format t "pseudo-rule "))
		(*equation* (format t "equation ")))
	  (setf (eqn-type e) '*rewrite-rule*)
	  (print-eqn e))
      (progn
	(format t "~%~%Rule ")
	(setf (eqn-type e) '*rewrite-rule*)
	(print-eqn e)
	(if (eqn-parents e)
	    (progn
	      (format t " from ~S on ~S "
		      (cdr (eqn-parents e))
		      (car (eqn-parents e))))
	  (format t " from user"))
	))))

(proclaim '(function print-failure (t) nil))
(defun print-failure (e)
  (unless *benchmark*
    (format t "~%~%Failure ")
    (setf (eqn-type e) '*failure*)
    (print-eqn e)
    (if (eqn-parents e)
	(progn
	  (format t " from ~S on ~S"
		  (cdr (eqn-parents e))
		  (car (eqn-parents e))))
      (format t " from user"))
    ))

(proclaim '(function print-pseudo-rule (t) nil))
(defun print-pseudo-rule (e)
  (unless *benchmark*
    (format t "~%~%Pseudo-rule " (eqn-id e))
    (print-eqn e)
    (if (eqn-parents e)
	(progn
	  (format t " from ~S on ~S"
		  (cdr (eqn-parents e))
		  (car (eqn-parents e))))
      (format t " from user"))
    ))

(proclaim '(function print-reduced (t) nil))
(defun print-reduced (e)
  (unless *benchmark*
    (format t "~%~%Reduced ")
    (print-eqn e)
    ))

(proclaim '(function print-reducible (t) nil))
(defun print-reducible (e)
  (unless *benchmark*
    (if (eqn-fsym-def-p e)
	(format t "~%~%*** Defining rule ~S reducible" (eqn-id e))
      (format t "~%~S reducible" (eqn-id e)))
    ))

(proclaim '(function print-new-pair (t) nil))
(defun print-new-pair (e)
  (format t "~%~%Pair ")
  (print-eqn e)
  (if (eqn-parents e)
      (progn
	(format t " from ~S on ~S"
		(cdr (eqn-parents e))
		(car (eqn-parents e))))
    (format t " from user")))

(proclaim '(function print-equation (t) nil))
(defun print-equation (e)
  (format t "~%~%Permuter ")
  (print-eqn e)
  (if (eqn-parents e)
      (progn
	(format t " from ~S on ~S"
		(cdr (eqn-parents e))
		(car (eqn-parents e))))
    (format t " from user")))

(defun print-path (p)
  (setf p (nreverse p))
  (princ "[")
  (do ((x p (cdr x)))
      ((null x))
      (princ (car x))
      (unless (null (cdr x))
	      (princ ".")))
  (princ "]")
  (setf p (nreverse p)))

(proclaim '(function print-precedence (t) nil))
(defun print-precedence (prec)
  (setf prec (remove-if #'(lambda (x) (eq (car x) 'freeze)) prec))
  (do
   ((p (car prec) (car pcdrs))
    (pcdrs (cdr prec) (cdr pcdrs)))
   ((null p))
   (case (car p)
	 (status
	  (format t "Status ~S ~S" (third p) (fsym-name (second p))))
	 (>
	  (format t "~S > ~S" (fsym-name (second p))
		  (fsym-name (third p))))
	 (freeze
	  (format t "~S <> ~S" (fsym-name (second p))
		  (fsym-name (third p))))
	 )
   (when pcdrs (format t ", "))))

(proclaim '(function print-nofail-rewrite (t t t) nil))
(defun print-nofail-rewrite (ft new-term eqn)
  (format t "~%*** Failure rewriting ")
  (print-term ft) (princ " to ") (print-term new-term)
  (princ " by ") (print-eqn eqn))






