;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: nodes.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Discrimination trees.  Permutative lookup routines are in "enodes.lsp".

;; For net-find-lhs and e-net-find-lhs, we usually want to find 
;; only a single matching rule, but sometimes we want to find
;; more than one.
(defconstant *single* 0)		;Find at most one rule
(defconstant *multiple-first* 1)	;Find the first of several rules
(defconstant *multiple-resume* 2)	;Find the next rule

;; When multiple rules are sought using net-find-lhs and e-net-find-lhs,
;; some state must be saved in between invocations.
(defvar *saved-B* nil)			;Top of choice stack
(eval-when (load compile eval)
	   (setf *saved-B* (make-array 1 :element-type 'fixnum)))
(defmacro saved-B () `(the fixnum (aref (the (array fixnum) *saved-B*) 0)))

(defvar *saved-LF* nil)			;Current leaf.

;; Given a node, retrieve the slotnum'th slot.
(defmacro assoc-slot (slotnum node)
  `(aref (the (array t) (node-slots ,node)) (the fixnum ,slotnum)))

;; Convert a function symbol ID into a slot number.
(defmacro slot-num (ft) `(fsym-slot-num (ft-symbol ,ft)))

;; Given a slot, find the arity of the associated function symbol.
(defmacro slot-arity (sl)
  `(fsym-arity (the fixnum (1- (to-fsym ,sl)))))


;; Initialize the discrimination net routines.
(proclaim '(function init-nets () nil))
(defun init-nets ()
  ;; Reuse slot arrays if the number of fsyms hasn't changed
  (when (not (= *slotarray-size* (1+ *num-fsyms*)))
	(while *slotarray-freelist*
	  (let ((x (pop1 *slotarray-freelist*)))
	    (floop :for SL :below *slotarray-size*
		   :do (free-cons
			(aref (the (array t) x) SL)))))
	(setf *slotarray-freelist* nil)
	(setf *slotarray-size* (1+ *num-fsyms*))
	)
  ;; Assign a slot num to each fsym, starting with 1.
  ;; Slot 0 is reserved for vars.
  ;; (fsym-slot-num i) = (1+ i), so that we 
  ;; can easily recover an fsym given its slot num
  (dotimes (i *num-fsyms*)
	   (setf (fsym-slot-num (to-fsym i)) (1+ i)))
  )

;; Insert a leaf into a doubly-linked list of leaves
(defmacro insert-leaf (SL L)
  (let ((V (gentemp "V")))
    `(let ((,V (slot-val ,SL)))
       (setf (leaf-next ,L) ,V)
       (when ,V (setf (leaf-prev ,V) ,L))
       (setf (leaf-prev ,L) nil)
       (setf (slot-val ,SL) ,L))))

;; Insert a subterm of an equation into a discrimination net
(proclaim '(function net-insert (t t t) t))
(defun net-insert (ft e net
		      &aux PN (SL net) (N (net-nodes net)) (S ft) L
		      (last (ft-next (ft-end ft))))
 (disabling-interrupts 
  ;(format t "~%Inserting ") (print-term ft)
  (until (eq S last)
      (when (null N)
	    (setf N (new-node))
	    (setf (node-prev-node N) PN)
	    (setf (node-prev-slot N) SL)
	    (setf (node-slots-used N) 0)
	    (setf (slot-val SL) N)
	    (unless (null PN)
		    (incf (node-slots-used PN))))
      ;; We don't distinguish among variables; they are treated
      ;; as a single wildcard symbol, in slot 0 of each node.
      (if (var? (ft-symbol S)) 
	  (setf SL (assoc-slot 0 N))
	(setf SL (assoc-slot (slot-num S) N)))
      (setf PN N)
      (setf N (slot-val SL))
      (setf S (ft-next S))
      )

  (setf L (new-leaf))
  (setf (leaf-eqn L) e)
  (setf (leaf-parent-slot L) SL)
  (setf (leaf-parent-node L) PN)
  (setf (leaf-subterm L) ft)
  (insert-leaf SL L)
  (when (null N)
	(incf (node-slots-used PN)))
  ;(print-net net)
  L
  ))

(defmacro delete-leaf (SL L)
  (let ((n (gentemp "N"))(p (gentemp "P")))
    `(let ((,n (leaf-next ,L))
	   (,p (leaf-prev ,L)))
       (when ,n (setf (leaf-prev ,n) ,p))
       (if ,p
	   (setf (leaf-next ,p) ,n)
	 (setf (slot-val ,SL) ,n)))))

;; Given a leaf, delete it from the net.
(proclaim '(function net-delete (t) nil))
(defun net-delete (L &aux (N (leaf-parent-node L))
		          (SL (leaf-parent-slot L)))
  
  (delete-leaf SL L)
  (free-leaf L)
  (while (and (null (slot-val SL)) (node-p N))
      (unless (= (the fixnum (decf (node-slots-used N))) 0)
	      (return))
      (setf SL (node-prev-slot N))
      (setf (slot-val SL) nil)
      (let ((Node N))
	(setf N (node-prev-node Node))
	(free-slot-array (node-slots Node))
	(free-node Node)
	)))


;; Find a rule or equation, the left side of which ft is an
;; instance.  Used to find rules to reduce ft.  Note that in
;; all of the net matching routines, we don't build substitutions
;; as we walk down the net -- we ignore variable bindings
;; until we get to a leaf, and then do a check for variable
;; binding consistency.
(proclaim '(function net-find-lhs (t t fixnum) t))
(with-stacks ((*stack* :stack))
(defun net-find-lhs (ft net mode &aux (stack (get-c-array *stack*))
			trail (N (net-nodes net)) (S ft) (B 0)
			(end (ft-next (ft-end ft))) lf)
  (declare (type fixnum mode))
  (declare-c-array stack)
  (declare (type fixnum B))
  
  (block
   FIND-LHS

  ;; When the net is empty, return.
  (when (null N) (return-from FIND-LHS))

  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))	;Current node
    (choice-S () `(c-aref stack (the fixnum (1+ B)))) ;Current subterm
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (1+ B)) ,x))
    (next-choice () `(incf B 2))
    (prev-choice () `(decf B 2))
    (choice-stack-empty? () `(zerop B))
    )
  (prog ()

    ;; Restore state if finding the next of several
    ;; rules.  
    (when (= mode *multiple-resume*)
	  (setf B (saved-B))
	  (setf lf *saved-LF*)
	  (go FIND-LEAF))
    TOP
    (until (eq S end)
      ;; Try the var slot first.
      (let ((var-slot (assoc-slot 0 N)))
	 (if (slot-val var-slot)
	     (progn
	       (unless (var? (ft-symbol S))
		       ;; Push a choicepoint
		       (set-choice-N N)
		       (set-choice-S S)
		       (next-choice))
	       (setf N (slot-val var-slot))
	       (setf S (ft-next (ft-end S))))
	   (progn
	     (when (var? (ft-symbol S)) (go BACKTRACK))
	     (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
		     (go BACKTRACK))
	     (setf S (ft-next S))))))

    ;; We're at the end of a branch.  See if there's a matching leaf.
    (setf lf N)
    FIND-LEAF
    (while lf  
      (setf trail (match-vars (eqn-lhs (leaf-eqn lf)) ft))
      (when trail (break-from-loop))
      (setf lf (leaf-next lf)))
    (unless trail (go BACKTRACK))
    (unless (= mode *single*)
	    (setf (saved-B) B)
	    (setf *saved-LF* (leaf-next lf)))
    ;; Return both the binding trail and the leaf.
    (return-from FIND-LHS (cons1 lf trail))
    
    BACKTRACK
    (when (choice-stack-empty?)
	  (return-from FIND-LHS nil))
    (prev-choice)
    (setf N (choice-N))
    (setf S (choice-S))
    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
	    (go BACKTRACK))
    (setf S (ft-next S))
    (go TOP)
    )))))


;; Find all subterms which are an instance of the left side of
;; rule.  If rewrite? is non-nil, then actually perform a rewrite
;; using rule.
(proclaim '(function net-find-matches (t t t) t))
(with-stacks ((*stack* :stack) (*fstack* :fixnum-stack))
(defun net-find-matches (rule net rewrite?
			   &aux (stack (get-c-array *stack*))
			   (ft (eqn-lhs rule))
			   (fstack (get-fixnum-array *fstack*))
			   (is-prule (eq (eqn-type rule) '*pseudo-rule*))
			   matches (N (net-nodes net))
			   (B 0) (FB 0) (S ft) (last (ft-next (ft-end ft)))
			   (A 0) (SL 0) (TRIED 0) lf)
  (declare-c-array stack)
  (declare-fixnum-array fstack)
  (declare (type fixnum B FB A SL TRIED))
  
  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (1+ B))))
    (choice-A () `(f-aref fstack FB))
    (choice-SL () `(f-aref fstack (the fixnum (+ FB 1))))
    (choice-tried () `(f-aref fstack (the fixnum (+ FB 2))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (1+ B)) ,x))
    (set-choice-A (x) `(f-aset fstack FB ,x))
    (set-choice-SL (x) `(f-aset fstack (the fixnum (+ FB 1)) ,x))
    (set-choice-tried (x) `(f-aset fstack (the fixnum (+ FB 2)) ,x))
    (next-choice () `(prog () (incf B 2) (incf FB 3)))
    (prev-choice () `(prog () (decf B 2) (decf FB 3)))
    (choice-stack-empty? () `(zerop B))
    )
   
   (prog ()
	 
   TOP
   (when (eq S last) (go DONE))
   (when (null N) (go BACKTRACK))
   ;(princ "Term = ")(print-term ft)(terpri)

   (unless (var? (ft-symbol S))
	   (setf N (slot-val (assoc-slot (slot-num S) N)))
	   (setf S (ft-next S))
	   (go TOP))
   
   (setf A 1)
   (setf TRIED 0)
   (setf SL 0)
   
   SKIP-SUBTERM
   (while (plusp A)
       ;(format t "A = ~S~%" A)
       (until (>= SL (the fixnum *slotarray-size*))
	   (when (slot-val (assoc-slot SL N))
		 (break-from-loop))
	   (incf SL))
       (unless (< SL (the fixnum *slotarray-size*)) (go BACKTRACK))
       (incf TRIED)
       (when (< TRIED (the fixnum (node-slots-used N)))
	     ;(princ "Pushing choicepoint")(terpri)
	     (set-choice-N N)
	     (set-choice-S S)
	     (set-choice-A A)
	     (set-choice-SL SL)
	     (set-choice-tried TRIED)
	     (next-choice))
       (decf A)
       (unless (= SL 0)
	       (incf A (slot-arity SL)))
       (setf N (slot-val (assoc-slot SL N)))
       (setf SL 0)
       (setf TRIED 0)
       )
   (setf S (ft-next S))
   (go TOP)
   
   BACKTRACK
   ;(format t "<<Backtracking>>~%")
   (when (choice-stack-empty?)
	 (return matches))
   (prev-choice)
   (setf N (choice-N))
   (setf S (choice-S))
   (setf A (choice-A))
   (setf SL (the fixnum (1+ (choice-SL))))
   (setf TRIED (choice-tried))
   (go SKIP-SUBTERM)

  
   DONE
   (setf lf N)
   (while lf
     (let ((trail (when (leaf-subterm lf) (match-vars ft (leaf-subterm lf)))))
       (when trail
	    ;; If rewriting, do the right thing depending on whether
            ;; rule is ordinary or a pseudo-rule
	    (when (or (not rewrite?)
		      (and is-prule (nofail-rewrite2 (leaf-subterm lf) trail rule))
		      (and (not is-prule) (ft-rewrite2 (leaf-subterm lf) trail rule)))
		  (if rewrite?
		      ;; The rewrite could make containing subterms
		      ;; incompatible with their indexing paths,
		      ;; so we make them "invisible" to subsequent matches.
		      (sloop for b in (eqn-backpointers (leaf-eqn lf))
			     do (setf (leaf-subterm b) nil))
		    (restore-vars trail))
		  (unless (member lf matches)
			  (push1 lf matches)))))
     (setf lf (leaf-next lf)))
   (go BACKTRACK)
   
   ))))


;; Add a rewrite rule to the indexing structures.
(proclaim '(function index-rule (t t t) nil))
(defun index-rule (eqn lhs-net subterm-net
		     &aux backpointers)
 (disabling-interrupts
  (setf backpointers (eqn-backpointers eqn))
  (unless (var? (ft-symbol (eqn-lhs eqn)))
	  (push1 (net-insert (eqn-lhs eqn) eqn lhs-net) backpointers))
  (when subterm-net
	(do-ft (eqn-lhs eqn)
	       :skip-first
	       (unless (var? (ft-sym))
		       (push1 (net-insert (ft) eqn subterm-net)
			      backpointers)))
	(do-ft (eqn-rhs eqn)
	       (unless (var? (ft-sym))
		       (push1 (net-insert (ft) eqn subterm-net)
			      backpointers)))
	)
  (setf (eqn-backpointers eqn) backpointers)
  ))
  

;; Delete all subterms of an equation from the indexing structures.
(proclaim '(function delete-from-nets (t) nil))
(defun delete-from-nets (e &aux (b (eqn-backpointers e)))
  (disabling-interrupts
   (while b
     (net-delete (pop1 b)))
   (setf (eqn-backpointers e) nil)))


;; Find an equation which subsumes the given one.
(proclaim '(function net-find-subsuming-eq (t t) t))
(with-stacks ((*stack* :stack))
(defun net-find-subsuming-eq (eqn net
    &aux (ft (eqn-lhs eqn))
         (stack (get-c-array *stack*))
	 (N (net-nodes net))
	 (S ft)
	 (B 0)
	 (end (ft-next (ft-end ft)))
	 lf)
  (declare-c-array stack)
  (declare (type fixnum B))
  
  (block
   FIND-SUBSUMING
  (when (null N) (return-from net-find-subsuming-eq nil))

  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (1+ B))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (1+ B)) ,x))
    (next-choice () `(incf B 2))
    (prev-choice () `(decf B 2))
    (choice-stack-empty? () `(zerop B))
    )
  (prog ()
    
    TOP
    (until (eq S end)
      (let ((var-slot (assoc-slot 0 N)))
	 (if (slot-val var-slot)
	     (progn
	       (unless (var? (ft-symbol S))
		       (set-choice-N N)
		       (set-choice-S S)
		       (next-choice))
	       (setf N (slot-val var-slot))
	       (setf S (ft-next (ft-end S))))
	   (progn
	     (when (var? (ft-symbol S)) (go BACKTRACK))
	     (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
		     (go BACKTRACK))
	     (setf S (ft-next S))))))
    
    (setf lf N)
    (while lf
      (when (subsumes (leaf-eqn lf) eqn)
	    (return-from net-find-subsuming-eq t))
      (setf lf (leaf-next lf)))
    
    BACKTRACK
    (when (choice-stack-empty?)
	  (return-from net-find-subsuming-eq nil))
    (prev-choice)
    (setf N (choice-N))
    (setf S (choice-S))
    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
	    (go BACKTRACK))
    (setf S (ft-next S))
    (go TOP)
    )))))


;; Glue to use net-find-subsuming-eq
(proclaim '(function find-subsuming-eq (t t) t))
(defun find-subsuming-eq (e net)
  (cond ((net-find-subsuming-eq e net) t)
	((progn (rotatef (eqn-lhs e) (eqn-rhs e))
		(prog1 (net-find-subsuming-eq e net)
		  (rotatef (eqn-lhs e) (eqn-rhs e))))
	 t)
	(t nil)))

;; Find equations subsumed by the given one.
(proclaim '(function find-subsumed-eqs (t t t) nil))
(defun find-subsumed-eqs (e net &aux matches m)
  ;; Don't use find-matches until subsumes is modified
  ;; to handle e-matching.
  (setf matches (net-find-matches e net nil))
  (until (null matches)
	 (setf m  (leaf-eqn (pop1 matches)))
	 (when (subsumes e m)
	       (check-abort-pairs m)
	       (delete-eqn m)
	       (delete-from-nets m)
	       (print-subsumed m e)
	       (kill-eqn m)))
  (rotatef (eqn-lhs e) (eqn-rhs e))
  (setf matches (net-find-matches e net nil))
  (until (null matches)
	 (setf m (leaf-eqn (pop1 matches)))
	 (when (subsumes e m)
	       (check-abort-pairs m)
	       (delete-eqn m)
	       (delete-from-nets m)
	       (print-subsumed m e)
	       (kill-eqn m)))
  (rotatef (eqn-lhs e) (eqn-rhs e))
  )



;; Add a failure to the net and delete any equations which
;; it subsumes.
(proclaim '(function index-unique-failure (t) t))
(defun index-unique-failure (e)
  (find-subsumed-eqs e *failures-net*)
  (add-failure-to-nets e)
  t)

;; Add a failure to the index structures.
(defun add-failure-to-nets (e)
  (index-rule e *subterm-net* *subterm-net*)
  (index-rule e *failures-net* nil))

;; When a new function symbol is introduced, it is necessary to
;; update the nets, since the number of slots in each
;; node is equal to the number of function symbols in the 
;; system.  
(proclaim '(function rebuild-nets () nil))
(defun rebuild-nets ()
  (init-nets)
  (rebuild-nets-aux *lhs-net*)
  (rebuild-nets-aux *subterm-net*)
  (rebuild-nets-aux *failures-net*)
  (rebuild-nets-aux *pseudo-rules-net*)
  )

;; Traverse a net, and replace the slotarray of each node
;; with a new slotarray to accomodate an additional 
;; function symbol.
(proclaim '(function rebuild-nets-aux (t) nil))
(with-stacks ((*stack* :stack))
(defun rebuild-nets-aux (net &aux (stack (get-c-array *stack*))
			  (N (net-nodes net)) (B 0) lf
			  newslots oldslots)
  (declare-c-array stack)
  (declare (type (array t) newslots oldslots))
  (declare (type fixnum B))
  (declare (object oldslots))
  
  (block
   REBUILD
   (when (null N) (return-from rebuild-nets-aux))
   
   (macrolet
    ;; Choicepoint structure
    (
     (choice-N () `(c-aref stack B))
     (set-choice-N (x) `(c-aset stack B ,x))
     (next-choice () `(incf B 1))
     (prev-choice () `(decf B 1))
     (choice-stack-empty? () `(zerop B))
     )
    
    (prog
     ()
     TOP
     
     (setf newslots (new-slot-array))
     (setf oldslots (node-slots N))
     (floop :for SL :below (1- (the fixnum *slotarray-size*))
	:do (progn
	      (let ((old-slot (aref oldslots SL))
		    (new-slot (aref newslots SL)))
		(setf (slot-val new-slot) (slot-val old-slot))
		(free-cons old-slot)
		(setf (aref oldslots SL) nil)
		(if (node-p (slot-val new-slot))
		    (progn
		      (setf (node-prev-slot (slot-val new-slot)) new-slot)
		      (set-choice-N (slot-val new-slot))
		      (next-choice))
		  (progn
		    (setf lf (slot-val new-slot))
		    (while lf
		      (setf (leaf-parent-slot lf) new-slot)
		      (setf lf (leaf-next lf))))))))
     ;; Currently, oldslots becomes garbage that must be 
     ;; retrieved by the LISP garbage collector.
     (setf (node-slots N) newslots)

     BACKTRACK
     (when (zerop B) (return-from rebuild-nets-aux))
     (prev-choice)
     (setf N (choice-N))
     (go TOP)
     )))))


#|
;; Older version of rebuild-nets: Delete all equations in the
;; system from the indices, then rebuild the nets from scratch.
;; This is considerably slower than the current version.
(proclaim '(function rebuild-nets () nil))
(defun rebuild-nets ()
  (do-queue e *rules* (delete-from-nets e))
  (do-queue e *marked-rules* (delete-from-nets e))
  (init-nets)
  (do-queue e *rules*
	    (case (eqn-type e)
		  (*rewrite-rule*
		   (index-rule e *lhs-net* *subterm-net*))
		  (*pseudo-rule*
		   (index-rule e *pseudo-rules-net* *subterm-net*))
		  (t
		   (add-failure-to-nets e))))
  (do-queue e *marked-rules*
	    (case (eqn-type e)
		  (*rewrite-rule*
		   (index-rule e *lhs-net* *subterm-net*))
		  (*pseudo-rule*
		   (index-rule e *pseudo-rules-net* *subterm-net*))
		  (t
		   (add-failure-to-nets e))))
  )
|#
