;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: newf.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Automatic function symbol introduction.

(proclaim '(function new-fsym (t) t))
(defun new-fsym (e &aux vars r1 r2 newf t1 t2 new-term (arity 0))
  (declare (type fixnum arity))
  (when *disable-new-fsyms* (return-from new-fsym nil))
  (setf (eqn-type e) '*failure*)
  (check-for-bad-eqn e)			;Check for trivial algebra, etc.
  (unless (and (not (postpone-eqn e))	;Decide whether to introduce a new fsym
	       (or *user-decide-new-fsyms*
		   (progn (format t "~%Confirm new fsym for~%") (print-eqn e)
			  (y-or-n-p "~%?"))))
	  (return-from new-fsym nil))
  ;; Assign e a unique ID if it doesn't already have one.
  (unless (eqn-id e) (setf (eqn-id e) (incf-eqn-counter)))
  (print-failure e)
  ;; Locate the variables common to both sides
  (setf vars (common-vars (eqn-lhs e) (eqn-rhs e)))
  ;; Generate a unique new function symbol
  (while
   (get-fsym-id (setf newf (intern
			  (format nil "F~S" (incf *new-fsym-counter*))))))
  (setf t1 (eqn-lhs e))
  (setf t2 (eqn-rhs e))
  ;; Determine the arity of the new function symbol
  (dolist (v vars) (incf arity))
  ;; Place the symbol appropriately in the Knuth-Bendix
  ;; lex ordering, and assign it a weight.  Note that
  ;; the RPO ordering will insert it automatically into
  ;; its own precedence relation when it needs to.
  (case arity
	(1
	 ;; If its a unary symbol, make it a special last symbol
	 ;; for the Knuth-Bendix ordering (i.e., assume it's an inverse),
	 ;; unless there's already one
	 (if *special-last-fsym*
	     (new-front-fsym (declare-fsym newf 1 1))
	   (new-special-last-fsym (declare-fsym newf 1 0))))
	(0
	 ;; Add constants to the front of the KB lex ordering, with
	 ;; weight *w0*
	 (new-front-fsym (declare-fsym newf 0 *w0*)))
	(t
	 ;; Add other fsyms to the front of the KB lex ordering,
	 ;; with weight = min weight of original lead symbols.
	 (new-front-fsym
	    (declare-fsym newf arity
			  (min (fsym-weight (ft-symbol t1))
			       (fsym-weight (ft-symbol t2)))))))

  ;; Build the new rules.
  (setf new-term (get-ft (get-fsym-id newf)))
  (dolist (v vars)
	  (ft-append new-term (get-ft (ft-symbol v))))
  (free-list vars)
  (setf r1 (new-eqn))
  (setf r2 (new-eqn))
  (setf (eqn-id r1) (incf-eqn-counter))
  (setf (eqn-id r2) (incf-eqn-counter))
  (setf (eqn-supported r1) (eqn-supported e))
  (setf (eqn-supported r2) (eqn-supported e))
  (setf (eqn-lhs r1) t1)
  (setf (eqn-lhs r2) t2)
  (setf (eqn-rhs r1) new-term)
  (setf (eqn-rhs r2) new-term)
  (setf (eqn-parents r1) (cons1 (eqn-id e) '*new-fsym*))
  (setf (eqn-parents r2) (cons1 (eqn-id e) '*new-fsym*))
  (setf (eqn-fsym-def-p r1) (get-fsym-id newf))
  (setf (eqn-fsym-def-p r2) (get-fsym-id newf))
  (setf (eqn-type r1) '*new-pair*)
  (setf (eqn-type r2) '*new-pair*)
  (copy-eqn-slots r1 r1)
  (copy-eqn-slots r2 r2)
  (normalize r1)
  (normalize r2)
  (enqueue-eqn r1 *pairs*)
  (enqueue-eqn r2 *pairs*)
  ;; Destroy the original equation.
  (kill-eqn e)
  (kill-ft new-term)
  t
  )


;; If a var occurs on either side of e, then either the algebra is 
;; trivial, or we have an equation that won't work for unification
;; completion.  In either case, we give up.
(proclaim '(function check-for-bad-eqn (t) t))
(defun check-for-bad-eqn (e)
  (when (or (var? (ft-symbol (eqn-lhs e)))
	    (var? (ft-symbol (eqn-rhs e))))
	(format t "~%*** Inappropriate equation detected ***~%")
	(print-eqn e)
	(error "Inappropriate equation detected")
	))

;; Decide whether to introduce a new function symbol.
;; Returns nil if a new symbol should be introduced,
;; and t if not.
;; These heuristics are basically ad-hoc, but they
;; seem to be pretty effective for some problems.
(proclaim '(function postpone-eqn (t) t))
(defun postpone-eqn (e)
  (cond
   (*user-decide-new-fsyms*
    (format t "~%~%New function symbol for ~%")
    (print-eqn e)
    (if (y-or-n-p "?") nil t))

   ;; Note: A simple linear permuter will have been caught
   ;; before entering the new function symbol routines.
   ;; This clause is to catch more general permutations.
   ((and *encourage-simple-permuters*
	 (permutation? (eqn-lhs e) (eqn-rhs e)))
    nil)

   ((and *encourage-simple-permuters*
	 (vars-contained (eqn-lhs e) (eqn-rhs e))
	 (vars-contained (eqn-rhs e) (eqn-lhs e))
	 (sides-match e)
	 )
    nil)

   ((and *encourage-simple-permuters*
	 (simple-args (eqn-lhs e))
	 (simple-args (eqn-rhs e))
	 (sides-unify e))
    nil)

   ;; If the equation already has a new function symbol
   ;; in it, then don't introduce yet another one.  This
   ;; heuristic helps prevent introducing an infinite 
   ;; number of new symbols in problems like the Higman groups.
   ((or (has-new-fsym (eqn-lhs e))
	(has-new-fsym (eqn-rhs e)))
    t)

   ;; If the lhs and rhs match, then another non-orientable
   ;; equation with matching sides will also be generated if
   ;; a new function symbol is introduced, causing an inifinte
   ;; number of new symbols to be added.
   ((and (vars-contained (eqn-lhs e) (eqn-rhs e))
	 (vars-contained (eqn-rhs e) (eqn-lhs e))
	 (sides-match e))
    t)

   (t nil)))

;; Find the vars common to t1 and t2.
(proclaim '(function common-vars (t t) t))
(with-stacks ((*trail* :trail))
(defun common-vars (t1 t2 &aux vars)
  (with-local-trail *trail*
    (do-ft t1
      (when (and (var? (ft-sym)) (unbound-var? (ft-sym)))
	    (trail (ft-sym))
	    (set-binding (ft-sym) t)))
    (do-ft t2
      (when (and (var? (ft-sym)) (bound-var? (ft-sym)))
	    (unless (sloop for x in vars
			   thereis (same-symbol? (ft-sym)
						 (ft-symbol x)))
		    (push1 (ft) vars)))))
    vars))

;; See if a new function symbol occurs in term.
(proclaim '(function has-new-fsym (t) t))
(defun has-new-fsym (term)
  (not (do-ft term
	      (when (and (not (var? (ft-sym)))
			 (> (ft-sym) *last-original-fsym*))
		    (abort)))))

;; See if term contains an original function symbol.
(proclaim '(function has-original-fsym (t) t))
(defun has-original-fsym (term)
  (not (do-ft term
	 (when (and (not (var? (ft-sym)))
		    (<= (ft-sym) *last-original-fsym*))
	       (abort)))))
	       

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The following stuff is a bit experimental.  The idea is that,
;;; when an equation defining a new function f symbol becomes 
;;; reducible by some rule r, then we should restart completion with
;;; the original set of input equations and r, and those equations 
;;; not containing f.  Effectively, we "backtrack" out of the
;;; introduction of a new symbol.
;;;   This was motivated by the one-law group problem (group1),
;;; which generates the identity too late to be useful (without
;;; appropriate postponement heuristics).
;;;   Experiments seem to indicate that this stuff should
;;; normally be switched off; the heuristic postponement routine
;;; above seems to be a more effective means of dealing with 
;;; function symbols.

;; Given that r2 is reducible by r1, is r2 a function symbol
;; definition and, if so, should that symbol be deleted and
;; completion restarted without it?
(defmacro reducible-fsym-def-p (r1 r2)
  `(and (eqn-fsym-def-p ,r2)
	(not (eq (eqn-fsym-def-p ,r1)
		 (eqn-fsym-def-p ,r2)))))

;; See if f is used in term.
(proclaim '(function fsym-occurs (fixnum t) t))
(defun fsym-occurs (f term)
  (declare (type fixnum f))
  (not
   (do-ft term
      (when (same-symbol? f (ft-sym))
	    (abort)))))

;; Delete equations containing the given function symbols
(proclaim '(function delete-eqs-with-fsyms (t t) nil))
(defun delete-eqs-with-fsyms (fs q)
  (do-queue e q
    (dolist (f fs)
	    (declare (type fixnum f))
	    (when (or (fsym-occurs f (eqn-lhs e))
		      (fsym-occurs f (eqn-rhs e)))
		  (check-abort-pairs e)
		  (delete-eqn e)
		  (print-deleted e)
		  (delete-from-nets e)
		  (kill-eqn e)
		  (break-from-loop)))))

;; Restart completion after finding a reducible function symbol definition.
(proclaim '(function new-fsym-restart (t) nil))
(defun new-fsym-restart (flist)
 ;(declare (ignore flist))
  (when *disable-fsym-restart* (return-from new-fsym-restart))
  (format t "~%~% *** Reducible fsym definition found. Restarting. ***~%")
  
  (delete-eqs-with-fsyms flist *rules*)
  (delete-eqs-with-fsyms flist *marked-rules*)
  (delete-eqs-with-fsyms flist *pairs*)
  
  (do-queue e *restart-q* (enqueue-eqn (dupl-eqn e) *pairs*))
  (do-queue e *marked-rules*
	    (delete-eqn e)
	    (enqueue-eqn e *rules*))
  (setf *p1* nil)
  (when *p2* (eqn-ys-to-xs *p2*))
  (setf *p2* nil)
  )

(proclaim '(function only-var-args (t) t))
(defun only-var-args (term)
  (do-ft term
    (when (and (not (eq (ft) term))
	       (not (var? (ft-sym))))
	  (abort))))

;; Does term contain only var and constant args?
(proclaim '(function simple-args (t) t))
(defun simple-args (term)
  (do-ft term
    :skip-first
    (when (and (not (var? (ft-sym)))
	       (> (fsym-arity (ft-sym)) 0))
	  (abort))))

(proclaim '(function sides-match (t) t))
(defun sides-match (e &aux tr ecopy)
  (setf ecopy (dupl-eqn e))
  (setf (eqn-parents ecopy) nil)
  (setf tr (match-terms (eqn-lhs ecopy) (eqn-rhs e)))
  (when tr
	(restore-vars tr)
	(kill-eqn ecopy)
	(return-from sides-match t))
  (setf tr (match-terms (eqn-rhs ecopy) (eqn-lhs e)))
  (when tr
	(restore-vars tr)
	(kill-eqn ecopy)
	(return-from sides-match t))
  nil)


(proclaim '(function both-sides-match (t) t))
(defun both-sides-match (e &aux tr ecopy)
  (setf ecopy (dupl-eqn e))
  (setf (eqn-parents ecopy) nil)
  (setf tr (match-terms (eqn-lhs ecopy) (eqn-rhs e)))
  (unless tr
	  (kill-eqn ecopy)
	  (return-from both-sides-match nil))
  (restore-vars tr)
  (setf tr (match-terms (eqn-rhs ecopy) (eqn-lhs e)))
  (unless tr
	  (kill-eqn ecopy)
	  (return-from both-sides-match nil))
  (restore-vars tr)
  (kill-eqn ecopy)
  t)

(proclaim '(function sides-unify (t) t))
(defun sides-unify (e &aux tr ecopy)
  (setf ecopy (dupl-eqn e))
  (setf (eqn-parents ecopy) nil)
  (setf tr (unify-terms (eqn-lhs ecopy) (eqn-rhs e)))
  (when tr
	(restore-vars tr)
	(kill-eqn ecopy)
	(return-from sides-unify t))
  (setf tr (unify-terms (eqn-rhs ecopy) (eqn-lhs e)))
  (when tr
	(restore-vars tr)
	(kill-eqn ecopy)
	(return-from sides-unify t))
  nil)


