;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: misc.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Miscellaneous macros and functions.

;;; Note to the uninitiated: Some of the routines below provide
;;; alternatives to builtin lisp functions, but they allocate and
;;; deallocate cons cells to the pool maintained in "heap.lsp".
;;; Push1, append1, mapcar1, and cons1 are always safe to use.
;;; Pop1 and delete1 aren't always safe.  Use them only if you 
;;; understand what's going on -- i.e. make sure they don't free
;;; a cons cell which might still be in use by some other part of
;;; the program.

(eval-when (load eval) (setf *print-level* 3))
(eval-when (load eval) (setf *print-length* 20))
(eval-when (load eval) (setf *print-pretty* nil))


;; (while test :return result body)
(defmacro while (test &body body)
  (if (eq (first body) :return)
      `(do () ((not ,test) ,(second body)) ,@(nthcdr 2 body))
    `(do () ((not ,test)) ,@body)))

;; (until test :return result body)
(defmacro until (test &body body)
  (if (eq (first body) :return)
      `(do () (,test ,(second body)) ,@(nthcdr 2 body))
  `(do () (,test) ,@body)))

;; (repeat test :return result body)
;; Do test *after* body
(defmacro repeat (test &body body)
  (if (eq (first body) :return)
      `(loop ,@(nthcdr 2 body) (when ,test (return ,(second body))))
    `(loop ,@body (when ,test (return)))))

(defmacro break-from-loop (&optional val) `(return ,val))

;; Causes KCL to emit better code in some cases.  If array has been
;; declared type (array t) and subscript has been declared type 
;; fixnum, then you can use aref instead of aref-with-decl.
(defmacro aref-with-decl (array subscript)
  `(aref (the (array t) ,array) (the fixnum ,subscript)))

;; Delete x from L, returning the cons cell for x to the heap.
(proclaim '(function delete1 (t t) t))
(defun delete1 (x L)
  (do ((prev nil current)
       (current L (cdr current)))
      ((null current))
      (when (eql x (car current))
	    (if prev
		(setf (cdr prev) (cdr current))
	      (setf L (cdr current)))
	    (free-cons current)
	    (return-from delete1 L))))

;; Cons x and y, allocating from the heap.
(defmacro cons1 (x y) `(new-cons ,x ,y))

;; Push item onto place, allocating from the heap.
(defmacro push1 (item place)
  (when (symbolp place)
        (return-from push1 `(setq ,place (cons1 ,item ,place))))
  (multiple-value-bind (vars vals stores store-form access-form)
      (get-setf-method place)
    `(let* ,(mapcar #'list
		    (append vars stores)
		    (append vals (list (list 'cons1 item access-form))))
       ,store-form)))

;; Pop place, returning the cons cell to the heap.
(defmacro pop1 (place)
  (when (symbolp place)
        (return-from pop1
          (let ((temp1 (gensym))
		(temp2 (gensym)))
            `(let ((,temp1 (car ,place))
		   (,temp2 ,place))
                (setq ,place (cdr ,place))
		(free-cons ,temp2)
                ,temp1))))
  (multiple-value-bind (vars vals stores store-form access-form)
      (get-setf-method place)
      (let ((temp (gensym)))
	`(let* ,(mapcar #'list
			(append vars stores)
			(append vals (list (list 'cdr access-form))))
	   (let ((,temp ,access-form))
	     (prog1 (car ,temp)
	            (free-cons ,temp)
	            ,store-form))))))

;; Constructive reverse, allocating from the heap.
(proclaim '(function rev1 (t) t))
(defun rev1 (list &aux result)
  (dolist (l list result)
	  (push1 l result)))

;; Constructive append, allocating from the heap.
(proclaim '(function append1 (t t) t))
(defun append1 (x y)
  (if (null x)
      y
    (cons1 (car x) (append1 (cdr x) y))))

;; Mapcar, allocating from heap.
(proclaim '(function mapcar1 (t t) t))
(defun mapcar1 (func list &aux result)
  (dolist (x list (nreverse result))
	  (push1 (funcall func x) result)))

;; (floop :for fixnum-var :below fixnum-limit :do code)
;; Sloop seems to do some bad things with fixnums in this kind of 
;; loop.  (Specifically, it generates a temp object which should 
;; really be a fixnum; so some unnecessary CMPmake_fixnum and 
;; number_compare's are inserted in the C code.) Floop will
;; generate better code in this case.
(defmacro floop (&key for (from 0) below do)
  (let ((end (gentemp "END")) (label (gentemp "LABEL")))
    `(let ((,for (the fixnum ,from))
	   (,end (the fixnum ,below)))
       (declare (type fixnum ,for ,end))
       (block ()
	      (tagbody
	       ,label
	       (when (>= ,for ,end) (return))
	       ,do
	       (incf ,for)
	       (go ,label))))))


;; Temporarily disable interrupts in AKCL.  This is useful to
;; maintain consistency of data structures and avoid dangling
;; pointers in case the user interrupts the program.
#+kcl (defmacro disabling-interrupts (&body code)
	`(let ((si::*interrupt-enable* nil))
	   ,@code))

#-kcl (defmacro disabling-interrupts (&body code)
	`(progn ,@code))


