;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: kbrpo.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Merged Knuth-Bendix and Recursive-Path ordering.
;;; The KB ordering is augmented with a restricted version
;;; of the RPO procedure which assumes no precedence relation
;;; among symbols.  Effectively, we add a multiset extension
;;; component to the KB ordering, along with some of the features
;;; of the recursive path ordering.  So, for instance, equations like
;;; (-y) + (-x) = x + y can be oriented using the kb-rpo ordering,
;;; while the kb ordering alone would fail.  We also gain the 
;;; efficiency benefits of the KB ordering, without being too 
;;; restricted by it. 

(proclaim '(function kb-rpo (t t) fixnum))
(defun kb-rpo-compare (t1 t2 &aux v1 v2)
  (let ((w1 (kb-weight t1))
	(w2 (kb-weight t2)))
    (declare (type fixnum w1 w2))
    (cond
     ((and (setf v2 (vars-subset-of t1 t2)) (< w2 w1)) *lr*)
     ((and (setf v1 (vars-subset-of t2 t1)) (< w1 w2)) *rl*)
     ((kb-rpo-greater t1 t2) *lr*)
     ((kb-rpo-greater t2 t1) *rl*)
     ((= w1 w2)
      (cond ((var? (ft-symbol t2))
	     (cond ((var? (ft-symbol t1)) 0)
		   ((and v2 (special-last-func-only t1 t2)) *lr*)
		   (t 0)))
	    ((var? (ft-symbol t1))
	     (cond ((and v1 (special-last-func-only t2 t1)) *rl*)
		   (t 0)))
	    (t
	     (let ((lw1 (fsym-lex-weight (ft-symbol t1)))
		   (lw2 (fsym-lex-weight (ft-symbol t2))))
	       (declare (type fixnum lw1 lw2))
	       (cond
		((> lw1 lw2) (if v2 *lr* 0))
		((< lw1 lw2) (if v1 *rl* 0))
		(t
		 (let ((lex (kb-rpo-lex-compare t1 t2)))
		   (declare (type fixnum lex))
		   (cond ((and v2 (= lex *lr*)) *lr*)
			 ((and v1 (= lex *rl*)) *rl*)
			 (t 0)))))))))
	    (t 0))))
  
(proclaim '(function kb-rpo-lex-compare (t t) fixnum))
(defun kb-rpo-lex-compare (ft1 ft2 &aux (x 0))
  (declare (type fixnum x))
  (assert (same-symbol? (ft-symbol ft1) (ft-symbol ft2)))
  (macrolet
   ((exit (x) `(progn (free-permutation b1)
			(free-permutation b2)
			(return-from kb-rpo-lex-compare ,x))))
  (cond
   ((not (fsym-mutations (ft-symbol ft1)))
    (kb-rpo-lex-recurse ft1 ft2))
   ((not (zerop (setf x (kb-rpo-lex-recurse ft1 ft2))))
    (let* ((p1 (fsym-mutations (ft-symbol ft1)))
	   (b1 (new-permutation))
	   (b2 (new-permutation)))
      (declare (object p1))
      (init-permute ft1 p1 b1)
      (init-permute ft2 p1 b2)
      (let ((p2 p1)) (declare (object p2))
	 (while p2
	   (permute ft2 (car p2) b2)
	   (setf p2 (cdr p2))
	   (unless (= x (kb-rpo-lex-recurse ft1 ft2))
		   (quit-permute ft2 b2)
		   (exit 0)))
	 (quit-permute ft2 b2))
      (while p1
	(permute ft1 (car p1) b1)
	(setf p1 (cdr p1))
	(unless (= x (kb-rpo-lex-recurse ft1 ft2))
		(quit-permute ft1 b1)
		(exit 0))
	(let ((p2 (fsym-mutations (ft-symbol ft2))))
	  (declare (object p2))
	  (while p2
	    (permute ft2 (car p2) b2)
	    (setf p2 (cdr p2))
	    (unless (= x (kb-rpo-lex-recurse ft1 ft2))
		    (quit-permute ft1 b1)
		    (quit-permute ft2 b2)
		    (exit 0)))
	  (quit-permute ft2 b2)))
      (quit-permute ft1 b1)
      (exit x)))
   (t 0))))
	     
	 
      
(proclaim '(function kb-rpo-lex-recurse (t t) fixnum))
(defun kb-rpo-lex-recurse (ft1 ft2 &aux (last (ft-next (ft-end ft1))))
  (setf ft1 (ft-next ft1))
  (setf ft2 (ft-next ft2))
  (until (eq ft1 last)
	 :return 0
	 (let ((s (kb-rpo-compare ft1 ft2)))
	   (declare (type fixnum s))
	   (unless (zerop s)
		   (return-from kb-rpo-lex-recurse s))
	   (unless (e-equal ft1 ft2)
		   (return-from kb-rpo-lex-recurse 0)))
	 (setf ft1 (ft-next (ft-end ft1)))
	 (setf ft2 (ft-next (ft-end ft2)))))


(proclaim '(function kb-rpo-multiargs-greater (t t) t))
(defun kb-rpo-multiargs-greater (t1 t2 &aux ml1 ml2)
  (do-args t1 (push1 (ft) ml1))
  (do-args t2 (push1 (ft) ml2))
  (let* (
	 tmp
	 (m2-m1
	  (progn
	    (sloop for x in ml2
		   when (> (count x ml2 :test #'e-equal-func)
			   (count x ml1 :test #'e-equal-func))
		   do (push1 x tmp)) 
	    tmp))
	 (m1-m2
	  (progn
	    (setf tmp nil)
	    (sloop for x in ml1
		   when (> (count x ml1 :test #'e-equal-func)
			   (count x ml2 :test #'e-equal-func))
		   do (push1 x tmp))
	    tmp))
	 (result
	  (and m1-m2
	       (sloop for x in m2-m1
		      always
		      (sloop for y
			     in m1-m2
			     thereis (= (kb-rpo-compare y x) *lr*)
			     ))))
	  )
    (free-list m2-m1)
    (free-list m1-m2)
    result))

(proclaim '(function kb-rpo-greater (t t) t))
(defun kb-rpo-greater (t1 t2
		       &aux (sym1 (ft-symbol t1))
		            (sym2 (ft-symbol t2)))
  (cond
   ((var? sym1) nil)
   ((var? sym2) (occurs? sym2 t1))
   (t
    (cond
     ((op-equal sym1 sym2 precedence)
	(kb-rpo-multiargs-greater t1 t2))
     (t (some-arg t1
	  (or (= (kb-rpo-compare (ft) t2) *lr*)
	      (let (p) (declare (special p))
		   (equiv-terms-aux (ft) t2)))))
     ))))
