;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: kb.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Knuth-bendix ordering, extended to handle permutations.



(defconstant *lr* -1)			;Equation is orientable as-is
(defconstant *rl* 1)			;Equation is orientable if flipped
(defconstant *none* 0)			;Equation is not orientable

;; Compare t1 and t2 using the Knuth-Bendix ordering
(proclaim '(function kb-compare (t t) fixnum))
(defun kb-compare (t1 t2 &aux v1 v2)
  (let ((w1 (kb-weight t1))
	(w2 (kb-weight t2)))
    (declare (type fixnum w1 w2))
    (cond
     ((and (setf v2 (vars-subset-of t1 t2)) (< w2 w1)) *lr*)
     ((and (setf v1 (vars-subset-of t2 t1)) (< w1 w2)) *rl*)
     ((= w1 w2)
      (cond ((var? (ft-symbol t2))
	     (cond ((var? (ft-symbol t1)) 0)
		   ((and v2 (special-last-func-only t1 t2)) *lr*)
		   (t 0)))
	    ((var? (ft-symbol t1))
	     (cond ((and v1 (special-last-func-only t2 t1)) *rl*)
		   (t 0)))
	    (t
	     (let ((lw1 (fsym-lex-weight (ft-symbol t1)))
		   (lw2 (fsym-lex-weight (ft-symbol t2))))
	       (declare (type fixnum lw1 lw2))
	       (cond
		((> lw1 lw2) (if v2 *lr* 0))
		((< lw1 lw2) (if v1 *rl* 0))
		(t
		 (let ((lex (lex-compare t1 t2)))
		   (declare (type fixnum lex))
		   (cond ((and v2 (= lex *lr*)) *lr*)
			 ((and v1 (= lex *rl*)) *rl*)
			 (t 0)))))))))
	    (t 0))))


;; Compute the Knuth-Bendix weight of a term
(proclaim '(function kb-weight (t) fixnum))
(defun kb-weight (ft &aux (k 0) (w0 *w0*)
		     (weights (get-fixnum-array *fsym-weights*)))
  (declare (type fixnum k w0))
  (declare-fixnum-array weights)
  (macrolet
   ((fsym-weight (x) `(f-aref weights ,x)))
   (do-ft ft :return k
     (if (fsym? (ft-sym))
	 (incf k (fsym-weight (ft-sym)))
       (incf k w0)))))
  

;; Try lexicographic extension of KB ordering, making
;; sure that the result is valid under applicable permutations.
(proclaim '(function lex-compare (t t) fixnum))
(defun lex-compare (ft1 ft2 &aux (x 0))
  (declare (type fixnum x))
  (assert (same-symbol? (ft-symbol ft1) (ft-symbol ft2)))
  (macrolet
   ((exit (x) `(progn (free-permutation b1)
			(free-permutation b2)
			(return-from lex-compare ,x))))
  (cond
   ((not (fsym-mutations (ft-symbol ft1))) (lex-recurse ft1 ft2))
   ((not (= (the fixnum (setf x (lex-recurse ft1 ft2))) 0))
    (let* ((p1 (fsym-mutations (ft-symbol ft1)))
	   (b1 (new-permutation))
	   (b2 (new-permutation)))
      (declare (object p1))
      (init-permute ft1 p1 b1)
      (init-permute ft2 p1 b2)
      (let ((p2 p1)) (declare (object p2))
	 (while p2
	   (permute ft2 (car p2) b2)
	   (setf p2 (cdr p2))
	   (unless (= x (lex-recurse ft1 ft2))
		   (quit-permute ft2 b2)
		   (exit 0)))
	 (quit-permute ft2 b2))
      (while p1
	(permute ft1 (car p1) b1)
	(setf p1 (cdr p1))
	(unless (= x (lex-recurse ft1 ft2))
		(quit-permute ft1 b1)
		(exit 0))
	(let ((p2 (fsym-mutations (ft-symbol ft2))))
	  (declare (object p2))
	  (while p2
	    (permute ft2 (car p2) b2)
	    (setf p2 (cdr p2))
	    (unless (= x (lex-recurse ft1 ft2))
		    (quit-permute ft1 b1)
		    (quit-permute ft2 b2)
		    (exit 0)))
	  (quit-permute ft2 b2)))
      (quit-permute ft1 b1)
      (exit x)))
   (t 0))))
	     
	 
      
(proclaim '(function lex-recurse (t t) fixnum))
(defun lex-recurse (ft1 ft2 &aux (last (ft-next (ft-end ft1))))
  (setf ft1 (ft-next ft1))
  (setf ft2 (ft-next ft2))
  (until (eq ft1 last)
	 :return 0
	 (let ((s (kb-compare ft1 ft2)))
	   (declare (type fixnum s))
	   (unless (= s 0)
		   (return-from lex-recurse s))
	   (unless (or (e-equal ft1 ft2))
		   (return-from lex-recurse 0))
	   )
	 (setf ft1 (ft-next (ft-end ft1)))
	 (setf ft2 (ft-next (ft-end ft2)))))

;; Check that the number of occurrences of each var in t2 is less than
;; or equal to the number of occurrences in t1.
;; Note that the trail could be used to record the counts, but
;; the auxiliary stack forces C integers to be used and prevents
;; allocation of fixnums by KCL.  
(proclaim '(function vars-subset-of (t t) t))
(with-stacks ((*trail* :trail)(*bindings* :fixnum-stack))
(defun vars-subset-of (t1 t2 &aux (trail (get-fixnum-array *trail*))
				  (bindings (get-fixnum-array *bindings*))
				  (tr-top 0))
  (declare-fixnum-array trail)
  (declare-fixnum-array bindings)
  (declare (type fixnum tr-top))
  (macrolet
   (
    (set-binding (b val) `(f-aset bindings ,b (the fixnum ,val)))
    (var-binding (v) `(the fixnum (f-aref bindings (the fixnum ,v))))
    (unbound-var? (v) `(=  (the fixnum (f-aref bindings (the fixnum ,v))) 0))
    (trail (v) `(f-stack-push (the fixnum ,v) trail tr-top))
    )
   (do-ft t1
	  (when (var? (ft-sym))
		(if (unbound-var? (ft-sym))
		    (progn
		      (trail (ft-sym))
		      (set-binding (ft-sym) 1))
		  (set-binding (ft-sym)
			       (the fixnum (1+ (var-binding (ft-sym))))))))
   (prog1
       (do-ft t2
	 (when (var? (ft-sym))
	       (if (unbound-var? (ft-sym))
		   (abort)
		 (progn
		   (set-binding (ft-sym)
				(the fixnum (1- (var-binding (ft-sym)))))
		   (when (minusp (var-binding (ft-sym))) (abort))))))
     (until (= tr-top 0)
	    (let ((x (f-stack-pop trail tr-top)))
	      (declare (type fixnum x))
	    (set-binding x 0)))))))

;; Check that each var in t2 occurs in t1
(proclaim '(function vars-contained (t t) t))
(with-stacks ((*trail* :trail)(*bindings* :stack))
(defun vars-contained (t1 t2 &aux (trail (get-fixnum-array *trail*))
			          (tr-top 0)
				  (bindings (get-c-array *bindings*)))
  (declare-fixnum-array trail)
  (declare-c-array bindings)
  (declare (type fixnum tr-top))
  (macrolet
   (
    (set-binding (b val) `(c-aset bindings ,b ,val))
    (unbound-var? (v) `(not (c-aref bindings ,v)))
    (trail (v) `(f-stack-push ,v trail tr-top))
    )
   (do-ft t1
     (when (and (var? (ft-sym)) (unbound-var? (ft-sym)))
	   (trail (ft-sym))
	   (set-binding (ft-sym) t)))
   (prog1
       (do-ft t2
	 (when (and (var? (ft-sym)) (unbound-var? (ft-sym)))
	       (abort)))
     (until (= tr-top 0)
	    (let ((x (f-stack-pop trail tr-top)))
	      (declare (type fixnum x))
	    (set-binding x nil)))))))

;; For orienting equations like -(-x) = x.
(proclaim '(function special-last-func-only (t t) t))
(defun special-last-func-only (term var)
  (do-ft term
	 (if (var? (ft-sym))
	     (return-from special-last-func-only
			  (same-symbol? (ft-sym) (ft-symbol var)))
	   (unless (same-symbol? (ft-sym) *special-last-fsym*)
		   (abort)))))

;; Let the user decide whether to orient the equation.
(proclaim '(function user-orient (t) t))
(defun user-orient (e &aux answer)
  (unless *user-orient-failures*
	  (return-from user-orient nil))
  (when (find-subsuming-eq e *failures-net*)
	(return-from user-orient nil))
  (format t "~%~%Orient ")
  (print-eqn e)
  (do ()(nil)
      (format t "? [Y(es) F(lip) N(o) R(esume)]~%")
      (setf answer (read))
      (case answer
	    (y (return-from user-orient t))
	    (f (rotatef (eqn-lhs e) (eqn-rhs e))
	       (return-from user-orient t))
	    (n (return-from user-orient nil))
	    (r (setf *user-orient-failures* nil)
	       (return-from user-orient nil))
	    )))
	    

;; Initialize.
(proclaim '(function init-ordering () nil))
(defun init-ordering ()
  ;; Determine value of *w0*, the weight for vars
  (setf *w0* most-positive-fixnum)
  (sloop for f below *num-fsyms*
	 declare (fixnum f)
	 when (and (= (fsym-arity (to-fsym f)) 0)
		   (< (fsym-weight (to-fsym f)) *w0*))
	 do (setf *w0* (fsym-weight (to-fsym f))))
  (when (= *w0* most-positive-fixnum)
	(setf *w0* 1))

  ;; Check for unary zero-weighted fsym
  (setf *special-last-fsym* nil)
  (when (and (= (fsym-arity (to-fsym (1- *num-fsyms*))) 1)
	     (= (fsym-weight (to-fsym (1- *num-fsyms*))) 0))
	(setf *special-last-fsym* (to-fsym (1- *num-fsyms*))))

  ;; Create initial lex ordering
  (sloop for i below *num-fsyms*
	 do (setf (fsym-lex-weight (to-fsym i)) i))

  ;; So we can distinguish original fsyms from ones introduced by new-fsym
  (setf *last-original-fsym* (to-fsym (1- *num-fsyms*)))
  (when (eq *which-order* 'kb-rpo)
	(setf *precedence* nil))

  )

;; Insert a function symbol at the beginning of the lex ordering
(proclaim '(function new-front-fsym (t) nil))
(defun new-front-fsym (f)
  (setf (fsym-lex-weight f) -1)
  (dotimes (i *num-fsyms*)
	   (incf (fsym-lex-weight (to-fsym i)))))

;; Insert a function symbol symbol at the back of the lex ordering
(proclaim '(function new-special-last-fsym (t) nil))
(defun new-special-last-fsym (f)
  (setf *special-last-fsym* f)
  (setf (fsym-lex-weight f) (1- *num-fsyms*)))

		    





