;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: heap.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Heap management for various data types.
;;; In HIPER, we try to avoid garbage collection as much as possible.
;;; This is accomplished by keeping freelists for each data type, 
;;; and by allocating from and deallocating to the freelists
;;; whenever possible.  In particular, as soon as we know that 
;;; a data structure is no longer needed (even if its just a cons cell),
;;; we return it to the heap manager.
;;; (See also "misc.lsp" for alternatives to some Lisp builtins like
;;; cons and push and pop).


;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Flat terms
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *ft-freelist* nil)

(defmacro new-ft ()
  `(if *ft-freelist*
       (let ((ft *ft-freelist*))
	 (setf *ft-freelist* (ft-next *ft-freelist*))
	 ft)
     (make-ft)))

(defmacro get-ft (sym)
  (let ((ft (gentemp "FT")))
    `(let ((,ft (new-ft)))
       (setf (ft-symbol ,ft) ,sym)
       (setf (ft-next ,ft) nil)
       (setf (ft-prev ,ft) nil)
       (setf (ft-end ,ft) ,ft)
       ,ft)))

(defmacro free-ft (ft)
  `(progn
     (setf (ft-next ,ft) *ft-freelist*)
     (setf *ft-freelist* ,ft)))

(defmacro kill-ft (ft)
  `(do-ft ,ft :safely (free-ft (ft))))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Cons cells
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *cons-freelist* nil)

(defmacro new-cons (x y)
  (let ((c (gensym)))
    `(if *cons-freelist*
	 (let ((,c *cons-freelist*))
	   (setf *cons-freelist* (cdr *cons-freelist*))
	   (setf (car ,c) ,x)
	   (setf (cdr ,c) ,y)
	   ,c)
       (cons ,x ,y))))
  
(defmacro free-cons (cell)
  (let ((c (gensym)))
    `(let ((,c ,cell))
       (rplacd ,c *cons-freelist*)
       (rplaca ,c '*free*)
       (setf *cons-freelist* ,c))))

(proclaim '(function free-conses (t) nil))
(defun free-conses (term)
  (when (and (consp term) (not (eq (car term) '*free*)))
	(let ((car (car term))
	      (cdr (cdr term)))
	  (free-cons term)
	  (free-conses car)
	  (free-conses cdr))))

(defmacro free-list (l)
  (let ((tmp (gentemp "LIST")))
    `(let ((,tmp ,l))
       (while ,tmp (pop1 ,tmp)))))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Equations
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *eqn-freelist* nil)

(proclaim '(function new-eqn () t))
(defun new-eqn ()
  (if *eqn-freelist*
       (let ((e *eqn-freelist*))
	 (setf *eqn-freelist* (eqn-next *eqn-freelist*))
	 (setf (eqn-type e) nil)
	 e)
     (make-eqn)))

(proclaim '(function free-eqn (t) nil))
(defun free-eqn (e)
  (setf (eqn-lhs e) nil)
  (setf (eqn-rhs e) nil)
  (setf (eqn-prev e) nil)
  (setf (eqn-fsym-def-p e) nil)
  (setf (eqn-parents e) nil)
  (setf (eqn-backpointers e) nil)
  (setf (eqn-next e) *eqn-freelist*)
  (setf (eqn-queue e) nil)
  (setf *eqn-freelist* e))

(proclaim '(function kill-eqn (t) nil))
(defun kill-eqn (e)
  (kill-ft (eqn-lhs e))
  (kill-ft (eqn-rhs e))
  (free-conses (eqn-parents e))
  (setf (eqn-backpointers e) nil)
  (free-eqn e))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Slotarrays
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *slotarray-freelist* nil)
(defvar *slotarray-size* 0)

;; Note that slot-vals of re-used slots are 
;; not initialized; they should have been
;; cleaned up (set to nil) before being returned.
(defun new-slot-array ()
  (if *slotarray-freelist*
      (pop1 *slotarray-freelist*)
    (let ((s (make-array *slotarray-size*)))
      (floop :for i :below *slotarray-size*
	     :do (setf (aref (the (array t) s) i) (new-slot)))
      s)))

(defmacro free-slot-array (slotarray)
  `(push1 ,slotarray *slotarray-freelist*))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Nodes
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *node-freelist* nil)

(proclaim '(function new-node () t))
(defun new-node ()
  (if *node-freelist*
      (let ((n *node-freelist*))
	(setf *node-freelist* (node-slots n))
	(setf (node-slots n) (new-slot-array))
	n)
    (make-node :slots (new-slot-array))))

(proclaim '(function free-node (t) nil))
(defun free-node (n)
  (setf (node-prev-node n) nil)
  (setf (node-prev-slot n) nil)
  (setf (node-slots n) *node-freelist*)
  (setf *node-freelist* n))


;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Stacks (arrays)
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *stack-freelist* nil)
(eval-when (load compile eval)
	   (setf *stack-freelist* (make-array 512 :fill-pointer 0)))

(defmacro new-stack ()
  `(if (plusp (the fixnum (fill-pointer (the (array t) *stack-freelist*))))
       (vector-pop (the (array t) *stack-freelist*))
     #+kcl(make-array *stack-size* :fill-pointer 0 :static t)
     #-kcl(make-array *stack-size* :fill-pointer 0)
     ))

(defmacro free-stack (stack)
  `(vector-push ,stack (the (array t) *stack-freelist*)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Trail stacks
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *trail-freelist* nil)
(defvar *next-trail* nil)
(eval-when (load compile eval)
	   (setf *trail-freelist* (make-array 512))
	   ;; A trick to allocate a global fixnum, without further
           ;; updates calling CMPmake_fixnum
	   (setf *next-trail* (make-array 1 :element-type 'fixnum))
	   (setf (aref *next-trail* 0) 0))

(defmacro new-trail ()
  `(let* ((n *next-trail*)(i (the fixnum (1- (aref n 0)))))
     (declare (object n) (type (array fixnum) n))
     (declare (type fixnum i))
     (if (>= i 0)
	 (progn (setf (aref n 0) i)
		(aref (the (array t) *trail-freelist*) i))
       #+kcl(make-array *trail-size* :element-type 'fixnum :static t)
       #-kcl(make-array *trail-size* :element-type 'fixnum)
     )))
  
(defmacro free-trail (trail)
  (let ((n (gentemp "N")) (i (gentemp "I")))
  `(let* ((,n *next-trail*)
	  (,i (aref ,n 0)))
     (declare (object ,n) (type (array fixnum) ,n))
     (declare (type fixnum ,i))
     (setf (aref (the (array t) *trail-freelist*) ,i) ,trail)
     (setf (aref ,n 0) (the fixnum (1+ ,i))))))

#|
(defmacro new-trail ()
  `(let* ((f *trail-freelist*))
     (declare (object f) (type (vector t) f))
     (if (plusp (the fixnum (fill-pointer f)))
	 (vector-pop f)
       #+kcl(make-array *trail-size* :element-type 'fixnum :static t)
       #-kcl(make-array *trail-size* :element-type 'fixnum)
     )))
  
(defmacro free-trail (trail)
  `(vector-push ,trail (the (vector t) *trail-freelist*)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Leaves
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *leaf-freelist* nil)

(proclaim '(function new-leaf () t))
(defun new-leaf ()
  (if *leaf-freelist*
      (let ((l *leaf-freelist*))
	(setf *leaf-freelist* (leaf-subterm l))
	(setf (leaf-subterm l) nil)
	l)
    (make-leaf)))

(proclaim '(function free-leaf (t) nil))
(defun free-leaf (l)
  (setf (leaf-subterm l) *leaf-freelist*)
  (setf *leaf-freelist* l)
  (setf (leaf-parent-node l) nil)
  (setf (leaf-parent-slot l) nil)
  (setf (leaf-eqn l) nil)
  )



;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Queues
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *queue-freelist* nil)

(proclaim '(function new-queue () t))
(defun new-queue ()
  (if *queue-freelist*
      (pop1 *queue-freelist*)
    (make-queue)))

(proclaim '(function free-queue (t) nil))
(defun free-queue (q)
  (kill-eqn (queue-head q))
  (kill-eqn (queue-tail q))
  (setf (queue-head q) nil)
  (setf (queue-tail q) nil)
  (setf (queue-pfunc1 q) nil)
  (setf (queue-pfunc2 q) nil)
  (push1 q *queue-freelist*))


 
;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ustates (e-unification states)
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *ustate-freelist* nil)

(defmacro new-ustate ()
  `(if *ustate-freelist*
       (let ((u *ustate-freelist*))
	 (setf *ustate-freelist* (ustate-trail u))
	 u)
     (let ((u (make-ustate)))
       (setf (ustate-stack u) (new-stack))
       (setf (ustate-trail u) (new-stack))
       u)))

(defmacro free-ustate (u)
  (let ((us (gentemp "USTATE")))
    `(let ((,us ,u))
       (setf (ustate-trail ,us) *ustate-freelist*)
       (setf *ustate-freelist* ,us))))


;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Uchoices (e-unification choicepoints)
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *uchoice-freelist* nil)

(defmacro new-uchoice ()
  `(if *uchoice-freelist*
       (let ((c *uchoice-freelist*))
	 (setf *uchoice-freelist* (uchoice-t1 c))
	 c)
     (make-uchoice)))

(defmacro free-uchoice (c)
  (let ((uc (gentemp "UCHOICE")))
    `(let ((,uc ,c))
       (setf (uchoice-t1 ,uc) *uchoice-freelist*)
       (setf *uchoice-freelist* ,uc))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Permutation arrays
;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *permutation-freelist* nil)
(defconstant *max-permuter-vars* 8)

(defmacro new-permutation ()
  `(if *permutation-freelist*
       (let ((p *permutation-freelist*))
	 (setf *permutation-freelist* (aref (the (array t) p) 0))
	 p)
     #+kcl (make-array #.(1+ *max-permuter-vars*) :static t)
     #-kcl (make-array #.(1+ *max-permuter-vars*))
     ))

(defmacro free-permutation (p)
  (let ((perm (gentemp "PERM")))
    `(let ((,perm ,p))
       (setf (aref (the (array t) ,perm) 0) *permutation-freelist*)
       (setf *permutation-freelist* ,perm))))








