;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: globals.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Global constants and variables. 

;; Maximum number of function symbols
(defconstant *max-fsyms* 256)

;; In HIPER, there are two global banks of variables: the X vars and
;; the Y vars.  The banks are implemented as a single array, each
;; element of which is a variable binding.
(defconstant *max-vars* 256)		;Number of variables per bank
(defconstant *max-x-vars* *max-vars*)	;1 + index of last x var
(defconstant *max-y-vars* (* 2 *max-vars*)) ;1 + index of last y var
(defconstant *max-all-vars* (* 2 *max-vars*)) ;Total number of vars
(defvar *vars*				;The actual variables
  #+kcl (make-array (list *max-all-vars*) :static t)
  #-kcl (make-array (list *max-all-vars*))
  )
(proclaim '(type (array t) *vars*))

;; Make all variables unbound.  This is sometimes useful if
;; you have interrupted HIPER in the middle of a run, and
;; wish to invoke some lower-level routine like unify-terms.
(defun init-vars ()			
  (sloop for i below #.*max-all-vars*
	 declare (fixnum i)
	 do (setf (aref (the (array t) *vars*) i) nil)))

;; Size of stack arrays and trail arrays.
(defconstant *stack-size* 512)
(defconstant *trail-size* 256)


;; We keep a number of counters so that we can keep track of 
;; certain statistics.  The counters are implemented as single-element
;; fixnum arrays, because AKCL won't allocate a global raw fixnum.
;; Hence, incrementing a global fixnum in AKCL could allocate
;; a new fixnum in the Lisp heap, and cause unnecessary garbage
;; collections.  With the fixnum array trick, the fixnum element
;; is directly incremented using a C increment.
(defvar *eqn-counter*)			;Number of equations retained
(eval-when (load compile eval)
	   (setf *eqn-counter* (make-array 1 :element-type 'fixnum)))
(defmacro incf-eqn-counter ()
  `(setf (aref (the (array fixnum) *eqn-counter*) 0)
	 (the fixnum (1+ (aref (the (array fixnum) *eqn-counter*) 0)))))
(defmacro eqn-counter () `(aref *eqn-counter* 0))

(defvar *pair-counter*)			;Number of pairs generated
(eval-when (load compile eval)
	   (setf *pair-counter* (make-array 1 :element-type 'fixnum)))
(defmacro incf-pair-counter ()
  `(setf (aref (the (array fixnum) *pair-counter*) 0)
	(the fixnum (1+ (aref (the (array fixnum) *pair-counter*) 0)))))
(defmacro pair-counter () `(aref *pair-counter* 0))

;; Current number of function symbols
(defvar *num-function-syms* 0)
(proclaim '(type fixnum *num-function-syms*))

;; Size of index arrays for queues
(defconstant *queue-index-size* 256)

;; The precedence relation for the recursive-path ordering functions
(defvar *precedence* '(nil))

;; When non-nil, HIPER will pause whenever it retains an equation.
(defvar *pause-for-user* nil)

;; The queue of equations waiting to be reduced and oriented.
(defvar *pairs*)

;; Rules waiting to be used to generate critical pairs.
(defvar *rules*)

;; Rules which have been used to generate critical pairs.
(defvar *marked-rules*)

;; The discrimination net index of left sides (top level only) of
;; rules.  During reduction, we look in this net to find a rule which can
;; reduce a given subterm.
(defvar *lhs-net*)

;; Index of all other subterms in the system.
(defvar *subterm-net*)

;; Index of the left and right top-level terms of deferred failures.
(defvar *failures-net*)

;; Index of the top-level left sides of pseudo-rules
(defvar *pseudo-rules-net*)

;; Queue containing duplicates of the original input equations.
;; When completion is restarted (as after unification is extended),
;; the original equations are re-introduced to the system.
(defvar *restart-q*)

;; When non-nil, signifies that E-completion is in effect.
;; When nil, only ordinary unification, matching, etc. are being used.
(defvar *equations* nil)

;; Start and end times for the last run.
(defvar *start-time* 0)
(defvar *end-time* 0)

;; When non-nil, allows permuters to be incorporated into 
;; unification, matching, equality, and termination.  Otherwise,
;; E-completion is disabled.
(defvar *enable-permuters* t)

;; Non-nil if pseudo-rules have been introduced.
(defvar *pseudo-rules* nil)

;; Prohibits pseudo-rules when non-nil.
(defvar *disable-pseudo-rules* nil)

;; Counter for the number of times the main loop has executed.  This
;; represents the number of equations that the system has processed.
(defvar *loop-counter*)
(eval-when (load compile eval)
	   (setf *loop-counter* (make-array 1 :element-type 'fixnum)))
(defmacro incf-loop-counter ()
  `(setf (aref (the (array fixnum) *loop-counter*) 0)
	(the fixnum (1+ (aref (the (array fixnum) *loop-counter*) 0)))))
(defmacro loop-counter () `(aref *loop-counter* 0))

;; Current number of function symbols
(defvar *num-fsyms* 0)

;; Directory and file name extension of input files.
(defvar *example-directory* "examples/")
(defvar *example-extension* ".comp")

;; Non-nil if set-of-support is in use.
(defvar *support-set* nil)

;; If nil, prohibits set-of-support.  Ordinarily, there is
;; no need to reset this flag.
(defvar *set-of-support* t)

;; When non-nil, turns off most printing of intermediate results.
;; Useful for timing HIPER.
(defvar *benchmark* nil)

;; When non-nil, prohibits the introduction of new function symbols.
(defvar *disable-new-fsyms* nil)

;; When non-nil, disables restarting after a reducible function
;; symbol definition is discovered.  Be careful with this; I haven't
;; tested it thouroughly.
(defvar *disable-fsym-restart* t)

;; When non-nil, overrides HIPER's heuristics for deciding when
;; to introduce a new function symbol, and asks the user to decide.
(defvar *user-decide-new-fsyms* nil)

;; Relax HIPER's heuristics for introducing new function symbols.
;; Sometimes this results in simple permuters being discovered 
;; more readily.
(defvar *encourage-simple-permuters* nil)

;; When nil, allows critical pairs with deferred failures.  
;; This usually is undesirable.
(defvar *passive-failures* t)

;; Current parents of critical pairs.
(defvar *p1* nil)
(defvar *p2* nil)

;; Highest-numbered function symbol in the input equations.
(defvar *last-original-fsym* 0)

;; Counter for generating new function symbol names.
(defvar *new-fsym-counter* 0)

;; When non-nil, the user is asked to orient failure equations.
(defvar *user-orient-failures* nil)

;; Determines which ordering to use.
;;    'kb = Knuth-Bendix
;;    'rpo = Automated Lexicographic Recursive Path
;;    'kb-rpo = Combined KB/RPO
(defvar *which-order* 'kb-rpo)

;; When non-nil, the user is asked form permission whenever
;; the automatic RPO decides to give lex status to a symbol.
;; Sometimes useful when trying to force the introduction of
;; permutative equations.
(defvar *user-verify-status* t)

;; Weight of a variable, for Knuth-Bendix ordering
(defvar *w0* 1)

;; When using the Knuth-Bendix ordering, it is often necessary to give
;; 0 weight to a unary function symbol (usually the inverse).  Such a 
;; symbol should be the last symbol declared.  This variable
;; is set to the id of such a symbol when there is one.
(defvar *special-last-fsym* nil)

