;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: flatterms.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Macros for accessing, testing, and traversing flatterms.

;; Note that this first group of macros operate on the ft-symbol of 
;; a flatterm, NOT on the flatterm itself.
;; E.g. use (same-symbol? (ft-symbol ft1) (ft-symbol ft2)), NOT
;; (same-symbol? ft1 ft2).  This is sometimes tedious, but 
;; better code can usually be generated with it.

;; Compare two symbols for equality
(defmacro same-symbol? (s1 s2) `(= (the fixnum ,s1) (the fixnum ,s2)))

;; Does sym represent a variable?
(defmacro var? (sym) `(< (the fixnum ,sym) *max-all-vars*))

;; Retrieve the binding of sym (which should already have been
;; checked with (var? sym)
(defmacro var-binding (sym)
  `(aref (the (array t) *vars*) (the fixnum ,sym)))

;; Is sym an unbound var? (sym should have already passed (var? sym))
(defmacro unbound-var? (sym) `(null (var-binding ,sym)))

;; Is sym a bound var? (sym should have already passed (var? sym))
(defmacro bound-var? (sym) `(var-binding ,sym))

;; Set the binding of the variable represented by sym to b.
;; Sym should be a known variable.
(defmacro set-binding (sym b) `(setf (var-binding ,sym) ,b))

;; Unbind a variable.
(defmacro make-unbound-var (sym) `(setf (var-binding ,sym) nil))

;; Is sym a function symbol?
(defmacro fsym? (sym) `(>= (the fixnum ,sym) *max-all-vars*))

;; Arity of sym
(defmacro struct-arity (sym) `(the fixnum (fsym-arity ,sym)))




;; Some useful building blocks for copying and constructing flatterms

;; Append sequence starting with ft1 with ft2
(defmacro ft-append (ft1 ft2)
  (assert (symbolp ft1))
  (let ((t2 (gentemp "FT")))
    `(let ((,t2 ,ft2))
       (if (null ,ft1)
	   (setf ,ft1 ,t2)
	 (progn
	   (setf (ft-next (ft-end ,ft1)) ,t2)
	   (setf (ft-prev ,t2) (ft-end ,ft1))
	   (setf (ft-end ,ft1) (ft-end ,t2)))))))

;; Prepend sequence beginning with ft1 with symbol ft2
(defmacro ft-prepend (ft1 ft2)
  (assert (symbolp ft1))
  (let ((t2 (gentemp "FT")))
    `(let ((,t2 ,ft2))
       (if (null ,ft1)
	   (setf ,ft1 ,t2)
	 (progn
	   (setf (ft-prev ,ft1) (ft-end ,t2))
	   (setf (ft-next (ft-end ,t2)) ,ft1)
	   (unless (var? (ft-symbol ,t2))
		   (floop :for i :below (fsym-arity (ft-symbol ,t2))
			  :do (setf (ft-end ,t2)
				    (ft-end (ft-next (ft-end ,t2))))))
	   (setf ,ft1 ,t2))))))
  
;; Prepend sequence beginning with ft1 with sequence ft2
(defmacro ft-concat (ft1 ft2)
  (assert (symbolp ft1))
  (let ((t2 (gentemp "FT")))
    `(let ((,t2 ,ft2))
       (if (null ,ft1)
	   (setf ,ft1 ,t2)
	 (progn
	   (setf (ft-prev ,ft1) (ft-end ,t2))
	   (setf (ft-next (ft-end ,t2)) ,ft1)
	   (setf ,ft1 ,t2))))))




;; Traversing flatterms from left to right. This macro provides a 
;; convenient way to hide many of the tedious, common details for
;; walking a term, yet it takes care to generate very efficient code.
;; For instance, to count the number of symbols in a term, you might
;; write something like this:
;; (defun sym-count (x &aux (i 0)) (do-ft x :return i (incf i)))
;;
;; From within a do-ft invocation, the following local
;; macros are available:
;;   (abort) -- Immediately return nil.
;;   (ft) -- The current flatterm.
;;   (ft-sym) -- The current symbol of the current flatterm
;;   (skip-to x) -- In the next loop, jump to flatterm x.
;;
;; Unless abort is called, do-ft returns t.
;;
;; When the keyword :skip-first occurs at the top-level of
;; code, then the first symbol of ft is skipped.
;;
;; If the keyword :return is the first item in code,
;; then the next item will be returned after a successful 
;; traversal.
;;
;; The keyword :safely will allow walking a term even when
;; it is being destructively modified, as when walking over
;; a flatterm to deallocate its components.
;;
;; There are many examples of usage scattered throughout HIPER.
;;
(defmacro do-ft (ft &body code &aux use-next use-sym)
  (when (or (member :safely code)
	    (sloop for x in-fringe code thereis (eq x 'skip-to)))
	(setf use-next t))
  (setf use-sym (sloop for x in-fringe code thereis (eq x 'ft-sym)))
  (let ((last (gentemp "LAST"))
	(blockname (gentemp "BLOCK"))
	(lft (gentemp "FT"))
	(sym (gentemp "SYM"))
	(next (gentemp "NEXT")))
    `(block ,blockname
       (let*
	   ,(append `((,lft ,ft) (,last (ft-next (ft-end ,lft))))
		    (when use-sym `((,sym 0)))
		    (when use-next `(,next)))
	 ,(when use-sym `(declare (type fixnum ,sym)))
	 (macrolet ((abort () `(return-from ,',blockname nil))
		    (ft () (quote ,lft))
		    (ft-sym () (quote ,sym))
		    (skip-to (x) `(setf ,',next ,x)))
		   ,(when (member :skip-first code)
			  `(setf ,lft (ft-next ,lft)))
		   (until (eq ,lft ,last)
			  ,@(if (eq (car code) :return)
				(prog1 `(:return ,(second code))
				  (setf code (nthcdr 2 code)))
			      `(:return t))
			  ,(when use-next `(setf ,next (ft-next ,lft)))
			  ,(when use-sym
				 `(setf ,sym (the fixnum (ft-symbol ,lft))))
			  ,@(append code
				    (if use-next
					`((setf ,lft ,next))
				      `((setf ,lft (ft-next ,lft)))))
			  ))))))


;; Like do-ft, but traverses from right to left.
(defmacro do-ft-backwards (ft &body code &aux use-prev use-sym)
  (when (or (member :safely code)
	    (sloop for x in-fringe code thereis (eq x 'skip-to)))
	(setf use-prev t))
  (setf use-sym (sloop for x in-fringe code thereis (eq x 'ft-sym)))
  (let ((first (gentemp "FIRST"))
	(blockname (gentemp "BLOCK"))
	(lft (gentemp "FT"))
	(sym (gentemp "SYM"))
	(prev (gentemp "PREV")))
    `(block ,blockname
	(let* ,(append `((,first ,ft)(,lft (ft-end ,first)))
		      (when use-sym `((,sym 0)))
		      (when use-prev `(,prev)))
	 ,(when use-sym `(declare (type fixnum ,sym)))
	 (macrolet ((abort () `(return-from ,',blockname nil))
		    (ft () (quote ,lft))
		    (ft-sym () (quote ,sym))
		    (skip-to (x) `(setf ,',prev ,x)))
		   (setf ,first (ft-prev ,first))
		   (until (eq ,lft ,first)
			  ,@(if (eq (car code) :return)
				(prog1 `(:return ,(second code))
				  (setf code (nthcdr 2 code)))
			      `(:return t))
			  ,(when use-prev `(setf ,prev (ft-prev ,lft)))
			  ,(when use-sym
				 `(setf ,sym (the fixnum (ft-symbol ,lft))))
			  ,@(append code
				    (if use-prev
					`((setf ,lft ,prev))
				      `((setf ,lft (ft-prev ,lft)))))
			  ))))))

;; Traverse the immediate args of a flatterm.  Since 
;; flatterms are "strung-out", immediate subterm arguments of 
;; a function symbol aren't necessarily consecutive
;; in the flatterm.
;;
;; E.g. Suppose we have f(g(x),f(x,y)) represented as f-g-x-f-x-y;
;; then it has immediate args g-x and f-x-y.
;;
(defmacro do-args (ft &body code)
  (let ((last (gentemp "LAST"))
	(blockname (gentemp "BLOCK"))
	(lft (gentemp "FT"))
	(sym (gentemp "SYM"))
	(next (gentemp "NEXT")))
    `(block ,blockname
	(let* (,next
	       (,lft ,ft)
	       (,last (ft-next (ft-end ,lft)))
	       (,sym  0))
	  (declare (type fixnum ,sym))
	  (macrolet ((abort () `(return-from ,',blockname nil))
		     (ft () (quote ,lft))
		     (ft-sym () (quote ,sym)))
		    (setf ,lft (ft-next ,lft))
		    (until (eq ,lft ,last)
			   :return t
			   (setf ,next (ft-next (ft-end ,lft)))
			   (setf ,sym (the fixnum (ft-symbol ,lft)))
			   ,@(append code `((setf ,lft ,next)))
			   ))))))

;; Return t if pred is true of all immediate args 
(defmacro all-args (ft &body pred)
  `(do-args ,ft (unless (progn ,@pred) (abort))))

;; Return t if pred is true of some immediate arg
(defmacro some-arg (ft &body pred)
  `(not (do-args ,ft (when (progn ,@pred) (abort)))))

;; Walk the args of two flatterms simultaneously.
(defmacro do-args2 (ft1 ft2 &body code)
  (let ((last (gentemp "LAST"))
	(blockname (gentemp "BLOCK"))
	(lft1 (gentemp "FT"))
	(lft2 (gentemp "FT"))
	(sym1 (gentemp "SYM"))
	(sym2 (gentemp "SYM"))
	(next1 (gentemp "NEXT"))
	(next2 (gentemp "NEXT")))
    `(block ,blockname
	(let* ((,lft1 ,ft1)
	       (,lft2 ,ft2)
	       (,last (ft-next (ft-end ,lft1)))
	       (,sym1 0) (,sym2 0) ,next1 ,next2)
	  (declare (type fixnum ,sym1 ,sym2))
	  (macrolet ((abort () `(return-from ,',blockname nil))
		     (ft1 () (quote ,lft1))
		     (ft2 () (quote ,lft2))
		     (ft-sym1 () (quote ,sym1))
		     (ft-sym2 () (quote ,sym2)))
		    (setf ,lft1 (ft-next ,lft1))
		    (setf ,lft2 (ft-next ,lft2))
		    (until (eq ,lft1 ,last)
			   :return t
			   (setf ,next1 (ft-next (ft-end ,lft1)))
			   (setf ,next2 (ft-next (ft-end ,lft2)))
			   (setf ,sym1 (the fixnum (ft-symbol ,lft1)))
			   (setf ,sym2 (the fixnum (ft-symbol ,lft2)))
			   ,@(append code
				     `((setf ,lft1 ,next1)
				       (setf ,lft2 ,next2)))
			   ))))))


;; Rename the vars of a flatterm, by adding a constant offset to
;; each one.  
(defmacro rename-vars (ft offset)
  `(do-ft ,ft (when (var? (ft-sym)) (incf (ft-symbol (ft)) ,offset))))

;; Rename the vars of an equation
(proclaim '(function rename-eqn-vars (t fixnum) t))
(defun rename-eqn-vars (eqn offset)
  (declare (type fixnum offset))
  (rename-vars (eqn-lhs eqn) offset)
  (rename-vars (eqn-rhs eqn) offset))

;; Rename x vars to y vars, and vice-versa, for a term
(defmacro xs-to-ys (ft)
  `(rename-vars ,ft *max-x-vars*))
(defmacro ys-to-xs (ft)
  `(rename-vars ,ft (- *max-x-vars*)))

;; Rename x vars to y vars, and vice-versa, for an equation
(defmacro eqn-xs-to-ys (eqn)
  `(rename-eqn-vars ,eqn *max-x-vars*))
(defmacro eqn-ys-to-xs (eqn)
  `(rename-eqn-vars ,eqn (- *max-x-vars*)))

;; Count the number of symbols in a term.
(proclaim '(function sym-count (t) fixnum))
(defun sym-count (t1 &aux (i 0))
  (declare (type fixnum i))
  (do-ft t1 :return i
	 (incf i)))

