;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: eunify.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; E-unification for simple linear permutative theories.

;; Top-level.  Dispatch to ordinary unification if possible.
(defmacro e-unify (t1 t2)
  `(if (not *equations*)
       (unify-terms ,t1 ,t2)
     (e-unify-terms ,t1 ,t2 t)))

;; Find another unifier if there is one.
(defmacro re-unify (t1 t2)
  `(when *equations* (e-unify-terms ,t1 ,t2 nil)))

;; Some state must be saved in between calls to e-unify-terms when
;; multiple unifiers are desired.
(defvar *eunify-tr-top* nil)		;Top of trail
(defvar *eunify-tr-frame* nil)		;Top frame of trail
(defvar *eunify-uc* nil)		;Unification choicepoint marker
(eval-when (load compile eval)
	   (setf *eunify-tr-top* (make-array 1 :element-type 'fixnum))
	   (setf *eunify-tr-frame* (make-array 1 :element-type 'fixnum))
	   (setf *eunify-uc* (make-array 1 :element-type 'fixnum)))

(defmacro eunify-tr-top ()
  `(the fixnum (aref (the (array fixnum) *eunify-tr-top*) 0)))
(defmacro eunify-tr-frame ()
  `(the fixnum (aref (the (array fixnum) *eunify-tr-frame*) 0)))
(defmacro eunify-uc ()
  `(the fixnum (aref (the (array fixnum) *eunify-uc*) 0)))


;; Using a backtracking algorithm, permute t1 until it unifies with
;; t2.  If reset is t, then assume that the first unifier is 
;; being sought.  If reset is nil, then find the next unifier for t1 and t2.
;; Basic idea: Walk down t1 and t2 simultaneously.  Whenever a bound
;; variable is encountered, push its binding onto the term stack, along with
;; the corresponding subterm of the opposite term, and continue.
;; Stacked terms are popped and unified after the current t1 and t2
;; have been traversed.  Whenever a permutative
;; function symbol is encountered in t1, push a choice point on the
;; choice stack, saving the information necessary to restart from this
;; point if needed.  On backtracking, restore t1 and t2 and the stack
;; from the top choice point, permute t1, and retry the unification from
;; that point.  When all permutations have been tried for a given
;; choicepoint, pop the choicepoint.
(proclaim '(function e-unify-terms (t t t) t))
(with-stacks ((*stack* :stack)(*choicestack* :stack)(*trail* :trail))
(defun e-unify-terms (t1 t2 reset
			 &aux (stk-top 0)
			 (tr-top  (eunify-tr-top))
			 (tr-frame (eunify-tr-frame))
			 (UC (eunify-uc))
			 (trail (get-fixnum-array *trail*))
			 (stack (get-c-array *stack*))
			 (chstack (get-c-array *choicestack*))
			 muts (sym1 0) (sym2 0) last)
  (declare (type fixnum sym1 sym2 stk-top tr-top tr-frame UC))
  (declare (type (array t) stack chstack))
  (declare (type (array fixnum) trail))

  (macrolet
   (
    ;; Choicepoint structure
    ;; UC marks the top of the choice stack

    ;; Subterm to permute
    (choice-t1 () `(c-aref chstack UC))
    ;; Subterm to unify against
    (choice-t2 () `(c-aref chstack (the fixnum (+ UC 1))))
    ;; End marker (not necessarily of t1)
    (choice-last () `(c-aref chstack (the fixnum (+ UC 2))))
    ;; Copy of stack when choicepoint was pushed
    (choice-stack-state () `(c-aref chstack (the fixnum (+ UC 3))))
    ;; Permutations applicable to t1
    (choice-muts () `(c-aref chstack (the fixnum (+ UC 4))))
    ;; Original args of t1; used to do permutations
    (choice-base () `(c-aref chstack (the fixnum (+ UC 5))))
    (set-choice-t1 (x) `(c-aset chstack UC ,x))
    (set-choice-t2 (x) `(c-aset chstack (the fixnum (+ UC 1)) ,x))
    (set-choice-last (x) `(c-aset chstack (the fixnum (+ UC 2)) ,x))
    (set-choice-stack-state (x) `(c-aset chstack (the fixnum (+ UC 3)) ,x))
    (set-choice-muts (x) `(c-aset chstack (the fixnum (+ UC 4)) ,x))
    (set-choice-base (x) `(c-aset chstack (the fixnum (+ UC 5)) ,x))
    (next-choice () `(incf UC 6))
    (prev-choice () `(decf UC 6))
    (choice-stack-empty? () `(zerop UC))
    )
  
  (prog
   ()

   ;; If finding the first unifier, initialize state.
   ;; Otherwise, the saved state will have already been loaded into
   ;; the &aux variables.
   (if reset
       (progn
	 (setf stk-top 0)
	 (setf tr-top 0)
	 (setf tr-frame 0)
	 (setf UC 0))
     (go BACKTRACK))

   ;; Push t1 and t2 onto the stack
   (c-stack-push t2 stack stk-top)
   (c-stack-push t1 stack stk-top)

   TOP
   (when (zerop stk-top) (go SUCCEED))	;Succeed when stack empty
   (when (> stk-top *stack-size*)
	 (error "stack overflow in restr-e-unify"))
   (setf t1 (c-stack-pop stack stk-top)) ;Pop t1
   (setf t2 (c-stack-pop stack stk-top)) ;Pop t2
   (setf last (ft-next (ft-end t1)))	;Remember end of t1
   LOOP
   (when (eq t1 last) (go TOP))		;done?
   (setf sym1 (ft-symbol t1))		;Cache the ft-symbol fields
   (setf sym2 (ft-symbol t2))
   
   ;; Build new choicepoint if appropriate
   (when (and (not (var? sym1)) (same-symbol? sym1 sym2)
	      (setf muts (fsym-mutations sym1)))
	 (set-choice-t1 t1)
	 (set-choice-t2 t2)
	 (set-choice-last last)
	 (set-choice-stack-state nil)
	 (set-choice-muts muts)
	 (let ((b (new-permutation)))
	   (set-choice-base b)
	   (init-permute t1 muts b))
	 (floop :for i :below stk-top	;Copy current term stack, top to bottom
		:do 
		  (set-choice-stack-state
		   (cons1 (c-aref stack
				  (the fixnum (- stk-top (the fixnum (+ i 1)))))
			  (choice-stack-state))))
	 (next-choice)
	 (push-trail-frame trail tr-frame tr-top)
	 )
  
   ;; Unify
   (cond
    ((var? sym1)
     (cond ((bound-var? sym1)
	    (c-stack-push t2 stack stk-top)
	    (c-stack-push (var-binding sym1) stack stk-top)
	    (setf t2 (ft-end t2)))
	   ((var? sym2)
	    (cond ((bound-var? sym2)
		   (c-stack-push (var-binding sym2) stack stk-top)
		   (c-stack-push t1 stack stk-top))
		  ((same-symbol? sym1 sym2) nil)
		  (t (trail-var sym1 trail tr-top)
		     (set-binding sym1 t2)
		     (setf t2 (ft-end t2)))))
	   ((occurs? (the fixnum sym1) t2)
	    (go BACKTRACK))
	   (t
	    (trail-var sym1 trail tr-top)
	    (set-binding sym1 t2)
	    (setf t2 (ft-end t2)))))
    ((var? sym2)
     (cond ((bound-var? sym2)
	    (c-stack-push (var-binding sym2) stack stk-top)
	    (c-stack-push t1 stack stk-top)
	    (setf t1 (ft-end t1)))
	   ((occurs? (the fixnum sym2) t1)
	    (go BACKTRACK))
	   (t
	    (trail-var sym2 trail tr-top)
	    (set-binding sym2 t1)
	    (setf t1 (ft-end t1)))))
    ((not (same-symbol? sym1 sym2))
     (go BACKTRACK)))
   
   ;; Move to the next symbol in each term
   (setf t1 (ft-next t1))
   (setf t2 (ft-next t2))
   (go LOOP)

   ;; Save locals and succeed
   SUCCEED
   (setf (eunify-tr-top) tr-top)
   (setf (eunify-tr-frame) tr-frame)
   (setf (eunify-uc) UC)
   (return-from e-unify-terms t)
   
   BACKTRACK
   (restore-trail-frame trail tr-frame tr-top)
   ;; Fail if choice stack empty
   (when (choice-stack-empty?)
	 (return-from e-unify-terms nil))
   ;; Restore state
   (prev-choice)
   (setf t1 (choice-t1))
   (setf t2 (choice-t2))
   (setf last (choice-last))
   ;; Permute
   (unless (setf muts (choice-muts))
	   (quit-permute t1 (choice-base))
	   (free-permutation (choice-base))
	   (free-list (choice-stack-state))
	   (pop-trail-frame trail tr-frame tr-top)
	   (go BACKTRACK))
   (permute t1 (car muts) (choice-base))
   (set-choice-muts (cdr muts))
   (setf stk-top 0)
   (dolist (x (choice-stack-state))	;Restore term stack
	   (c-stack-push x stack stk-top))
   (next-choice)			;Save choicepoint
   (setf t1 (ft-next t1)) 
   (setf t2 (ft-next t2))
   (go LOOP)
   ))))


  
(defun test-e-unify (x y &aux t1 t2 us)
  (setf t1 (parse x))
  (setf t2 (parse-term y))
  (setf us (e-unify t1 t2))
  (terpri)
  (while us
    (terpri)(princ "==> ")(print-term-after-unify t1)
            (princ " & ") (print-term-after-unify t2)
    (setf us (re-unify t1 t2))))
	


