;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: enodes.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Permutative discrimination net routines.  
;;; E-matching is interleaved with the lookup operations.  As a path
;;; is traversed, permutations are remembered in choicepoints.  On
;;; backtracking, the key term is permuted and the path re-tried.

;; Top-level hooks, which dispatch to more efficient, orinary routines
;; if appropriate.
(defmacro find-lhs (redex net mode)
  `(if *equations*
       (e-net-find-lhs ,redex ,net ,mode)
     (net-find-lhs ,redex ,net ,mode)))

(defmacro find-matches (eqn net mode)
  `(if *equations*
       (e-net-find-matches ,eqn ,net ,mode)
     (net-find-matches ,eqn ,net ,mode)))


;; Like net-find-lhs
(proclaim '(function e-net-find-lhs (t t fixnum) t))
(with-stacks ((*stack* :stack))
(defun e-net-find-lhs (ft net mode &aux (stack (get-c-array *stack*))
			trail (N (net-nodes net)) (S ft) (B 0)
			(end (ft-next (ft-end ft))) lf)
  (declare (type fixnum mode))
  (declare-c-array stack)
  (declare (type fixnum B))
  
  (block
   FIND-LHS

  ;(format t "~%Looking for ")(print-term ft)
  (when (null N) (return-from FIND-LHS))


  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (+ B 1))))
    (choice-base () `(c-aref stack (the fixnum (+ B 2))))
    (choice-perms () `(c-aref stack (the fixnum (+ B 3))))
    (choice-permute? () `(c-aref stack (the fixnum (+ B 4))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (+ B 1)) ,x))
    (set-choice-base (x) `(c-aset stack (the fixnum (+ B 2)) ,x))
    (set-choice-perms (x) `(c-aset stack (the fixnum (+ B 3)) ,x))
    (set-choice-permute? (x) `(c-aset stack (the fixnum (+ B 4)) ,x))
    (next-choice () `(incf B 5))
    (prev-choice () `(decf B 5))
    (choice-stack-empty? () `(zerop B))
    )
  (prog ()
    
    (when (= mode *multiple-resume*)
	  (setf B (saved-B))
	  (setf lf *saved-LF*)
	  (go FIND-LEAF))
    
    TOP
    (until (eq S end)
      (let ((var-slot (assoc-slot 0 N)))
	(if (slot-val var-slot)
	    (progn
	      (unless (and (var? (ft-symbol S))
			   (slot-val (assoc-slot (slot-num S) N)))
		      (set-choice-N N)
		      (set-choice-S S)
		      (set-choice-permute? nil)
		      (let* ((p (fsym-mutations (ft-symbol S)))
			     (base (when p (new-permutation))))
			(declare (object p))
			(when p (init-permute S p base))
			(set-choice-perms p)
			(set-choice-base base))
		      (next-choice))
	      (setf N (slot-val var-slot))
	      (setf S (ft-next (ft-end S))))
	  (progn
	    (when (var? (ft-symbol S)) (go BACKTRACK))
	    (let ((p (fsym-mutations (ft-symbol S))))
	      (when (and p (slot-val (assoc-slot (slot-num S) N)))
		    (set-choice-N N)
		    (set-choice-S S)
		    (set-choice-permute? t)
		    (let ((base (new-permutation)))
		      (init-permute S p base)
		      (set-choice-perms p)
		      (set-choice-base base))
		    (next-choice)))
	    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
		    (go BACKTRACK))
	    (setf S (ft-next S))))))
    
    (setf lf N)
    FIND-LEAF
    (while lf
      (setf trail (e-match-vars (eqn-lhs (leaf-eqn lf)) ft))
      (when trail (break-from-loop))
      (setf lf (leaf-next lf)))
    (unless trail (go BACKTRACK))
    ;(format t " Found ")(print-eqn (leaf-eqn lf))
    (if (= mode *single*)
	(until (choice-stack-empty?)
	       (prev-choice)
	       (when (choice-base) (free-permutation (choice-base))))
      (progn
	(setf *saved-LF* (leaf-next lf))
	(setf (saved-B) B)))
    (return-from FIND-LHS (cons1 lf trail))
    
    BACKTRACK
    (when (choice-stack-empty?)
	  (return-from FIND-LHS nil))
    (prev-choice)
    (setf N (choice-N))
    (setf S (choice-S))
    (when (choice-permute?)
	  (let ((p (choice-perms)))
	    (declare (object p))
	    (permute S (car p) (choice-base))
	    (set-choice-perms (cdr p))
	    (unless (choice-perms)
		    (free-permutation (choice-base)))))
    (set-choice-permute? t)
    (when (choice-perms) (next-choice)) ;Save choicepoint
    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
	    (go BACKTRACK))
    (setf S (ft-next S))
    (go TOP)
    ))))

;; Undo permutations and deallocate base arrays after e-net-find-lhs.
;; ### Note: This function must be within the scope of the with-stacks
;; containing e-net-find-lhs!!!  (It must share the choice stack).
(proclaim '(function e-net-cleanup () nil))
(defun e-net-cleanup ( &aux (B (saved-B))
			    (stack (get-c-array *stack*)))
  (declare (type fixnum B))
  (declare-c-array stack)
  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (+ B 1))))
    (choice-base () `(c-aref stack (the fixnum (+ B 2))))
    (choice-perms () `(c-aref stack (the fixnum (+ B 3))))
    (choice-permute? () `(c-aref stack (the fixnum (+ B 4))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (+ B 1)) ,x))
    (set-choice-base (x) `(c-aset stack (the fixnum (+ B 2)) ,x))
    (set-choice-perms (x) `(c-aset stack (the fixnum (+ B 3)) ,x))
    (set-choice-permute? (x) `(c-aset stack (the fixnum (+ B 4)) ,x))
    (next-choice () `(incf B 5))
    (prev-choice () `(decf B 5))
    (choice-stack-empty? () `(zerop B))
    )
   (until (choice-stack-empty?)
	  (prev-choice)
	  (when (choice-base) (free-permutation (choice-base))))))

) ;Matches with-stack at top of e-net-find-lhs


;; Like net-find-matches.
(proclaim '(function e-net-find-matches (t t t) t))
(with-stacks ((*stack* :stack) (*fstack* :fixnum-stack))
(defun e-net-find-matches (rule net rewrite?
			   &aux (stack (get-c-array *stack*))
			   (ft (eqn-lhs rule))
			   (fstack (get-fixnum-array *fstack*))
			   matches (N (net-nodes net))
			   (is-prule (eq (eqn-type rule) '*pseudo-rule*))
			   (B 0) (FB 0) (S ft) (last (ft-next (ft-end ft)))
			   (A 0) (SL 0) (TRIED 0) lf)
  (declare-c-array stack)
  (declare-fixnum-array fstack)
  (declare (type fixnum B FB A SL TRIED))
  
  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (+ B 1))))
    (choice-perms () `(c-aref stack (the fixnum (+ B 2))))
    (choice-base () `(c-aref stack (the fixnum (+ B 3))))
    (choice-A () `(f-aref fstack FB))
    (choice-SL () `(f-aref fstack (the fixnum (+ FB 1))))
    (choice-tried () `(f-aref fstack (the fixnum (+ FB 2))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (+ B 1)) ,x))
    (set-choice-perms (x) `(c-aset stack (the fixnum (+ B 2)) ,x))
    (set-choice-base (x) `(c-aset stack (the fixnum (+ B 3)) ,x))
    (set-choice-A (x) `(f-aset fstack FB ,x))
    (set-choice-SL (x) `(f-aset fstack (the fixnum (+ FB 1)) ,x))
    (set-choice-tried (x) `(f-aset fstack (the fixnum (+ FB 2)) ,x))
    (next-choice () `(prog () (incf B 4) (incf FB 3)))
    (prev-choice () `(prog () (decf B 4) (decf FB 3)))
    (choice-stack-empty? () `(zerop B))
    )
   
   (prog ()
	 
   TOP
   (when (eq S last) (go DONE))
   (when (null N) (go BACKTRACK))
   ;(princ "Term = ")(print-term ft)(terpri)

   (unless (var? (ft-symbol S))
	   (when (setf N (slot-val (assoc-slot (slot-num S) N)))
		 (let* ((p (fsym-mutations (ft-symbol S)))
			(base (when p (new-permutation))))
			(when p
			      (set-choice-N N)
			      (set-choice-S S)
			      (set-choice-base base)
			      (init-permute S p base)
			      (set-choice-perms p)
			      (next-choice))))
	   (setf S (ft-next S))
	   (go TOP))
		   
   (setf A 1)
   (setf TRIED 0)
   (setf SL 0)
   
   SKIP-SUBTERM
   (while (plusp A)
       ;(format t "A = ~S~%" A)
       (until (>= SL (the fixnum *slotarray-size*))
	   (when (slot-val (assoc-slot SL N))
		 (break-from-loop))
	   (incf SL))
       (unless (< SL (the fixnum *slotarray-size*)) (go BACKTRACK))
       (incf TRIED)
       (when (< TRIED (the fixnum (node-slots-used N)))
	     ;(princ "Pushing choicepoint")(terpri)
	     (set-choice-N N)
	     (set-choice-S S)
	     (set-choice-A A)
	     (set-choice-SL SL)
	     (set-choice-tried TRIED)
	     (set-choice-perms nil)
	     (next-choice))
       (decf A)
       (unless (= SL 0)
	       (incf A (slot-arity SL)))
       (setf N (slot-val (assoc-slot SL N)))
       (setf SL 0)
       (setf TRIED 0)
       )
   (setf S (ft-next S))
   (go TOP)
   
   BACKTRACK
   ;(format t "<<Backtracking>>~%")
   (when (choice-stack-empty?)
	 (return matches))
   (prev-choice)
   (setf N (choice-N))
   (setf S (choice-S))
   (let ((p (choice-perms)))
     (cond
      (p (permute S (car p) (choice-base))
	 (set-choice-perms (cdr p))
	 (if (choice-perms)
	     (next-choice)
	   (progn
	     (free-permutation (choice-base))))
	 (setf S (ft-next S))
	 (go TOP))
      (t (setf A (choice-A))
	 (setf SL (the fixnum (1+ (choice-SL))))
	 (setf TRIED (choice-tried))
	 (go SKIP-SUBTERM))))
	 
  
   DONE
   (setf lf N)
   (while lf
     (let ((trail (when (leaf-subterm lf) (e-match-vars ft (leaf-subterm lf)))))
       (when trail
	  (when (or (not rewrite?)
		    (and is-prule (nofail-rewrite2 (leaf-subterm lf) trail rule))
		    (and (not is-prule) (ft-rewrite2 (leaf-subterm lf) trail rule)))
		(if rewrite?
		    (sloop for b in (eqn-backpointers (leaf-eqn lf))
			   do (setf (leaf-subterm b) nil))
		  (restore-vars trail))
		(unless (member lf matches)
			(push1 lf matches)))))
     (setf lf (leaf-next lf)))
   (go BACKTRACK)
  
   ))))


;; Like net-find-subsuming-eq
(proclaim '(function e-net-find-subsuming-eq (t t) t))
(with-stacks ((*stack* :stack))
(defun e-net-find-subsuming-eq (eqn net
		   &aux
		   (ft (eqn-lhs eqn))
		   (stack (get-c-array *stack*))
		   (N (net-nodes net)) (S ft) (B 0)
		   (end (ft-next (ft-end ft))) lf)
  (declare-c-array stack)
  (declare (type fixnum B))
  
  (block
   FIND-LHS

  ;(format t "~%Looking for ")(print-term ft)
  (when (null N) (return-from FIND-LHS))

  (macrolet
   ;; Choicepoint structure
   (
    (choice-N () `(c-aref stack B))
    (choice-S () `(c-aref stack (the fixnum (+ B 1))))
    (choice-base () `(c-aref stack (the fixnum (+ B 2))))
    (choice-perms () `(c-aref stack (the fixnum (+ B 3))))
    (choice-permute? () `(c-aref stack (the fixnum (+ B 4))))
    (set-choice-N (x) `(c-aset stack B ,x))
    (set-choice-S (x) `(c-aset stack (the fixnum (+ B 1)) ,x))
    (set-choice-base (x) `(c-aset stack (the fixnum (+ B 2)) ,x))
    (set-choice-perms (x) `(c-aset stack (the fixnum (+ B 3)) ,x))
    (set-choice-permute? (x) `(c-aset stack (the fixnum (+ B 4)) ,x))
    (next-choice () `(incf B 5))
    (prev-choice () `(decf B 5))
    (choice-stack-empty? () `(zerop B))
    )
  (prog ()
    
    TOP
    (until (eq S end)
      (let ((var-slot (assoc-slot 0 N)))
	(if (slot-val var-slot)
	    (progn
	      (unless (and (var? (ft-symbol S))
			   (slot-val (assoc-slot (slot-num S) N)))
		      (set-choice-N N)
		      (set-choice-S S)
		      (set-choice-permute? nil)
		      (let* ((p (fsym-mutations (ft-symbol S)))
			     (base (when p (new-permutation))))
			(declare (object p))
			(when p (init-permute S p base))
			(set-choice-perms p)
			(set-choice-base base))
		      (next-choice))
	      (setf N (slot-val var-slot))
	      (setf S (ft-next (ft-end S))))
	  (progn
	    (when (var? (ft-symbol S)) (go BACKTRACK))
	    (let ((p (fsym-mutations (ft-symbol S))))
	      (when (and p (slot-val (assoc-slot (slot-num S) N)))
		    (set-choice-N N)
		    (set-choice-S S)
		    (set-choice-permute? t)
		    (let ((base (new-permutation)))
		      (init-permute S p base)
		      (set-choice-perms p)
		      (set-choice-base base))
		    (next-choice)))
	    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
		    (go BACKTRACK))
	    (setf S (ft-next S))))))
    
    (setf lf N)
    (while lf
      (when (subsumes (leaf-eqn lf) eqn)
	    (until (choice-stack-empty?)
		   (prev-choice)
		   (when (choice-base) (free-permutation (choice-base))))
	    (return-from e-net-find-subsuming-eq t))
      (setf lf (leaf-next lf)))

    
    BACKTRACK
    (when (choice-stack-empty?)
	  (return-from FIND-LHS nil))
    (prev-choice)
    (setf N (choice-N))
    (setf S (choice-S))
    (when (choice-permute?)
	  (let ((p (choice-perms)))
	    (declare (object p))
	    (permute S (car p) (choice-base))
	    (set-choice-perms (cdr p))
	    (unless (choice-perms)
		    (free-permutation (choice-base)))))
    (set-choice-permute? t)
    (when (choice-perms) (next-choice)) ;Save choicepoint
    (unless (setf N (slot-val (assoc-slot (slot-num S) N)))
	    (go BACKTRACK))
    (setf S (ft-next S))
    (go TOP)
    )))))

