;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File: copy.lsp
;;; System: HIPER
;;; Programmer: Jim Christian
;;; Date: April, 1989
;;; Copyright (c) 1989 by Jim Christian.  All rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copying and rewriting of flatterms and equations.

;; Basic copy, no dereferencing of variables.
(proclaim '(function dupl-ft (t) t))
(defun dupl-ft (ft &aux new-ft)
  (do-ft-backwards ft :return new-ft
    (ft-prepend new-ft (get-ft (ft-symbol (ft))))))

#|
;; Note the idiom (get-ft (ft-symbol (ft))) instead of (get-ft (ft-sym)).
;; This avoids the conversion of the C int (ft-sym) to a lisp fixnum
(defmacro dupl-ft (ft)
  (let ((new-ft (gentemp "NEW-FT"))
	(old-ft (gentemp "OLD-FT")))
    `(let (,new-ft (,old-ft ,ft))
       (do-ft-backwards ,old-ft :return ,new-ft
	  (ft-prepend ,new-ft (get-ft (ft-symbol (ft))))))))
|#

;; Dereference one level of bindings
(proclaim '(function dupl-ft-after-match (t) t))
(defun dupl-ft-after-match (ft &aux new-ft)
  (do-ft-backwards ft :return new-ft
     (if (and (var? (ft-sym)) (bound-var? (ft-sym)))
	 (ft-concat new-ft (dupl-ft (var-binding (ft-sym))))
       (ft-prepend new-ft (get-ft (ft-symbol (ft)))))))

;; Allows unbound vars; Always derefs
(proclaim '(function dupl-ft-after-unify (t) t))
(defun dupl-ft-after-unify (ft &aux new-ft)
  (do-ft-backwards ft :return new-ft
     (if (and (var? (ft-sym)) (bound-var? (ft-sym)))
	 (ft-concat new-ft (dupl-ft-after-unify (var-binding (ft-sym))))
       (ft-prepend new-ft (get-ft (ft-symbol (ft)))))))

#|
(proclaim '(function patch-ends (t t t) nil))
(defun patch-ends (ft old-end new-end)
  (setf (ft-end ft) new-end)
  (do-ft-backwards ft
     (when (eq (ft-end (ft)) old-end)
	   (setf (ft-end (ft)) new-end))))
|#

;; Adjust end-pointer of an inserted flatterm
(defmacro patch-ends (ft old-end new-end)
  (assert (and (symbolp ft) (symbolp old-end) (symbolp new-end)))
  `(progn
     (setf (ft-end ,ft) ,new-end)
     (do-ft-backwards ,ft
       (when (eq (ft-end (ft)) ,old-end)
	     (setf (ft-end (ft)) ,new-end)))))

;;Destructively replace old-ft by new-ft, deallocating structures when possible
(proclaim '(function ft-insert (t t) t))
(defun ft-insert (old-ft new-ft)
  (let ((prev (ft-prev old-ft))
	(old-last (ft-end old-ft))
	(new-last (ft-end new-ft)))
    ;; If rewriting at top, simply bash old with new
    ;; Otherwise, hook tail of new into tail of old, and 
    ;; point head of prev to new.
    (cond (prev
	   (setf (ft-next prev) new-ft)
	   (setf (ft-prev new-ft) prev)
	   (setf (ft-symbol old-last) (ft-symbol new-last))
	   (setf (ft-prev old-last) (ft-prev new-last))
	   (setf (ft-next (ft-prev new-last)) old-last)
	   (patch-ends new-ft new-last old-last)
	   (do-ft old-ft :safely
		  (unless (eq (ft) old-last) (free-ft (ft))))
	   (unless (eq new-last new-ft) (free-ft new-last))
	   (ft-next prev))
	  (t
	   (setf (ft-symbol old-ft) (ft-symbol new-ft))
	   (setf (ft-next old-ft) (ft-next new-ft))
	   (if (eq (ft-end new-ft) new-ft)
	       (setf (ft-end old-ft) old-ft)
	     (setf (ft-end old-ft) (ft-end new-ft)))
	   (when (ft-next new-ft) (setf (ft-prev (ft-next new-ft)) old-ft))
	   (free-ft new-ft)
	   (until (eq old-last old-ft)
		  (setf prev (ft-prev old-last))
		  (unless (eq old-last old-ft) (free-ft old-last))
		  (setf old-last prev))
	   old-ft)
    )))

;; Rewrite a subterm using the rule associated with a discrimination
;; net leaf
(defmacro ft-rewrite (ft lf-trail)
  (let ((new-ft (gentemp "NEW-FT"))
	(eqn (gentemp "EQN"))
	(trail (gentemp "TRAIL")))
    `(let* ((,eqn (leaf-eqn (car ,lf-trail)))
	    (,trail (cdr ,lf-trail))
	    (,new-ft (dupl-ft-after-match (eqn-rhs ,eqn))))
       (prog1 (ft-insert ,ft ,new-ft)
	 (restore-vars ,trail)
	 (free-cons ,lf-trail)))))

;; Another version of ft-rewrite
(defmacro ft-rewrite2 (ft trail eqn)
  (let ((new-ft (gentemp "NEW-FT")))
    `(let* ((,new-ft (dupl-ft-after-match (eqn-rhs ,eqn))))
       (prog1 (ft-insert ,ft ,new-ft)
	 (restore-vars ,trail)))))

;; Duplicate ft2 and insert it into a copy of ft2 at the location
;; specified by where.  Used to contstruct a new critical pair.
(proclaim '(function dupl-insert (t t t) t))
(defun dupl-insert (ft1 ft2 where &aux new-ft new-ft2 where-end)
  (setf new-ft2 (dupl-ft-after-unify ft2))
  (setf where-end (ft-end where))
  (do-ft-backwards ft1 :return new-ft
     (cond
      ((eq (ft) where-end)
       (ft-concat new-ft new-ft2)
       (skip-to (ft-prev where)))
      ((and (var? (ft-sym)) (bound-var? (ft-sym)))
       (ft-concat new-ft (dupl-ft-after-unify (var-binding (ft-sym)))))
      (t
       (ft-prepend new-ft (get-ft (ft-symbol (ft))))))))

;; Copy an equation and all of its slots.  (You'll usually want to
;; use this instead of the Lisp copy-eqn function.)
(proclaim '(function dupl-eqn (t) t))
(defun dupl-eqn (e &aux new-e)
  (setf new-e (new-eqn))
  (copy-eqn-slots e new-e)
  new-e
  )

(proclaim '(function copy-eqn-slots (t t) nil))
(defun copy-eqn-slots (from to)
  (setf (eqn-parents to) (eqn-parents from))
  (setf (eqn-type to) (eqn-type from))
  (setf (eqn-id to) (eqn-id from))
  (setf (eqn-lhs to) (dupl-ft (eqn-lhs from)))
  (setf (eqn-rhs to) (dupl-ft (eqn-rhs from)))
  (setf (eqn-supported to) (eqn-supported from))
  )


;; Number all variables of an equation consecutively from 0.
(proclaim '(function normalize (t) t))
(with-stacks ((*trail* :trail))
(defun normalize (eqn &aux (i 0))
  (with-local-trail *trail*
    (do-ft (eqn-lhs eqn)
	   (when (and (var? (ft-sym)) (not (var-binding (ft-sym))))
		 (trail (ft-sym))
		 (set-binding (ft-sym) i)
		 (incf i)))
    (do-ft (eqn-rhs eqn)
	   (when (and (var? (ft-sym)) (not (var-binding (ft-sym))))
		 (trail (ft-sym))
		 (set-binding (ft-sym) i)
		 (incf i)))
    (do-ft (eqn-lhs eqn)
	   (when (var? (ft-sym))
		 (setf (ft-symbol (ft)) (var-binding (ft-sym)))))
    (do-ft (eqn-rhs eqn)
	   (when (var? (ft-sym))
		 (setf (ft-symbol (ft)) (var-binding (ft-sym)))))
    (when (> i *max-vars*)
	  (error "Too many vars in equation ~S" (eqn-id eqn)))
    )))
