#include "header.h"

/*
 *         Copyright (C) Argonne National Laboratory
 *
 *   Argonne does not guarantee this software in any manner and is
 *   not responsible for any damages that may result from its use.
 *   Furthermore, Argonne does not provide any formal support for this
 *   software.  This is an experimental program.  This software
 *   or any part of it may be freely copied and redistributed,
 *   provided that this paragraph is included in each source file.
 *
 */


/*
 *    This file contains routines to manage the shared data structures.
 *   
 */

#define TERM_TAB_SIZE 2000

/* Hash table of shared terms */

static struct term_ptr *Term_tab[TERM_TAB_SIZE];

/*************
 *
 *    int hash_term(term)
 *
 *        Return the hash value of a term.  It is assumed that
 *    the subterms are already integrated.  The hash value is
 *    constructed from the functor and the addresses of the
 *    subterms.
 *
 *************/

static int hash_term(t)
struct term *t;
{
    struct rel *r;
    int hashval;
    
    if (t->type == NAME)   /* name */
	hashval = t->sym_num;
    else if (t->type == VARIABLE)  /* variable */
	hashval = t->sym_num + t->varnum + 25;
    else {  /* complex */
	hashval = t->sym_num;
	r = t->farg;
	while (r != NULL) {
	    hashval >>= 1;  /* shift right one bit */
	    hashval ^= (int) r->argval; /* exclusive or */
	    r = r->narg;
	    }
	}
    return(abs(hashval) % TERM_TAB_SIZE);
}  /* hash_term */

/*************
 *
 *    int term_compare(t1, t2)
 *
 *        Special purpose term comparison for term integration.
 *    Succeed iff functors are identical and subterm pointers are
 *    identical.  (Note that this routine is not recursive.)
 *    (For general purpose routine, see `term_ident'.)
 *
 *************/

static int term_compare(t1, t2)
struct term *t1;
struct term *t2;
{
    struct rel *r1, *r2;
    
    if (t1->type != t2->type)
	return(0);
    else if (t1->type == NAME) /* name */
	return(t1->sym_num == t2->sym_num);
    else if (t1->type == VARIABLE) /* variable */
	return(t1->varnum == t2->varnum && t1->sym_num == t2->sym_num);
    else if (t1->sym_num != t2->sym_num)
	return(0);  /* both complex with different functors */
    else {
	r1 = t1->farg;
	r2 = t2->farg;
	while (r1 != NULL && r2 != NULL) {
	    if (r1->argval != r2->argval)
		return(0);
	    else {
		r1 = r1->narg;
		r2 = r2->narg;
		}
	    }
	return(r1 == NULL && r2 == NULL);
	}
}  /* term_compare */

/*************
 *
 *    struct term *integrate_term(term)
 *
 *        Incorporate a term into the shared data structures.
 *    Either ther term itself is integrated and returned, or
 *    the term is deallocated and a previously integrated copy
 *    is returned.  A good way to invoke it is
 *
 *           t = integrate(t)
 *
 *************/

struct term *integrate_term(t)
struct term *t;
{
    int hashval, found;
    struct term_ptr *p, *prev;
    struct rel *r, *r2;
    
    if (t->type == COMPLEX) {  /* complex term */
	r = t->farg;
	while (r != NULL) {
	    r->argval = integrate_term(r->argval);
	    r = r->narg;
	    }
	}
    
    hashval = hash_term(t);
    p = Term_tab[hashval];
    prev = NULL;
    found = 0;
    while (found == 0 && p != NULL)
	if (term_compare(t, p->term))
	    found = 1;
	else {
	    prev = p;
	    p = p->next;
	    }
    
    if (found) {    /* p->term is integrated copy */
	if (t == p->term) {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, integrate_term, already integrated.\007\n");
	    fprintf(stdout, "ABEND, integrate_term, already integrated: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	if (t->type == COMPLEX) { /* if complex, free rels */
	    r = t->farg;
	    while (r != NULL) {
		r2 = r;
		r = r->narg;
		free_rel(r2);
		}
	    }
	free_term(t);
	return(p->term);
	}
    else {    /* not in bucket, so insert at start of bucket and return */
	if (t->type == COMPLEX) { /* if complex, set up containment lists */
	    r = t->farg;
	    while (r != NULL) {
		r->argof = t;
		r->nocc = r->argval->occ.rel;
		r->argval->occ.rel = r;
		r = r->narg;
		}
	    }
	p = get_term_ptr();
	p->term = t;
	p->next = NULL;
	if (prev == NULL)
	    Term_tab[hashval] = p;
	else
	    prev->next = p;
	return(t);
	}
}  /* integrate_term */

/*************
 *
 *    disintegrate_term(term)
 *
 *       Remove a previously integrated term from the shared data
 *    structures.  A warning is printed if the term has a list of
 *    superterms.
 *
 *************/

void disintegrate_term(t)
struct term *t;
{
    int hashval;
    struct rel *r1, *r2, *r3;
    struct term_ptr *p1, *p2;
    
    if (t->occ.rel != NULL) {
	fprintf(stderr, "WARNING, disintegrate_term, contained term.\n");
	printf("WARNING, disintegrate_term, contained term: ");
	print_term_nl(stdout, t);
	}
    else {
	hashval = hash_term(t);
	p1 = Term_tab[hashval];
	p2 = NULL;
	while (p1 != NULL && p1->term != t) {
	    p2 = p1;
	    p1 = p1->next;
	    }
	if (p1 == NULL)  {
	    output_stats(stdout, 4);
	    fprintf(stderr, "ABEND, disintegrate_term cannot find term.\007\n");
	    fprintf(stdout, "ABEND, disintegrate_term cannot find term: ");
	    print_term_nl(stdout, t);
	    exit(1);
	    }
	else {
	    if (p2 == NULL)
		Term_tab[hashval] = p1->next;
	    else
		p2->next = p1->next;
	    free_term_ptr(p1);
	    if (t->type == COMPLEX) {
		r1 = t->farg;
		while (r1 != NULL) {
		    r2 = r1->argval->occ.rel;
		    r3 = NULL;
		    while (r2 != NULL && r2 != r1) {
		        r3 = r2;
		        r2 = r2->nocc;
		        }
		    if (r2 == NULL) {
			output_stats(stdout, 4);
		        fprintf(stderr, "ABEND, disintegrate_term bad containment.\007\n");
		        fprintf(stdout, "ABEND, disintegrate_term bad containment: ");
		        print_term_nl(stdout, t);
		        exit(1);
		        }
		    else {
		        if (r3 == NULL)
		            r1->argval->occ.rel = r1->nocc;
		        else
		            r3->nocc = r1->nocc;
		        if (r1->argval->occ.rel == NULL)
		            disintegrate_term(r2->argval);
		        }
		    r3 = r1;
		    r1 = r1->narg;
		    free_rel(r3);
		    }
		}
	    free_term(t);
	    }
	}
}  /* disintegrate_term */

/*************
 *
 *    zap_term(term)
 *
 *        Deallocate a nonshared term.  A warning is printed
 *    the term or any of its subterms contains a list of superterms.
 *
 *************/

void zap_term(t)
struct term *t;
{
    struct rel *r1, *r2;
    
    if (t->occ.rel != NULL) {
	fprintf(stderr, "WARNING, zap_term, contained term.\n");
	printf("WARNING, zap_term, contained term: ");
	print_term_nl(stdout, t);
	printf("\n");
	}
    else {
	if (t->type == COMPLEX) { /* complex term */
	    r1 = t->farg;
	    while (r1 != NULL) {
		zap_term(r1->argval);
		r2 = r1;
		r1 = r1->narg;
		free_rel(r2);
		}
	    }
	free_term(t);
	}
}  /* zap_term */

/*************
 *
 *    print_term_tab(file_ptr) -- Print the table of integrated terms.
 *
 *************/

void print_term_tab(fp)
FILE *fp;
{
    int i;
    struct term_ptr *p;
    
    for(i=0; i<TERM_TAB_SIZE; i++)
	if(Term_tab[i] != NULL) {
	    fprintf(fp, "bucket %d: ",i);
	    p = Term_tab[i];
	    while(p != NULL) {
	       print_term(fp, p->term);
	       fprintf(fp, " ");
	       p = p->next;
	       }
	    fprintf(fp, "\n");
	    }
}  /* print_term_tab */

