functor Rename
  (structure Term : TERM
   structure Basic : BASIC
   structure Lists : LISTS
  ) : RENAME =
 struct 

   (*** Switches ***)

  structure Switch =
  struct
    exception UnknownSwitch = Basic.UnknownSwitch

    (* Control *)
    fun control s = raise UnknownSwitch("Rename.control",s)

    (* Warning *)
    fun warn s = raise UnknownSwitch("Rename.warn",s)

    (* Tracing *)
    fun trace s = raise UnknownSwitch("Rename.trace",s)

  end  (* structure Switch *)

  open Switch

   structure Term = Term
   open Term

   (*** Generating unique variables ***)

   local 
     val num = ref nil : string list ref          (* variable names in use *)
     val stored = ref nil : string list list ref  (* stack of usage lists *)

     fun getbase s =                              (* get ``root'' of var name *)
         let val sl = rev(explode s)
             fun chop (l as (h::t)) =
                      if ("0"<=h andalso h<="9") orelse h="'" 
                         orelse h="_" then chop t
                         else implode(rev l)
               | chop nil = "$"
         in
           if s=Term.anonymous then s
           else chop sl
         end

     val try = ref ~1 

     fun getnext v =                              (* change var name *)
         (inc try;
          case !try of
            0 => v | 1 => v^"'" | n => v^(makestring (n-2)))
                 
   in
     (* determine free vars in list of terms to avoid name clashes *)
     fun init_newvar tml =
     let fun cum (Uvar(Varbind(_,b),_)) res = cum b res
           | cum (Evar(Varbind(_,_),_,_,ref(SOME(b)))) res = cum b res
           | cum (Evar(Varbind(_,b),_,_,_)) res = cum b res
           | cum (tm as (Const _)) res = res
           | cum (Appl(M1,M2)) res = cum M2 (cum M1 res)
           | cum (Abst(xofA,M)) res = cum M (vcum xofA res)
           | cum (Pi((xofA,B),_)) res = cum B (vcum xofA res)
           | cum (HasType(M,A)) res = cum M (cum A res)
           | cum (Bvar x) res = if Lists.is_in x res then res else x::res
           | cum tm res = res
         and vcum (Varbind(x,A)) res = cum A 
                                           (if Lists.is_in x res 
                                               then res else x::res)
     in
        num := (fold (fn (tm,cuml) => cum tm cuml) tml nil)
     end
  
     (* determine a new variable name *)
     fun newvar v =
         let val vbase = getbase v
             val _ = try := ~1
             fun next_available v =
                 let val v'=getnext v in
                     if Lists.is_in v' (!num) then next_available v
                        else (num := (v' ::(!num)); v')
                 end
             val res =
                 if v=Term.anonymous then v
                 else if not(Lists.is_in v (!num))
                 then (num := (v::(!num)); v)
                 else next_available vbase
         in
           (res)
         end

     fun newvar_conflict str = Lists.is_in str (!num)

     (* temporarily save the var names seen so far *)
     fun push_state () = (stored := (!num :: (!stored));
                          num := nil )

     (* restore original list of var names *)
     fun restore_state () =
         case !stored of
             nil => raise Basic.Illegal 
                           "Rename.pop_state: no state has been saved"
           | (h::t) => ( num := h;
                         stored := t )

    end

  end
