(in-package 'spa)

;;;(defsetf snlp-plan-initial-conditions set-plan-initial-conditions!)
;;;(defsetf snlp-plan-goal-conditions set-plan-goal-conditions!)

;;;*******************************************************************
;;; Retrieving a plan:   
;;;
;;;   Plan retrieval is either automatic or interactive.  In the latter
;;;   case the user is asked if a library plan is appropriate, and if so, 
;;;   he provides a mapping of problem constants to plan variables.
;;;
;;;   In the automatic case we call the CHECKER function for each entry 
;;;   in the library, which decides, in an idiosyncratic way, whether its
;;;   plan is appropriate.  It is called with INITIAL and GOAL, and returns
;;;   either NIL or a copy of the library plan, appropriately 
;;;   instantiated and fitted for the problem.

(defvar *retrieve-interactively* nil)

;;;*******************************************************************

(defun retrieve-plan (initial goal &optional (lib-hint nil))
  (let ((lib-plan (if *retrieve-interactively*
                      (search-lib-interactively initial goal lib-hint)
                      (search-lib-automatically initial goal lib-hint))))
    (cond
      ((null lib-plan)  
       (debug-msg :planlib "Retrieval failed, returning empty plan.")
       (make-empty-plan initial goal))
      (t lib-plan))))

;;;***************************************************************************
;;;  Automatic searching....  just call the plan entry's CHECKER function.

(defun search-lib-automatically (initial goal &optional (lib-hint nil))
  (cond
   (lib-hint 
    (let ((the-entry (find-lib-entry lib-hint)))
      (if (null the-entry)
          (error "No entry associated with hint ~a" lib-hint))
      (try-plan-lib-entry the-entry initial goal)))
   (t (some #'(lambda (plan-lib-entry) 
                (try-plan-lib-entry plan-lib-entry initial goal))
            *plan-library*))))

(defun try-plan-lib-entry (entry initial goal)
  (debug-msg :planlib "Trying lib entry ~a" entry)
  (cond
    ((not (domain-compatible? entry))
     (debug-msg :planlib " ... rejecting incompatible domain")
     nil)
    (t (funcall (lib-entry-checker-fun entry) initial goal))))

(defun domain-compatible? (entry)
  (if (null *current-domain-names*)
      (debug-msg :planlib 
        "WARNING:  No domain is loaded.  All library retrievals will fail!"))
  (member (lib-entry-domain-name entry) *current-domain-names*))

;;;*****************************************************************************
;;;   Interactive searching:  for each plan library entry, first ask the 
;;;   user.  If user says YES, ask for constants corresponding to the 
;;;   variables in the library plan.

(defun search-lib-interactively (initial goal &optional (lib-hint nil))
  (declare (ignore lib-hint))
  (some #'(lambda (entry) (try-entry-interactively entry initial goal))
        *plan-library*))

(defun try-entry-interactively (entry initial goal)
  (cond
    ((not (y-or-n-p "Is ~a a good match?  " (lib-entry-problem-name entry)))
     nil)
    (t (really-select-entry entry initial goal))))

(defun really-select-entry (entry initial goal)
  (let* ((the-plan (copy-plan-completely (funcall (lib-entry-plan-fun entry))))
         (the-bindings (snlp-plan-bindings the-plan)))
    (print-lists "Library plan initials:" (snlp-plan-initial-conditions the-plan)
                 "Input initials:"        initial)
    (format t "~%")
    (print-lists "Library plan goals:"   (snlp-plan-goal-conditions the-plan)
                 "Input goals:"          goal)
    (dolist (var (lib-plan-variables the-plan))
      (format t "What's the mapping for variable ~a?   " var)
      (let ((input-const (read)))
        (add-constraint (list input-const var) the-bindings)))
    the-plan))

(defun print-lists (header1 l1 header2 l2)
  (format t "~&~a~40T~a~%" header1 header2)
  (do ((ll1 l1 (cdr ll1))
       (ll2 l2 (cdr ll2)))
      ((and (null ll1) (null ll2)) (values))
    (format t "   ~a~40T   ~a~%" 
            (if ll1 (car ll1) "")
            (if ll2 (car ll2) ""))))

(defun lib-plan-variables (lib-plan)
  (let ((forms (append (snlp-plan-initial-conditions lib-plan)
                       (snlp-plan-goal-conditions lib-plan)))
        (vars '()))
    (dolist (form forms)
      (dolist (elt form) 
        (if (variable? elt)
            (pushnew elt vars))))
    vars))
