;;;***************************************************************************
(in-package 'spa)
;;;***************************************************************************
;;;  This file contains two different fitting algorithms.  Each takes
;;;  an input initial and goal, and a library plan, and returns an 
;;;  assignment of constants to the variables in the library plan 
;;;  that cause (hopefully many of the) input goals to unify with the 
;;;  library plan's goals and likewise for the initials.  The two 
;;;  algorithms are:
;;;    1.  maximize-satisfied-goals --- return the assignment that 
;;;        maximizes the number
;;;        of goal forms matched;  break ties by seeing how many 
;;;        initials match.  this can be *very* expensive computationally.
;;;    2.  bottom-up --- this is pretty block-stacking specific.  try 
;;;        to match the last form in the library plan goal against the last 
;;;        form in the input plan goal, and so on "up".  this is very 
;;;'       cheap, but may be of limited applicability.
;;;

;;;***************************************************************************
;;;  MAXIMIZE-SATISFIED-GOALS
;;;  Returns a plan with bindings such that:
;;;
;;;    1.  it is non-dominated in the number of goal conditions
;;;        it matches, then we break ties by:
;;;
;;;    2.  the number of links in the initial conditions that are 
;;;        left open is minimized.
;;;

(defun maximize-satisfied-goals (lib-plan input-initial input-goal)
  (let ((lib-plan (copy-plan-completely lib-plan))
        (real-initial (mapcar #'as-condx input-initial))
        (real-goal (mapcar #'as-condx input-goal)))
    (print-initials real-initial real-goal lib-plan)
    (let* ((cs-matches 
            (maximal-goal-match real-goal
                                (snlp-plan-goal-conditions lib-plan)
                                (copy-cs (snlp-plan-bindings lib-plan))))
           (new-plans (mapcar #'(lambda (cs) 
                                  (prepare-plan lib-plan 
                                                cs 
                                                real-initial 
                                                real-goal))
                              cs-matches))
           (best-plan nil)
           (best-num-links 9999999))
      (do ((plans new-plans (cdr plans)))
          ((null plans) (values))
          (let* ((next-plan (car plans))
                 (open-links (open-initial-links next-plan)))
            (when (< open-links best-num-links)
                  (setq best-num-links open-links)
                  (setq best-plan next-plan))))
      (print-match-bindings lib-plan best-plan)
      (debug-msg :planlib "Returning plan ~a" best-plan)
      best-plan)))

(defun print-initials (input-initial input-goal lib-plan)
  (debug-progn :planlib 
    (let ((inst-plan (instantiate-plan lib-plan)))
      (format t "Input initial:~%")
      (dolist (form input-initial) (format t "  ~a~%" form))
      (format t "Library initial:~%")
      (dolist (form (snlp-plan-initial-conditions inst-plan)) 
              (format t "  ~a~%" form))
      (format t "Input goal:~%")
      (dolist (form input-goal) (format t "  ~a~%" form))
      (format t "Library goal:~%")
      (dolist (form (snlp-plan-goal-conditions inst-plan)) 
              (format t "  ~a~%" form)))))
  
(defun print-match-bindings (lib-plan inst-plan)
  (debug-progn :planlib
   (let* ((the-bindings (snlp-plan-bindings inst-plan))
          (the-vars (remove-duplicates 
                     (mapcan #'cond-vars (snlp-plan-goal-conditions lib-plan))))
          (the-constants (mapcar #'(lambda (var) (var-value var the-bindings))
                                 the-vars)))
     (debug-msg :planlib "Bindings are ~a" (mapcar #'cons the-vars the-constants)))))

  
;;;*********************************************************
  
(defun maximal-goal-match (in-goal lib-goal plan-bindings)
  (multiple-value-bind (cs-list num-unmatched)
                       (real-mgm in-goal lib-goal plan-bindings 0)
     (debug-msg :planlib "Returning ~d options with ~d unmatched goals." 
            (length cs-list) num-unmatched)
     cs-list))
  

;;; input:  input-conds lib-conds bindings-so-far (CS) 
;;; output: (1) list of result CS, (2) number of residual unmatched conds.
  
  
(defun real-mgm (in-conds lib-conds cs-in num-unmatched)
  (cond
   ((null in-conds) 
    (values (list cs-in) num-unmatched))
   ((null lib-conds) 
    (values (list cs-in) (+ num-unmatched (length in-conds))))
   (t (let* ((next-cond (car in-conds))
             (cond-cs-pairs (unify-single-cond next-cond lib-conds cs-in)))
        (cond
         ((null cond-cs-pairs)
          (real-mgm (remove next-cond in-conds) 
                    lib-conds 
                    cs-in 
                    (+ num-unmatched 1)))
         (t (do ((unified-pairs cond-cs-pairs (cdr unified-pairs))
                 (output-list '())
                 (best-num 99999999))
                ((null unified-pairs) (values output-list best-num))
                (let ((next-lib-cond (car (car unified-pairs)))
                      (next-cs (cadr (car unified-pairs))))
                  (multiple-value-bind (final-cs-list final-num)
                                       (real-mgm (remove next-cond in-conds)
                                                 (remove next-lib-cond lib-conds)
                                                 next-cs 
                                                 num-unmatched)
                     (cond
                      ((< final-num best-num)
                       (setq best-num final-num)
                       (setq output-list final-cs-list))
                      ((= final-num best-num)
                       (setf output-list 
                             (append final-cs-list output-list)))))))))))))


;;;  Try unifying NEXT-COND with each of LIB-CONDS.    Return a list 
;;;  of pairs:  (CONDX . CS) where CONDX is the condition from the
;;;  library that unified, and CS is the resulting constraint set.
;;;  One element in the list for each unification that worked.

(defun unify-single-cond (next-cond lib-conds cs-in)
  (let ((cs-list (mapcar #'(lambda (condx)
                             (let* ((ncs (copy-cs cs-in))
                                    (rslt (unify! (cond-form next-cond) 
                                                  (cond-form condx) ncs)))
                               (if (eq rslt ':FAIL) ':FAIL ncs)))
                         lib-conds)))
    (delete-if #'(lambda (pair) (eq ':fail (cadr pair)))
               (mapcar #'list lib-conds cs-list))))
  
  
  
;;;*****************************************************************
  
(defun prepare-plan (lib-plan cs in-initial in-goal)
  (let ((new-plan (copy-plan-completely lib-plan)))
    (setf (snlp-plan-bindings new-plan) cs)
    (fit-plan! new-plan in-initial in-goal)
    new-plan))
  
  
(defun open-initial-links (lib-plan)
  (do ((opens (snlp-plan-open lib-plan) (cdr opens))
       (count 0))
      ((null opens) count)
    (when (not (goal-step-id? (open-step-id (car opens))))
      (incf count))))
  
;;;***************************************************************************
;;;***************************************************************************
;;;  Bottom up checker

(defun bottom-up-fitter (lib-plan initial goal)
  (let* ((lib-plan-goal (mapcar #'condx-to-list
                                (snlp-plan-goal-conditions lib-plan)))
         (real-input-goal (mapcar #'condx-to-list goal))
         (lib-plan-cs (snlp-plan-bindings lib-plan))
         (new-cs (copy-cs lib-plan-cs)))
    (bottom-up-match! real-input-goal lib-plan-goal new-cs)
    (prepare-plan lib-plan new-cs initial goal)))

(defun bottom-up-match! (input-goal-list plan-goal-list cs)
  (do ((iglist (reverse input-goal-list) (cdr iglist))
       (pglist (reverse plan-goal-list) (cdr pglist)))
      ((or (null iglist) (null pglist)) (values))
    (let ((next-goal-form (car iglist))
          (next-plan-form (car pglist)))
      ;;(format t "binding ~a and ~a~%" next-goal-form next-plan-form)
      (let ((gftop (cadr next-goal-form))
            (gfbottom (caddr next-goal-form))
            (pftop (cadr next-plan-form))
            (pfbottom (caddr next-plan-form)))
        (add-constraint! (make-constraint :var1 gftop :var2 pftop) 
                         cs)
        (add-constraint! (make-constraint :var1 gfbottom :var2 pfbottom) 
                         cs)))))

(defun match-checker (lib-problem input-problem)
  (let* ((input-goal (lib-entry-goal (find-lib-entry input-problem)))
         (lib-plan   (lib-entry-plan (find-lib-entry lib-problem)))
         (plan-goal  (snlp-plan-goal-conditions lib-plan))
         (cs         (copy-cs (snlp-plan-bindings lib-plan))))
    (bottom-up-match! input-goal plan-goal cs)
    cs))

