" (c) 1991 Copyright (c) University of Washington
  Written by Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

(in-package 'spa)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;  This parameter takes a partial plan and returns the next 
;;;  condition (open or unsafe) to retract.  Defined in CHOOSE-FUNS.LISP
;;;

(defvar *retract-condition-chooser*)

(defun RETRACT-PLAN (entry)
  (let* ((iplan (qentry-iplan entry))
         (choice (funcall *retract-condition-chooser* iplan)))
    (cond
     ((null choice)                     ; nothing left to retract
      (debug-msg :retract "Retraction impossible")
      nil)                              ; add zero new entries
     (t
      (let ((new-parent (copy-plan-completely iplan)))
        (retract-decision choice new-parent)
        (let ((new-parent-entry (make-qentry :iplan new-parent :dir :retract))
              (new-sibling-entries
               (mapcar #'(lambda (new-plan) 
                           (make-qentry :iplan new-plan :dir :extend))
                       (siblings iplan new-parent choice))))
          (when *debug-save-tree*
            (setf *dbst-most-retracted* new-parent-entry)
            (setf (qentry-parent entry) new-parent-entry)
            (setf (qentry-retracted entry) choice)
            (setf (qentry-retracted-from new-parent-entry) entry)
            (setf (qentry-children new-parent-entry)
	          (mapcar #'qd-list new-sibling-entries))
            (setf (qentry-resolved new-parent-entry) *dbst-resolved*)
            (dolist (s new-sibling-entries)
              (setf (qentry-parent s) new-parent-entry)))
          (cons new-parent-entry new-sibling-entries)))))))


(defun RETRACT-DECISION (decision iplan &key recursive)
  "retract indicated decision, and all other decisions which depend on it,
   from iplan.  destructively modifies iplan"
  (unless recursive
    (debug-msg :retract "***************************************************"))
  (deletef (decision-id decision)
	   (snlp-plan-decisions iplan) :key #'decision-id)
  (ecase (decision-type decision)
    ((:new-link) (retract-link decision iplan))
    ((:new-step) (retract-step decision iplan))
    ((:promote :demote :separate) (retract-protect decision iplan)))
  (values))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun RETRACT-LINK (decision iplan)
  ;; things to delete:
  ;;   1} the link itself
  ;;   2} the ordering established by this link
  ;;   3} the bindings established by this link
  ;;   4} any unresolved unsafes which reference this link
  ;;   5} the references to this decision in producer and consumer steps
  ;; things to add:
  ;;   6} open condition that used to be satisfied by this link
  ;; things to recursively retract:
  ;;   7} any other decisions made to protect this link
  ;;   8} the producing step, if it is no longer supported

  (let* ((link      (get-link (decision-link decision) iplan))
	 (ordering  (get-ordering (link-ordering link) iplan))
	 (cf-list   (link-bindings link))
	 (condition (link-condition link))
	 (p-step    (get-step (link-producer link) iplan))
	 (c-step    (get-step (link-consumer link) iplan)))

    (debug-msg :retract "Retracting ~a"
	       (instantiate link (snlp-plan-bindings iplan)))

    (deletef link (snlp-plan-links iplan))
    (deletef ordering (snlp-plan-ordering iplan))
    (delete-constraints! cf-list (snlp-plan-bindings iplan))
    (deletef (link-id link) (snlp-plan-unsafe iplan) :key #'unsafe-link)

    (when p-step  ; step still exists?
      (deletef (decision-id decision) (step-producing-decisions p-step)))

    (when c-step  ; step still exists?
      (deletef (decision-id decision) (step-consuming-decisions c-step))
      (push (make-open :condition condition :step-id (step-id c-step))
	    (snlp-plan-open iplan)))

    (dolist (d (link-protecting-decisions link))
      (retract-decision (get-decision d iplan) iplan :recursive t))

    (when (and p-step (not (step-supported? p-step)))
      (retract-decision
       (get-decision-establishing-step (step-id p-step) iplan)
       iplan :recursive t))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun RETRACT-STEP (decision iplan)
  ;; if step is not the initial or goal step, do
  ;; things to delete:
  ;;  1} the step itself
  ;;  2} the bindings established from the template
  ;;  3} any open conditions for this step
  ;;  4} any pending unsafes because of this step
  ;; things to recursively retract:
  ;;  5} all avoid decisions (i.e. decisions made to protect other links
  ;;     from this step)
  ;;  6} all links into or out of this step
  (let*  ((step-id (decision-step decision))
	  (step    (get-step step-id iplan))
	  (cf-list (decision-cf-list decision)))
    (unless (or (init-step? step) (goal-step? step))
      
      (debug-msg :retract "Retracting ~a"
		 (instantiate-form (step-action step)
				   (snlp-plan-bindings iplan)))

      (deletef step (snlp-plan-steps iplan))
      (delete-constraints! cf-list (snlp-plan-bindings iplan))
      (deletef step-id (snlp-plan-open iplan) :key #'open-step-id)
      (deletef step-id (snlp-plan-unsafe iplan) :key #'unsafe-clobber-step)
      
      (dolist (d (step-avoiding-decisions step))
	(retract-decision (get-decision d iplan) iplan :recursive t))

      (dolist (d (step-producing-decisions step))
	(retract-decision (get-decision d iplan) iplan :recursive t))

      (dolist (d (step-consuming-decisions step))
	(retract-decision (get-decision d iplan) iplan :recursive t))))
  (values))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun RETRACT-PROTECT (decision iplan)
  ;; things to delete:
  ;;  1} the ordering or binding
  ;;  2} references to this decision in other links and steps
  ;; things to add:
  ;;  3} the unsafe that was formerly resolved by this decision,
  ;;     if it is still pertinent
  (let ((link (get-link (decision-link decision) iplan))
	(step (get-step (decision-step decision) iplan)))

    (ecase (decision-type decision)
      ((:promote :demote)
       (let ((ordering (get-ordering (decision-ordering decision) iplan)))
	 (debug-msg :retract "Retracting ~a" ordering)
	 (deletef ordering (snlp-plan-ordering iplan))))
      ((:separate)
       (let ((cf-list (decision-cf-list decision)))
	 (debug-msg :retract "Retracting binding ~a" cf-list)
	 (delete-constraints! cf-list (snlp-plan-bindings iplan)))))

    (when link
      (deletef (decision-id decision) (link-protecting-decisions link)))
    (when step
      (deletef (decision-id decision) (step-avoiding-decisions step)))
    (when (and step link)
      (debug-msg :retract-o "Re-establishing ~a" (decision-unsafe decision))
      (push (decision-unsafe decision)
	    (snlp-plan-unsafe iplan))))
  (values))
    


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun SIBLINGS (ip parent choice)
  ;; Create the children of parent, except for ip.
  ;; We rely here on the fact that we will only retract
  ;; things in a certain order, such that there will be no
  ;; recursive retractions (except for a link retracting
  ;; an unsupported step).  An additional not-so-obvious
  ;; implication of this is that each retraction of links
  ;; or protection must re-assert an open (resp. unsafe)
  ;; which we know will be at the head of the list.

  (debug-msg :retract-o "Generating siblings for parent & ~a" choice)
  (let ((template-plan (copy-plan-for-extend parent)))
    (ecase (decision-type choice)
      ((:new-step)
       ;; the only time we will be here is if we retract a completely
       ;; useless step (left over from fitting, say). so extending will
       ;; never re-generate ip, so return the unchanged plan, which
       ;; will then be used to generate all possible extensions.
       (when *debug-save-tree* (setf *dbst-resolved* 'useless-step))
       (list template-plan))
      ((:new-link)
       ;; first we need to find out if a step was retracted
       ;; along with the link
       (let* ((link (get-link (decision-link choice) ip))
	      (p-step (get-step (link-producer link) parent))
	      (opn (pop (snlp-plan-open template-plan))))
	 (when *debug-save-tree* (setf *dbst-resolved* opn))
	 (cond (p-step
		;; step exists, look for re-established link only
		(nconc (add-step opn template-plan)
		       (remove-added-link link (new-link opn template-plan))))
	       (t
		;; step disappeared, catch the equiv add-step
		(nconc (remove-added-step
			link (get-step (link-producer link) ip)
			(add-step opn template-plan))
		       (new-link opn template-plan))))))
      ((:promote)
       (let ((uns (pop (snlp-plan-unsafe template-plan))))
	 (when *debug-save-tree* (setf *dbst-resolved* uns))
	 (append (separate uns template-plan)
		 (demote uns template-plan))))
      ((:demote)
       (let ((uns (pop (snlp-plan-unsafe template-plan))))
	 (when *debug-save-tree* (setf *dbst-resolved* uns))
	 (append (separate uns template-plan)
		 (promote uns template-plan))))
      ((:separate)
       (let ((uns (pop (snlp-plan-unsafe template-plan))))
	 (when *debug-save-tree* (setf *dbst-resolved* uns))
	 (append (promote uns template-plan)
		 (demote uns template-plan)
		 (remove-separation (decision-cf-list choice)
				    (separate uns template-plan))))))))


(defun REMOVE-ADDED-LINK (link plan-set)
  (delete-if
   #'(lambda (p)
       (let ((newest-link (car (snlp-plan-links p))))
	 (and (eq (link-producer link) (link-producer newest-link))
	      (eq (link-consumer link) (link-consumer newest-link))
	      (equalp (link-condition link) (link-condition newest-link))
	      (cf-list-equiv? (link-bindings link)
			      (link-bindings newest-link)))))
   plan-set))


(defun REMOVE-ADDED-STEP (link step plan-set)
  (delete-if
   #'(lambda (p)
       (let ((newest-step (car (snlp-plan-steps p)))
	     (newest-link (car (snlp-plan-links p))))
	 (and (eq (link-consumer link) (link-consumer newest-link))
	      (equalp (step-action step) (step-action newest-step))
	      (equalp (link-condition link) (link-condition newest-link))
	      (cf-list-equiv? (link-bindings link)
			      (link-bindings newest-link)))))
   plan-set))


(defun REMOVE-SEPARATION (cf-list plan-set)
  (delete-if
   #'(lambda (p)
       (let ((newest-separation (car (snlp-plan-decisions p))))
	 (cf-list-equiv? cf-list (decision-cf-list newest-separation))))
   plan-set))

    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; One can only retract a choice if no other choice depends on it
;;; One can only retract a link if 
;;;  0.   It is not a handsoff link
;;;  1.   The link aint been protected by promotion, demotion, separation AND
;;;  2A.  Its supporting step supports another link, OR
;;;  2B.  Its supporting step is the initial state, OR
;;;  2C.  The links supporting step can be eliminated, I.e.
;;;    a. The link's supporting step is not the consumer of another link AND
;;;    b. Another link has not been protected from the link's supporting step
;;; One can always retract a protecting step, whether ordering decision
;;; (promotion, demotion) or binding decision (separation).

(defun RETRACTABLE-DECISION (decision ip)
  (ecase (decision-type decision)
    ((:new-step)
     (let ((step (get-step (decision-step decision) ip)))
       (and (not (init-step? step))
	    (not (goal-step? step))
	    (null (step-producing-decisions step))
	    (null (step-avoiding-decisions step)))))
    ((:new-link)
     (let* ((link (get-link (decision-link decision) ip))
	    (step (get-step (link-producer link) ip)))
       (and (not (cond-handsoff (link-condition link)))
	    (null (link-protecting-decisions link))
	    (or (> 1 (length (step-producing-decisions step)))
		(init-step? step)
		(and (null (step-consuming-decisions step))
		     (null (step-avoiding-decisions step)))))))
    ((:separate :demote :promote)
     t)))


;;; is this a _useful_ step?
(defun STEP-SUPPORTED? (step)
  (or (init-step? step)
      (not (null (step-producing-decisions step)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Domain Dependent stuff

;;; Here's where domain dependent info can be put
;;; Prefer links over bindings over ordering decisions
;;; Retract choice of achieving on rather than clear if possible
(defun CHOICE-TO-RETRACT (ip)
  (let ((choice-list
	 (remove-if-not #'(lambda (d) (retractable-decision d ip))
			(snlp-plan-decisions ip))))
  (if *auto-choose*
      (auto-choose choice-list ip)
      (man-choose  choice-list ip))))

(defun AUTO-CHOOSE (choice-list ip)
  (debug-msg :retract "   ~2a Possible retractions" (length choice-list))
  (debug-progn :retract-o
    (dolist (c choice-list)
      (display-decision c ip)))
    
  ;; Always get rid of silly steps first, then links, ...
  (or (find :new-step choice-list :key #'decision-type)
      (find :new-link choice-list :key #'decision-type)
      (car choice-list)))


;;; requires utils/cmenu.lisp
;; Items are specified as (<printed form> <value returned>). 
(defun MAN-CHOOSE (choice-list ip)
  (format t "~%Retracting decision that lead to plan ~a" ip)
  (display-plan ip)
  (let ((menu nil))
    (dolist (b (decisions-of-type :separate choice-list))
      (push (list (display-decision b ip nil) b) menu))
    (dolist (o (decisions-of-type :promote choice-list))
      (push (list (display-decision o ip nil) o) menu))
    (dolist (o (decisions-of-type :demote choice-list))
      (push (list (display-decision o ip nil) o) menu))
    (dolist (l (decisions-of-type :new-link choice-list))
      (push (list (display-decision l ip nil) l) menu))
    (dolist (s (decisions-of-type :new-step choice-list))
      (push (list (display-decision s ip nil) s) menu))

    (utils::cmenu-choose menu "Choose what to Retract")))

(defun DECISIONS-OF-TYPE (type list)
  (remove-if-not #'(lambda (d) (eq (decision-type d) type)) list))
