(in-package 'spa)

;;;  Some small demos for SPA.   You should be in the SPA
;;;  package to run these.

;;;  Generate plan for Sussman anomaly, supplying 
;;;  initial and goal forms explicitly.  Two returns:
;;;  the plan itself and the number of CPU msec to generate it.
;;;  Try running DISPLAY-PLAN on the first.

(defun generate-sussman ()
  (setf *intern-plan-names* t)
  (setf *save-cs* nil)
  (load-domain 'blocksworld)
  (let ((init0 '((on C A) 
                 (on A Table) 
                 (on B Table)
                 (clear C) 
                 (clear B)))
        (goal0 '((on A B) 
                 (on B C))))
    (plan-from-scratch :initial init0 :goal goal0)))


;;; Calling 'generate-sussman' produces the following plan   
;;; (see it using (display-plan *)
;;;
;;; IPLAN #<IP47: S=3; O=0; U=0>
;;; Initial: ((ON C A) (ON A TABLE) (ON B TABLE) (CLEAR C) (CLEAR B))
;;;                          consumes from      produces for       threatened
;;;    1   : (NEWTOWER C)    (INIT INIT)        (3)                ((INIT 2))
;;;    2   : (PUTON B C)     (INIT INIT INIT)   (GOAL)             ((INIT 1))
;;;    3   : (PUTON A B)     (1 INIT INIT)      (GOAL)             ((INIT 2))
;;;
;;; Goal   : ((ON A B) (ON B C))
;;; Complete!
;;;
;;; To avoid an infinite loop in cases when there is no solution, it
;;; is necessary to set the variable spa::*cbr-iplan-limit* to the
;;; maximum number of incomplete plans that the planner can generate
;;; at one time.  This variable's value can be increased for harder
;;; planning problems.

;;;********************************************************************
;;;  This example shows how you can call ADAPT-PLAN directly if 
;;;  you want to (but you probably don't) and some of the internals 
;;;  of the retrieval/fitting process.
;;;  You want to run REFIT-DEMO below.

(defvar *p1* nil "Plan to achieve `stored' initial/goal")
(defvar *p2* nil "Variabilized p1")
(defvar *p3* nil "Plan p2 fitted to `new' init/goal")
(defvar *p4* nil "Final plan to achieve `new' initial/goal")

;;; -----------------------------------------------------------------------

(defun refit-demo ()
  (test-fitting '((on C A) (on A D)
                  (on B Table) (on D Table) (on E Table)
                  (clear C) (clear B) (clear E))
                '((on A B) (on B C) (on D E))
                '((on j k) (on k table)
                  (on l m) (on m table) 
                  (clear j) (clear l))
                '((on l k) (on k j) (on j table))))

;;; -----------------------------------------------------------------------

(defun test-fitting (stored-initial
                     stored-goal
                     new-initial
                     new-goal 
                     &key (rank-fun 'prefer-short))
  (load-domain 'blocksworld) 
  (time-stat "Begin generate stored plan")
  (setf *p1* (plan-from-scratch :initial stored-initial
                                :goal stored-goal 
                                :rank-fun rank-fun))

  (time-stat "finished plan one")
  (setf *p2* (variabilize-plan-selectively *p1* 'all '(table)))
  (setf *p3* (maximize-satisfied-goals *p2* new-initial new-goal))
  (time-stat "done fitting")
  (setf *p4* (adapt-plan *p3* :rank-fun rank-fun))
  (time-stat "finished plan two"))


;;;  One choice of rank functions that maps a queue entry into a 
;;;  number (lower is better).  Others appear in STANDARD-CONTROL-FUNS.LISP

(defun prefer-short (e)
  ;; strongly prefer shorter plans
  (let* ((plan (qentry-iplan e))
         (num-steps (length (SNLP-plan-steps plan)))
         (unsafe (length (SNLP-plan-unsafe plan)))
         (open (length (SNLP-plan-open plan))))
    (floor (* (+ unsafe open (* 2.0 (- num-steps 2)))
              (if (eql (qentry-dir e) :retract) 8 5)))))

;;;******************************************************************
;;; Here's a couple of examples using the problem library and the 
;;; routines in EXPERIMENTS.LISP.  These all refer to experiments in 
;;; Kambhampati's thesis.   These will produce a lot of output 
;;; because debugging is turned on.

(defun run-generative-demo ()
  (initialize-plan-library)
  (run-generative-experiment :problem-name '8bs :debugging t :manual-gc nil))

(defun run-refit-demo ()
  (initialize-plan-library)
  (run-refit-experiment :input-problem-name '8bs1 
                        :lib-problem-name   '4bs1
                        :debugging t
                        :manual-gc nil))

